# -*- coding: utf-8 -*-
from datetime import datetime
import urllib2
from fabric.api import env, run, local, cd
from fabric.contrib import files
from fabric.utils import puts
import re


class _Settings(object):
    def __init__(self, settings_module, suppress_import_errors=True):
        self.SETTINGS_MODULE = settings_module
        try:
            mod = __import__(self.SETTINGS_MODULE, globals(), locals(), [], -1)
        except ImportError:
            if not suppress_import_errors:
                raise
        else:
            self._load_from_module(mod)

    def _load_from_module(self, module):
        """Load settings from module to object."""
        for setting in dir(module):
            if setting.isupper():
                setting_value = getattr(module, setting)
                setattr(self, setting, setting_value)

    MPFS_REMOTE_REPO_PATH = '~/mpfs/'
    REMOTES_PREFIX = 'fabric_'
    DB_NAME = 'mpfs_dev'
    CONDUCTOR_BASE_URL = 'http://c.yandex-team.ru'

local_settings = _Settings('fablocals')


env.roledefs = getattr(local_settings, 'ENV_ROLEDEFS', {
    'dev': lambda: ['%s-ubuntu.dev.yandex.net' % env.user],
    'test': [],
    'production': [],
})

env.roles = ['dev']


def _remote_repo_url(protocol, user, host, port, path):
    """
    Build git repo URL.
    :param protocol:
    :param user: User name or None.
    :param host:
    :param port: Port or None.
    :param path:
    :return:
    """
    url = '%s://' % protocol
    if user:
        url += '%s@' % user
    url += '%s' % host
    if port:
        url += ':%s' % str(port)
    if not ''.startswith('/'):
        url += '/'
    url += path
    return url


def _role_to_remote(role):
    return '%s%s' % (local_settings.REMOTES_PREFIX, role)


def _local_branch():
    branch = filter(lambda s: s.startswith('*'), local('git branch --list', capture=True).splitlines(False))[0][2:]
    return branch


def set_groups(*args):
    """Set hosts to execute following commands from conductor group name."""
    url = '%s%s/%s' % (local_settings.CONDUCTOR_BASE_URL, '/api/groups2hosts', ','.join(args))
    print url
    hosts = urllib2.urlopen(url).read().splitlines()
    print hosts
    env.hosts = hosts


def install_wc():
    """Заменяет все установленные в системе файлы MPFS на симлинки в рабочую копию """
    bin_dst_dir = '/usr/bin'
    files_names = [('fcgiworker.py', 'common'), ('uwsgi_api.py', 'api'), ('uwsgi_common.py', 'common')]
    for file_name, folder in files_names:
        src_path = '%sapps/%s/%s' % (local_settings.MPFS_REMOTE_REPO_PATH, folder, file_name)
        dst_path = '%s/%s' % (bin_dst_dir, file_name)
        run('sudo rm %s' % dst_path)
        run('sudo ln -s %s %s' % (src_path, dst_path))
    run('sudo rm -r /usr/lib/python2.7/dist-packages/mpfs')
    run('sudo ln -s %slib/mpfs /usr/lib/python2.7/dist-packages/mpfs' % local_settings.MPFS_REMOTE_REPO_PATH)
    run('sudo rm -r /etc/mpfs')
    run('sudo mkdir /etc/mpfs')
    run('sudo ln -s %scommon/keys /etc/mpfs/keys')
    conf_paths = map(lambda s: '%s%s' % (local_settings.MPFS_REMOTE_REPO_PATH, s), ('common/conf', 'disk/conf'))
    for p in conf_paths:
        run('sudo ls -1 %s | xargs -I {} sudo ln -s %s/{} /etc/mpfs/{}' % (p, p))


def uninstall_wc():
    bin_dst_dir = '/usr/bin'
    files_names = ['fcgiworker.py', 'uwsgi_api.py', 'uwsgi_common.py']
    for file_name in files_names:
        dst_path = '%s/%s' % (bin_dst_dir, file_name)
        run('sudo rm %s' % dst_path)
    run('sudo rm -r /usr/lib/python2.7/dist-packages/mpfs')
    run('sudo rm -r /etc/mpfs')


def setup_hosts():
    with cd(local_settings.MPFS_REMOTE_REPO_PATH):
        run('git config receive.denyCurrentBranch ignore')


def checkout(branch=None):
    """Checkout branch on remote hosts. Setup remotes if necessary.

    :param branch: Branch to checkout. If None then current branch will be used.
    """
    branch = branch or _local_branch()
    with cd(local_settings.MPFS_REMOTE_REPO_PATH):
        run('git checkout %s' % branch)


def uninstall():
    """Uninstall MPFS from hosts."""
    rm_candidates = run('sudo rpm -qa | grep python-mpfs --color=never', shell=False, warn_only=True).splitlines(False)
    if rm_candidates:
        run('sudo rpm -e --nodeps %s' % ' '.join(rm_candidates), shell=False)


def install(skip_tests=True):
    """Install hosts working copy contents."""
    cmd = 'tools/bi.py'
    if skip_tests:
        cmd += ' -s'
    with cd(local_settings.MPFS_REMOTE_REPO_PATH):
        if files.exists('../rpmbuild'):
            run('rm -r ../rpmbuild')
        # uninstall()
        run(cmd)


def clean():
    """Clean remote working copy from untracked files and directories."""
    with cd(local_settings.MPFS_REMOTE_REPO_PATH):
        run('git checkout -- .')
        run('git clean -f -d')


def db_copy(frm, to, quiet=True):
    context = {'frm': frm, 'to': to}
    cmd = 'echo -e "use %(frm)s\ndb.copyDatabase(%(frm)s, %(to)s)" | mongo'
    run(cmd % context, quiet=quiet)


def db_drop(db, quiet=True):
    context = {'db': db}
    cmd = 'echo -e "use %(db)s\ndb.dropDatabase()" | mongo'
    run(cmd % context, quiet=quiet)


def _get_tests(quiet=True):
    with cd(local_settings.MPFS_REMOTE_REPO_PATH):
        ret = filter(lambda s: s.endswith('_suit.py'), run('ls -1 test/', quiet=quiet).splitlines())
    return ret


def setup_configs(path):
    conf_paths = map(lambda s: '%s%s' % (local_settings.MPFS_REMOTE_REPO_PATH, s), ('common/conf', 'disk/conf'))
    run('sudo rm -rf %s' % (path,))
    run('sudo mkdir -p %s' % (path,))
    for p in conf_paths:
        run('sudo ls -1 %s | xargs -I {} sudo ln -s %s/{} %s/{}' % (p, p, path.rstrip('/'),))


def run_test(suit=None, db=local_settings.DB_NAME, summary=True, quiet=True):
    backup = '%s_%s_%s' % (db, datetime.utcnow().strftime('%Y%m%dT%H%M%S'), env.user)
    try:
        db_copy(db, backup, quiet=quiet)
        tests = ['%s_suit.py' % suit] if suit else _get_tests(quiet=quiet)
        results = []
        with cd(local_settings.MPFS_REMOTE_REPO_PATH):
            curdir = run('pwd', quiet=quiet)
            libdir = '%s/lib/' % curdir
            confdir= '/tmp/mpfs-conf'
            setup_configs(confdir)
            cmd = 'sudo -u nginx /usr/bin/env PYTHONPATH=%s MPFS_CONFIG_PATH=%s python %s'
            with cd('test'):
                for t in tests:
                    db_drop(db, quiet=quiet)
                    puts('Run %s' % t)
                    out = run(cmd % (libdir, confdir, t), warn_only=True, quiet=quiet)
                    results.append({
                        'suit': t,
                        'out': out,
                        'result': out.splitlines()[-1],
                    })
        if summary:
            puts('#')
            puts('# RUN_TEST SUMMARY')
            puts('#' * 78)
            for r in results:
                puts('%(suit)s - %(result)s' % r)
    finally:
        db_drop(db, quiet=quiet)
        db_copy(backup, db, quiet=quiet)
        db_drop(backup, quiet=quiet)
    return results


def test(suit=None, db=local_settings.DB_NAME, summary=True, quiet=True):
    failed_suits = filter(lambda res: not res['result'].startswith('OK'),
                          run_test(suit=suit, db=db, summary=summary, quiet=quiet))

    pattern = re.compile(r'=+(\n|\r\n)(?P<type>[^:]+):\s+(?P<method>[^\s]+)\s+.+(\n|\r\n)+-+(\n|\r\n)+(?P<stacktrace>([^\r\n]+(\n|\r\n))+)(\n|\r\n)', re.MULTILINE)
    failed_tests = []
    for failed_suit in failed_suits:
        for m in re.finditer(pattern, failed_suit['out']):
            failed_tests.append({
                'suit': failed_suit['suit'],
                'type': m.group('type'),
                'method': m.group('method'),
                'stacktrace': m.group('stacktrace'),
            })
    puts('#')
    puts('# TEST SUMMARY')
    puts('#' * 78)
    for t in failed_tests:
        puts('=' * 78)
        puts('%(type)s: %(method)s() %(suit)s' % t)
        puts('-' * 78)
        for l in t['stacktrace'].splitlines():
            puts(l)
    return failed_tests


def diff_test(src, dst=None, suit=None, stacktraces=True, db=local_settings.DB_NAME, nested_summaries=False, quiet=True):
    if dst:
        clean()
        checkout(dst)
    dst_fails = test(suit, db, summary=nested_summaries, quiet=quiet)
    checkout(src)
    src_fails = test(suit, db, summary=nested_summaries, quiet=quiet)

    get_index = lambda test_result: '%(suit)s:%(method)s:%(type)s' % test_result
    get_indexed = lambda fails: dict(map(lambda t: (get_index(t), t), fails))

    dst_fails = get_indexed(dst_fails)
    src_fails = get_indexed(src_fails)

    dst_fail_set = set(dst_fails.keys())
    src_fail_set = set(src_fails.keys())

    new_fails = dst_fail_set - src_fail_set
    removed_fails = src_fail_set - dst_fail_set

    all_fails = dict(src_fails.items() + dst_fails.items())

    puts('#')
    puts('# DIFF_TEST SUMMARY %s -> %s' % (src, dst))
    puts('#' * 78)
    for k in removed_fails:
        puts('--- %(suit)s: %(method)s ... %(type)s' % all_fails[k])
    for k in new_fails:
        puts('+++ %(suit)s: %(method)s ... %(type)s' % all_fails[k])
    if stacktraces:
        puts('#')
        puts('# NEW FAILS STACK TRACES')
        puts('#' * 78)
        puts('\n')
        for k in new_fails:
            fail = all_fails[k]
            puts('=' * 78)
            puts('%(suit)s: %(method)s ... %(type)s' % fail)
            puts('-' * 78)
            for l in fail['stacktrace'].splitlines(False):
                puts(l)
