#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Генератор дажборда для графита

Чтобы применить полученный конфиг, надо:
    1. Перейти в соотвествующий дашборд в графите
    2. Dashboard -> Edit Dashboard
    3. Вставить сгенерированный этим скриптом json
    4. Update
    5. Проверить, что всё отрисовалось
    6. Dashboard -> Save as "<dashboard name>"
"""
import json
import itertools
import collections

from argparse import ArgumentParser

from mpfs.engine.process import setup_anyone_script
setup_anyone_script()

from mpfs.engine.queue2.job_types import QUEUE2_JOB_TYPES_MAP
from mpfs.config import settings


COLORS = [
    "black", "blue", "green",
    "red", "yellow", "orange", "purple",
    "brown", "aqua", "gray", "grey",
    "magenta", "pink", "gold", "rose",
    "darkblue", "darkgreen", "darkred",
    "darkgray", "darkgrey",
]


def get_item_color_map(items):
    return dict(zip(items, itertools.cycle(COLORS)))


def get_mpfs_pended_operation_names():
    return sorted([k for k, v in settings.operations['types'].iteritems() if v['pended']])


def get_mpfs_task_names():
    return sorted([i.__name__ for i in QUEUE2_JOB_TYPES_MAP.values()])


class GraphLine(object):
    def __init__(self, data_source, color=None, alias=None):
        self._data_source = data_source
        self._color = color
        self._alias = alias

    def build(self):
        return self.set_legend_name(
            self.color(
                self._data_source,
                self._color
            ),
            self._alias
        )

    @staticmethod
    def color(graph_line, color):
        if not color:
            return graph_line
        return 'color(%s, "%s")' % (graph_line, color)

    @staticmethod
    def set_legend_name(graph_line, name):
        if not name:
            return graph_line
        return 'alias(%s, "%s")' % (graph_line, name)

    def __repr__(self):
        return "<%s>(%s)" % (self.__class__.__name__, self._alias)


class Graph(object):
    def __init__(self, title=None, hide_legend=False):
        self._lines = []
        self._title = title
        self._hide_legend = hide_legend

    def add_line(self, graph_line):
        if not isinstance(graph_line, GraphLine):
            raise TypeError("Bad `graph_line` type")
        self._lines.append(graph_line)

    def build(self):
        return {
            "target": [t.build() for t in self._lines],
            "title": self._title if self._title else 'EMPTY TITLE',
            "hideLegend": 'true' if self._hide_legend else "false",
        }

    def __repr__(self):
        return "<%s>(%s)(%s)" % (self.__class__.__name__, self._title, self._lines)


class GraphPack(object):
    '''
    config = {
        'data_tmpl': None,
        'entities': {
            'graphs': [],
            'lines': [],
        }
    }
    '''

    def __init__(self, config):
        if not isinstance(config, dict):
            raise TypeError()
        if not config.get('data_tmpl') or not config.get('entities'):
            raise ValueError()
        self.config = config

        for entity in config['entities']['lines'] + config['entities']['graphs']:
            if not isinstance(entity, GraphEnity):
                raise TypeError()

    def build(self):
        graph_pack = []
        graphs_entities = [e.values for e in self.config['entities']['graphs']]
        lines_entities = [e.values for e in self.config['entities']['lines']]
        data_source_tmpl = self.config['data_tmpl']
        for graphs_group in itertools.product(*graphs_entities):
            title = ','.join(["%s.%s" % (g.entity.name, g.value) for g in graphs_group])
            graph = Graph(title=title)
            for lines_group in itertools.product(*lines_entities):
                alias = ','.join(["%s.%s" % (g.entity.name, g.value) for g in lines_group])
                try:
                    color = lines_group[0].color
                except IndexError:
                    try:
                        color = graphs_group[0].color
                    except IndexError:
                        color = 'green'
                hide_legend = False
                params = {v.entity.name: v.value for v in itertools.chain(graphs_group, lines_group)}
                data_source = data_source_tmpl % params
                graph_line = GraphLine(data_source, alias=alias, color=color)
                graph.add_line(graph_line)
            graph_pack.append(graph.build())
        return graph_pack


class GraphEnityValue(object):
    """
    Значение сущности
    """
    def __init__(self, value, entity, color):
        self.value = value
        self.entity = entity
        self.color = color

    def __repr__(self):
        return "<%s>.%s.%s" % (self.__class__.__name__, self.entity.name, self.value)


class GraphEnity(object):
    """
    Сущность, по которой строим графики

    Имеет имя и значения
    Пример: имя `operation`, значения: `copy`, `move` итд
    """
    DEFAULT_COLOR = 'green'

    def __init__(self, name, values, value_color_map):
        self.name = name
        self.values = []
        for value in values:
            color = value_color_map.get(value, self.DEFAULT_COLOR)
            self.values.append(GraphEnityValue(value, self, color))

    def __repr__(self):
        return "<%s>.%s.%s" % (self.__class__.__name__, self.name, [i.value for i in self.values])


def get_available_entities():
    statuses = collections.OrderedDict([
        ('ok', 'green'),
        ('fail', 'red'),
    ])
    times = {
        'lifetime': 'blue',
        'processed': 'green'
    }
    operations = get_mpfs_pended_operation_names()
    operation_color_map = get_item_color_map(operations)
    tasks = get_mpfs_task_names()
    task_color_map = get_item_color_map(tasks)
    return {
        'status': GraphEnity('status', statuses.keys(), statuses),
        'time': GraphEnity('time', times.keys(), times),
        'operation': GraphEnity('operation', operations, operation_color_map),
        'task': GraphEnity('task', tasks, task_color_map),
    }


def build_by_configs(configs):
    dashboard = []
    for config in configs:
        dashboard += GraphPack(config).build()
    print json.dumps(dashboard)


def build_waser_disk_operation():
    # https://gr-ng.yandex-team.ru/dashboard/#waser_disk_operation
    entities = get_available_entities()
    configs = (
        {
            'data_tmpl': 'sumSeries(media.disk.combaine.disk_mworker.mpfs_queue_stats.mpfs_count_status_opertype_*_%(status)s)',
            'entities': {
                'lines': [],
                'graphs': [entities['status']],
            }
        },
        {
            'data_tmpl': 'media.disk.combaine.disk_mworker.mpfs_queue_stats.mpfs_count_status_opertype_%(operation)s_%(status)s',
            'entities': {
                'graphs': [entities['status']],
                'lines': [entities['operation']],
            }
        },
        {
            'data_tmpl': 'media.disk.combaine.disk_mworker.mpfs_queue_stats.mpfs_count_status_opertype_%(operation)s_%(status)s',
            'entities': {
                'graphs': [entities['operation']],
                'lines': [entities['status']],
            }
        }
    )
    build_by_configs(configs)


def build_waser_disk_task():
    # https://gr-ng.yandex-team.ru/dashboard/#waser_disk_task
    entities = get_available_entities()
    configs = (
        {
            'data_tmpl': 'sumSeries(media.disk.combaine.disk_mworker.mpfs_queue_stats.mpfs_count_status_task_mpfs-*_%(status)s)',
            'entities': {
                'lines': [],
                'graphs': [entities['status']],
            }
        },
        {
            'data_tmpl': 'media.disk.combaine.disk_mworker.mpfs_queue_stats.mpfs_count_status_task_mpfs-*%(task)s_%(status)s',
            'entities': {
                'graphs': [entities['task']],
                'lines': [entities['status']],
            }
        },
    )
    build_by_configs(configs)


def build_waser_disk_task_times():
    # https://gr-ng.yandex-team.ru/dashboard/#waser_disk_task_times
    entities = get_available_entities()
    configs = (
        {
            'data_tmpl': 'media.disk.combaine.disk_mworker.mpfs_queue_stats.mpfs_count_%(time)s_time_task_mpfs-*%(task)s',
            'entities': {
                'graphs': [entities['task']],
                'lines': [entities['time']],
            }
        },
    )
    build_by_configs(configs)


if __name__ == '__main__':
    dashboards = {
        'waser_disk_task_times': build_waser_disk_task_times,
        'waser_disk_task': build_waser_disk_task,
        'waser_disk_operation': build_waser_disk_operation,
    }
    parser = ArgumentParser(description='Генератор дашборда для графита')
    parser.add_argument('name', choices=dashboards.keys(), help='Название дашборда.')
    namespace = parser.parse_args()
    dashboards[namespace.name]()
