# -*- coding: utf-8 -*-
u"""
Генератор отчетов по шардам MPFS базы.

Используется Aggregation Pipeline: https://docs.mongodb.org/v3.0/core/aggregation-pipeline/.
Кладет результат в локальную папку.

Поставить последнюю версию pymongo для этого скрипта:
> sudo apt-get install python-pymongo python-bson
Вернуть как было:
> sudo apt-get install python-pymongo=2.6.2-yandex6 python-bson=2.6.2-yandex6
"""
import sys
import pymongo
import csv
import inspect

from datetime import datetime
from argparse import ArgumentParser
from mpfs.config import settings


if pymongo.version_tuple[0] != 3:
    raise NotImplementedError("Supports only pymongo >= 3")


def rounding(field_name, trim):
    return {
        "$subtract": [
            field_name,
            {"$mod": [field_name, trim]}
        ]
    }


def round_and_shift(field_name, trim, shift):
    return {
        "$divide": [
            rounding(field_name, trim),
            shift,
        ]
    }


class MongoReport(object):
    """
    Базовый отчет
    """
    pipeline = None
    keys_order = None

    def __init__(self, collection):
        self.coll = collection

    def _aggregate(self):
        coll = self.coll
        db = coll.database
        conn = db.client
        for value in self.coll.aggregate(self.pipeline):
            result = {
                'collection': coll.name,
                'database': db.name,
                'host': conn.address[0]
            }
            result.update(value)
            yield result

    def result(self):
        return list(self._aggregate())


class UsersPerFileNumber(MongoReport):
    u"""
    Пользователей по кол-ву файлов
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file"}},
        # считаем у каждого пользователя число файлов
        {"$group": {"_id": "$uid", "file_num": {"$sum": 1}}},
        # округляем по 10-ов
        {"$project": {"file_num_rounded": rounding("$file_num", 10)}},
        # считаем количество пользователей по кол-ву файлов
        {"$group": {"_id": "$file_num_rounded", "uids": {"$sum": 1}}},
        {"$project": {"number of file": "$_id", "number of uids": "$uids", "_id": 0}},
    ]
    keys_order = ['host', 'database', 'number of file', 'number of uids']


class UsersPerFileSize(MongoReport):
    u"""
    Пользователей по занимаемому месту
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file", "data.size": {"$exists": True}}},
        # считаем у каждого пользователя число файлов
        {"$group": {"_id": "$uid", "file_size": {"$sum": "$data.size"}}},
        # округляем до 100MB
        {"$project": {"file_size_rounded": round_and_shift("$file_size", 10 ** 8, 10 ** 6)}},
        # считаем количество пользователей по кол-ву файлов
        {"$group": {"_id": "$file_size_rounded", "uids": {"$sum": 1}}},
        {"$project": {"file size (MB)": "$_id", "number of uids": "$uids", "_id": 0}},
    ]
    keys_order = ['host', 'database', 'file size (MB)', 'number of uids']


class FilesSizeAndNumber(MongoReport):
    u"""
    Суммарный размер файлов и их количество в коллекции
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file", "data.size": {"$exists": True}}},
        {"$group": {"_id": "$type", "size(B)": {"$sum": "$data.size"}, "number of files": {"$sum": 1}}},
    ]
    keys_order = ['host', 'database', 'size(B)', 'number of files']


class FilesSizeAndNumberPerUid(MongoReport):
    u"""
    Суммарный размер файлов и их количество по пользователям
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file", "data.size": {"$exists": True}}},
        {"$group": {"_id": "$uid", "size(B)": {"$sum": "$data.size"}, "number of files": {"$sum": 1}}},
    ]
    keys_order = ['host', 'database', '_id', 'size(B)', 'number of files']


class StidsPerDoc(MongoReport):
    u"""
    Количество stid-ов у одного документа
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file", "data.stids": {"$exists": True}}},
        # раскрываем массив stid-ов
        {"$project": {"stid_num": {"$size": "$data.stids"}}},
        {"$group": {
            "_id": "$stid_num",
            "count": {"$sum": 1},
        }},
        {"$project": {"stids in doc": "$_id", "number of docs": "$count", "_id": 0}},
    ]
    keys_order = ['host', 'database', 'stids in doc', 'number of docs']


class StidsDeduplicationRate(MongoReport):
    u"""
    Степень дедупликации file_stid в рамках одной коллекции
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file", "data.stids": {"$exists": True}, "data.size": {"$exists": True}}},
        # раскрываем массив stid-ов
        {"$unwind": "$data.stids"},
        {"$project": {"stid_info": "$data.stids"}},
        {"$match": {"stid_info.type": "file_mid"}},
        # считаем повторения для каждого stid-а
        {"$group": {
            "_id": "$stid_info.stid",
            "stid_num": {"$sum": 1},
        }},
        # считаем количество stid-ов в разбике по частоте повторений
        {"$group": {
            "_id": "$stid_num",
            "count": {"$sum": 1},
        }},
        {"$project": {"reduplication rate": "$_id", "number of stids": "$count", "_id": 0}},
    ]
    keys_order = ['host', 'database', 'reduplication rate', 'number of stids']


class HiddenDataSizePerDays(MongoReport):
    u"""
    Размер hidden_data по дням
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file", "dtime": {"$exists": True}, "data.size": {"$exists": True}}},
        # округляем dtime до дней
        {"$project": {
            "dtime_rounded": rounding("$dtime", 86400),
            'data.size': 1,
        }},
        {"$group": {"_id": "$dtime_rounded", "files_size": {"$sum": "$data.size"}, "files_num": {"$sum": 1}}},
        {"$project": {
            "file size(B)": "$files_size",
            "file number": "$files_num"
        }},
    ]
    keys_order = ['host', 'database', 'date', 'file size(B)', 'file number']

    def result(self):
        result = []
        for value in self._aggregate():
            value['date'] = datetime.fromtimestamp(int(value['_id'])).strftime('%Y-%m-%d')
            result.append(value)
        return result


class FileNumberPerSize(MongoReport):
    u"""
    Распределение файлов по размеру
    """
    pipeline = [
        # оставляем только файлы
        {"$match": {"type": "file", "data.size": {"$exists": True}}},
        # округляем size до 100 KB
        {"$project": {"file_size_rounded": round_and_shift("$data.size", 10 ** 5, 10 ** 6)}},
        # группируем по размеру и считаем количество файлов
        {"$group": {"_id": "$file_size_rounded", "files": {"$sum": 1}}},
        {"$project": {"file size(MB)": "$_id", "number of files": "$files", "_id": 0}},
    ]
    keys_order = ['host', 'database', 'file size(MB)', 'number of files']


class OperationLifeTime(MongoReport):
    u"""
    Распределение кол-ва операций по mtime
    """
    pipeline = [
        # округляем size до 100 KB
        {"$project": {"mtime_rounded": rounding("$mtime", 86400), "state": "$state"}},
        # группируем по размеру и считаем количество файлов
        {"$group": {"_id": {
            "mtime_rounded": "$mtime_rounded",
            "state": "$state",
        }, "operations": {"$sum": 1}}},
        {"$project": {"mtime_rounded": "$_id.mtime_rounded", "state": "$_id.state", "operations": "$operations"}},
    ]
    keys_order = ['host', 'database', 'state', 'mtime_rounded', 'operations']


class UsersPerChangelogNumber(MongoReport):
    u"""
    Пользователей по размеру changelog
    """
    pipeline = [
        # считаем у каждого пользователя число файлов
        {"$group": {"_id": "$uid", "num": {"$sum": 1}}},
        # считаем количество пользователей по кол-ву файлов
        {"$group": {"_id": "$num", "uids": {"$sum": 1}}},
        {"$project": {"changelog records num": "$_id", "number of uids": "$uids", "_id": 0}},
    ]
    keys_order = ['host', 'database', 'changelog records num', 'number of uids']


class ShareGroupsPerMembers(MongoReport):
    u"""
    Распределение групп по кол-ву участников(group_links)
    """
    pipeline = [
        {"$group": {"_id": "$gid", "members_num": {"$sum": 1}}},
        {"$group": {"_id": "$members_num", "gids_num": {"$sum": 1}}},
        {"$project": {"members num": "$_id", "groups num": "$gids_num"}},
    ]
    keys_order = ['host', 'database', 'members num', 'groups num']


class ShareUidsPerGroupsNum(MongoReport):
    u"""
    Распределение числа пользователей по числу групп(groups)
    """
    pipeline = [
        {"$group": {"_id": "$owner", "groups_num": {"$sum": 1}}},
        {"$group": {"_id": "$groups_num", "uids_num": {"$sum": 1}}},
        {"$project": {"groups num": "$_id", "uids num": "$uids_num"}},
    ]
    keys_order = ['host', 'database', 'groups num', 'uids num']


class ShareMembersPerGroups(MongoReport):
    u"""
    Распределение участников групп по кол-ву групп(group_links)
    """
    pipeline = [
        {"$group": {"_id": "$uid", "groups_num": {"$sum": 1}}},
        {"$group": {"_id": "$groups_num", "uids_num": {"$sum": 1}}},
        {"$project": {"groups num": "$_id", "uids num": "$uids_num"}},
    ]
    keys_order = ['host', 'database', 'groups num', 'uids num']


# нет доступных шардов монги
SHARD_HOSTS_MAP = {}

# получаем доступные отчеты
REPORTS_CLASSES_MAP = dict([c for c in inspect.getmembers(sys.modules[__name__], inspect.isclass) if issubclass(c[1], MongoReport)])


def main(report_class, collections, shard_name=''):
    start_dt = datetime.now()
    report_desc = report_class.__doc__.strip()
    file_name = u"./%s_%s(%s)(%s).csv" % (start_dt.strftime('%Y-%m-%d'),
                                          report_class.__name__,
                                          shard_name,
                                          ','.join([i.name for i in collections]))
    keys_order = report_class.keys_order
    with open(file_name, 'wb') as csv_fh:
        csv_writer = csv.writer(csv_fh)
        csv_writer.writerow([report_desc.encode('utf-8')])
        for i, coll in enumerate(collections):
            result = report_class(coll).result()
            if not result:
                continue
            if keys_order is None:
                keys_order = result[0].keys()
            if i == 0:
                csv_writer.writerow(keys_order)
            for value_dict in result:
                csv_writer.writerow([str(value_dict[k]) for k in keys_order])


if __name__ == '__main__':
    shards_names = SHARD_HOSTS_MAP.keys()
    reports_names = REPORTS_CLASSES_MAP.keys()

    parser = ArgumentParser(description=__doc__)
    parser.add_argument('-r', required=True, choices=reports_names, dest='report', help=u'Название отчета.')
    parser.add_argument('-s', required=True, choices=shards_names, dest='shard', help=u'Имя шарда.')
    parser.add_argument('-c', required=True, dest='collection', help=u'Название коллекции.')
    args = parser.parse_args()

    client = pymongo.MongoClient(SHARD_HOSTS_MAP[args.shard])
    collections = [client[args.collection][args.collection]]

    report = REPORTS_CLASSES_MAP[args.report]
    main(report, collections, shard_name=args.shard)
