#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
Скрипт обновления протобуфа в платформе.
Запускать из под своего пользователя, т.к. обновляет файлы в рабочей копии MPFS.
"""
import logging
import os
import re
import shlex
import subprocess
import sys
import shutil

MPFS_TMP_PATH = os.path.join('/tmp', 'mpfs-tmp')
MPFS_CONF_PATH = os.path.join(MPFS_TMP_PATH, 'conf')
MPFS_PROTO_PATH = os.path.join('/tmp', 'mpfs-proto')
PACKAGE_RE = re.compile(r'^import( public)? "(?P<package>.+)";', re.MULTILINE)
PROTOBUF_BASE_URL = 'http://127.0.0.1/v1/123/schema/protobuf/'


def is_mpfs_root(path):
    return os.path.exists(os.path.join(os.path.normpath(path), 'common', 'lib', 'mpfs'))


def get_mpfs_root():
    path = os.getcwd()

    while len(path.strip('/').split('/')) > 2 and not is_mpfs_root(path):
        path = os.path.dirname(path)
    if not is_mpfs_root(path):
        raise Exception('Unable to find MPFS Working Copy root directory.')

    return path


def sh(cmdline):
    args = shlex.split(cmdline)
    retcode = subprocess.call(args)
    if retcode != 0:
        raise Exception('Error executing command: "%s". Return code: %s.' % (cmdline, retcode))


def setup_configs(conf_path=MPFS_CONF_PATH, package='disk'):
    if os.path.exists(conf_path):
        shutil.rmtree(conf_path)
    os.makedirs(conf_path)
    mpfs_root = get_mpfs_root()
    path = os.path.join(mpfs_root, 'common', 'conf')
    configs = [os.path.join(path, p) for p in os.listdir(path)]
    try:
        path = os.path.join(mpfs_root, package, 'conf')
        configs += [os.path.join(path, p) for p in os.listdir(path)]
    except OSError:
        pass
    for c in configs:
        p = os.path.join(conf_path, os.path.basename(c))
        if os.path.exists(p):
            os.unlink(p)
        os.symlink(c, p)


def setup_environment(conf_path=MPFS_CONF_PATH):
    mpfs_root = get_mpfs_root()
    os.environ['PYTHONPATH'] = os.path.join(mpfs_root, 'common', 'lib')
    setup_configs(conf_path=conf_path)
    os.environ['MPFS_CONFIG_PATH'] = conf_path
    sys.path.insert(0, os.path.join(mpfs_root, 'common', 'lib'))

    import mpfs.engine.process

    # вырубаем логи
    from mpfs.common.util import logger
    logging.raiseExceptions = False
    logger.configure_logging({'version': 1})
    logging.basicConfig(level=logging.NOTSET)

    # инициализируем mpfs только после того как скопировали конфиги иначе он сменит пользователя на nginx,
    # и у нас не хватит прав чтоб скопировать конфиги в своей папке
    mpfs.engine.process.setup(setuid=False)


def is_program_installed(name):
    for path in os.environ['PATH'].split(os.pathsep):
        path = path.strip('"')
        file = os.path.join(path, name)
        if os.path.exists(file):
            return True
    return False


def main(conf_path=MPFS_CONF_PATH):
    # проверяем, что protoc установлен
    print 'Checking protobuf compiller.'
    if not is_program_installed('protoc'):
        raise Exception('Compiler for protocol buffer definition files not found. '
                        'Try "sudo apt-get install protobuf-compiler"')

    print 'Setting up environment.'
    setup_environment(conf_path=conf_path)

    from mpfs.platform.dispatchers import InternalDispatcher
    from mpfs.platform.handlers import GetApiInfoHandler, HeadApiInfoHandler
    from mpfs.platform.resources import NamespaceResource, res
    from mpfs.platform.routers import RegexRouter
    from mpfs.platform.system.system.handlers import GetPing
    from mpfs.platform.utils import build_flask_request
    from mpfs.platform.v1.resources import V1Namespace

    print 'Initializing platform.'
    class APIRootNamespace(NamespaceResource):
        relations = {
            'GET': GetApiInfoHandler,
            'HEAD': HeadApiInfoHandler,
            'ping': res(relations={'GET': GetPing}, hidden=True),
            'v1': V1Namespace,
        }
    router = RegexRouter(APIRootNamespace)
    dispatcher = InternalDispatcher(router)

    root_proto_name = 'cloud-api.proto'
    root_proto_path = os.path.join(MPFS_PROTO_PATH, root_proto_name)
    print 'Downloading root .proto-file to "%s".' % (root_proto_path,)
    req = build_flask_request('GET', url='%s%s' % (PROTOBUF_BASE_URL, root_proto_name,))
    resp = dispatcher.dispatch(req)
    if not os.path.exists(MPFS_PROTO_PATH):
        os.makedirs(MPFS_PROTO_PATH)
    with open(root_proto_path, 'w') as f:
        f.write(resp.content.encode('utf-8'))

    # скачиваем proto-файлы
    packages = []
    for match in PACKAGE_RE.finditer(resp.content):
        package = match.groupdict()['package']
        pkg_dir_path = os.path.join(MPFS_PROTO_PATH, *os.path.split(package)[:-1])
        pkg_path = os.path.join(MPFS_PROTO_PATH, package)

        print 'Downloading "%s" to "%s".' % (package, pkg_path,)

        req = build_flask_request('GET', url='%s%s' % (PROTOBUF_BASE_URL, package,))
        resp = dispatcher.dispatch(req)

        if not os.path.exists(pkg_dir_path):
            os.makedirs(pkg_dir_path)
        with open(pkg_path, 'w') as f:
            f.write(resp.content.encode('utf-8'))

        packages.append(package)

    # компилируем proto-файлы
    bin_path = os.path.join(MPFS_PROTO_PATH, 'bin')
    if not os.path.exists(bin_path):
        os.makedirs(bin_path)
    for package in packages:
        pkg_path = os.path.join(MPFS_PROTO_PATH, package)
        mod_path = os.path.splitext(package)[0]
        mod_name = os.path.split(mod_path)[-1]
        py_src_path = os.path.join(bin_path, '%s_pb2.py' % (mod_path,))
        py_dst_path = os.path.join(get_mpfs_root(), 'common', 'lib', 'mpfs', 'platform', mod_path, '%s_pb2.py' % (mod_name,))
        print 'Compilling "%s" to "%s".' % (pkg_path, py_src_path)
        sh('protoc -I="%s" --python_out "%s" "%s"' % (MPFS_PROTO_PATH, bin_path, pkg_path,))
        print 'Copying "%s" to "%s".' % (py_src_path, py_dst_path)
        sh('cp %s %s' % (py_src_path, py_dst_path,))

    print 'Done.'


if __name__ == '__main__':
    main()
