# -*- coding: utf-8 -*-
"""
Скрипт для работы с стартерком для MPFS-х нужд.

Как получить токен:
https://wiki.yandex-team.ru/users/triklozoid/jirampfsclient

Установка startrek_client:
sudo pip install https://github.yandex-team.ru/devtools/startrek-python-client/archive/1.0.1.zip
"""
from __future__ import unicode_literals

import getpass
import os
import re
import sys
import logging
import itertools
import tempfile
import time
import collections
import requests
import subprocess

logging.basicConfig(filename=os.path.join(tempfile.gettempdir(), 'st.log'), level='INFO')
requests.packages.urllib3.disable_warnings()

from startrek_client import Startrek
from argparse import ArgumentParser
from distutils.util import strtobool


RN_TMPL = """===Я.Диск. Release Notes. %(for_what)s===
%(release_type)s
Тестирование: %(testing_date)s
Продакшн: %(production_date)s

%(body)s

Релизный таск: https://st.yandex-team.ru/%(release_key)s
"""


class Color(object):
    """Раскрашивает текст в консоли"""
    HEADER = '\033[95m'
    OK_BLUE = '\033[94m'
    OK_GREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'

    _closing = '\033[0m'

    @classmethod
    def paint(cls, text, color):
        return "%s%s%s" % (color, text, cls._closing)

    @classmethod
    def ppaint(cls, text, color, fh=sys.stdout):
        print >>fh, cls.paint(text, color)


class MPFSStartrekClient(Startrek):
    mpfs_user_agent = 'Mozilla/5.0 (Windows NT 5.1; rv:31.0) Gecko/20100101 Firefox/31.0'
    mpfs_queue_name = 'DISKBACK'

    def __init__(self, **kwargs):
        self.mpfs_token = self.get_startrek_token()
        super(MPFSStartrekClient, self).__init__(useragent=self.mpfs_user_agent, token=self.mpfs_token, **kwargs)
        self._connection.session.verify = False
        self.mpfs_queue = self.queues[self.mpfs_queue_name]
        self._fix_versions_map = {}

    @staticmethod
    def get_startrek_token(token_path=None):
        """
        Получение токена для стартрека

        Места в порядке приоритета:
            * ENV->STARTREK_TOKEN
            * Переданный путь
            * ~/.startrek_mpfs_client_token
            * ~/.startrek_token
            * ~/.tracker_mpfs_client_token
        """
        if 'STARTREK_TOKEN' in os.environ:
            return os.environ['STARTREK_TOKEN']

        home_dir = os.path.expanduser('~')

        token_file_pathes = [
            os.path.join(home_dir, '.startrek_mpfs_client_token'),
            os.path.join(home_dir, '.startrek_token'),
            os.path.join(home_dir, '.tracker_mpfs_client_token'),
        ]
        if token_path:
            token_file_pathes.insert(0, token_path)

        for path in token_file_pathes:
            if os.path.exists(path):
                with open(path) as fh:
                    return fh.read().strip("\n")
        raise LookupError("Cant get startrek token")

    def get_fix_version(self, name, upsert=True):
        """Получить fix version по имени и при необходимости создать"""
        if name in self._fix_versions_map:
            return self._fix_versions_map[name]

        versions = self.mpfs_queue.versions
        search_result = [v for v in versions if v.name == name]
        if len(search_result) > 1:
            raise ValueError("Fix version name should be uniq")
        elif len(search_result) == 1:
            version = search_result[0]
        elif upsert:
            version = self.versions.create(queue=self.mpfs_queue_name, name=name)
        else:
            raise KeyError("Fix version not found")
        self._fix_versions_map[version.name] = version
        return version

    def get_issues(self, query):
        """Получение тикетов"""
        query = query.encode('utf-8')
        return list(self.issues.find(query))

    def create_issue(self, **kwargs):
        return self.issues.create(**kwargs)

    def update_fix_versions(self, issue, fix_versions):
        return issue.update(fixVersions=fix_versions)


class DryRunMPFSStartrekClient(MPFSStartrekClient):
    """Клиент для отладки"""
    def create_issue(self, **kwargs):
        class DummyIssue(object):
            pass
        dummy_issue = DummyIssue()
        dummy_issue.key = 'CHEMODAN-DRY-RUN'
        dummy_issue.summary = kwargs['summary']
        dummy_issue.assignee = DummyIssue()
        dummy_issue.assignee.display = kwargs['assignee']['login']
        return dummy_issue

    def get_fix_version(self, name, upsert=True):
        if upsert:
            try:
                return super(DryRunMPFSStartrekClient, self).get_fix_version(name, upsert=False)
            except KeyError:
                pass
        else:
            return super(DryRunMPFSStartrekClient, self).get_fix_version(name, upsert=False)

    def update_fix_versions(self, issue, fix_versions):
        pass


# чтобы было проще money patch-ить
ST_CLIENT = MPFSStartrekClient()
# для отладки:
#ST_CLIENT = DryRunMPFSStartrekClient()


class Release(object):
    version_template = "%(ver_major)s.%(ver_minor)s"
    full_version_template = "%(ver_major)s.%(ver_minor)s-%(build)s"
    package_component_map = {
        'disk': 'mpfs',
        'platform': 'REST API'
    }
    fix_version_template = "mpfs %(package)s %(ver_major)s.%(ver_minor)s"

    def __init__(self, package, version):
        if package not in self.package_component_map:
            raise ValueError('Bad package type: "%s"' % package)

        ver_major, ver_minor, build = self.parse_mpfs_version(version)

        self.package = package
        self.release_parts = {
            'package': package,
            'ver_major': ver_major,
            'ver_minor': ver_minor,
            'build': build,
        }

    @property
    def rvars(self):
        rvars = self.release_parts.copy()
        rvars['version'] = self.version
        rvars['full_version'] = self.full_version
        rvars['fix_version'] = self.fix_version
        rvars['component'] = self.component
        return rvars

    @property
    def version(self):
        return self.version_template % self.release_parts

    @property
    def fix_version(self):
        return self.fix_version_template % self.release_parts

    @property
    def component(self):
        return self.package_component_map[self.package]

    @property
    def full_version(self):
        return self.full_version_template % self.release_parts

    def __str__(self):
        return "%s %s" % (self.package, self.version)

    def __repr__(self):
        return "%s %s %s" % (self.__class__, self.package, self.full_version)

    @staticmethod
    def parse_mpfs_version(version):
        res = re.search(r'^(\d+)\.(\d+)(?:-(\d+))?$', version)
        if res:
            groups = res.groups()
            if groups[2] is None:
                groups = (groups[0], groups[1], 1)
            return [int(i) for i in groups]
        raise ValueError("Cant parse mpfs version")


class TicketError(Exception):
    pass


class NotUniqError(TicketError):
    pass


class AlreadyExistsError(TicketError):
    pass


class NotExistsError(TicketError):
    pass


class RelatedNotExistsError(NotExistsError):
    pass


class ReleaseTicket(object):
    """
    Обычный релизный тикет
    """
    follower_logins = [
        'akinfold',
        'kis8ya',
    ]
    is_uniq = True
    issue_dict = {
        'queue': 'DISKBACK',
        'type': {'name': 'Задача'},
        'followers': [{'login': l} for l in follower_logins],
        'followingMaillists': [
            'disk-mpfs@yandex-team.ru',
            'disk-test@yandex-team.ru'
        ],
    }

    packages_info = {
        'disk': {
            'summary': 'Протестировать mpfs для Диска, версия %(full_version)s',
            'search_query': 'Queue: DISKBACK Summary: "Протестировать mpfs для Диска, версия %(version)s" Summary: !"hotfix"',
            'assignee_login': getpass.getuser(),
            'components': {'name': 'mpfs'},
        },
        'platform': {
            'summary': 'Протестировать mpfs для Платформы, версия %(full_version)s',
            'search_query': 'Queue: DISKBACK Summary: "Протестировать mpfs для Платформы, версия %(version)s" Summary: !"hotfix"',
            'assignee_login': getpass.getuser(),
            'components': {'name': 'REST API'},
        },
    }

    def __init__(self, release):
        self.st_client = ST_CLIENT
        self.release = release
        self.package_info = self.packages_info[release.package]
        self._issues = None
        self._issues_in_release = None

    @property
    def assignee_login(self):
        return self.package_info.get('assignee_login', 'akinfold')

    @property
    def search_query(self):
        return self.package_info['search_query'] % self.release.rvars

    @property
    def summary(self):
        return self.package_info['summary'] % self.release.rvars

    @property
    def issues(self):
        """
        Получить тикеты, попадающие под `search_query`

        Проверяет тикет на соответсвие флагу `is_uniq`
        """
        # возвращает список
        if not self._issues:
            self._issues = self.st_client.get_issues(self.search_query)
            num = len(self._issues)
            if num == 0:
                raise NotExistsError()
            elif self.is_uniq and num > 1:
                raise NotUniqError()
        return self._issues

    def assert_can_be_created(self):
        # если тикет не должен быть уникальным, то ок
        if not self.is_uniq:
            return

        try:
            self.issues
        except NotExistsError:
            # не нашли тикет - ok
            pass
        else:
            # тикет уже есть - нельзя содавать новый
            e = AlreadyExistsError("Release issue already exists")
            e.ticket = self
            raise e

    def create(self, description=''):
        """Создание релизного тикета"""
        if not isinstance(description, unicode):
            raise TypeError('Bad description')

        self.assert_can_be_created()

        # создаем если нет fix version
        self.st_client.get_fix_version(self.release.fix_version, upsert=True)

        issue_dict = self.issue_dict.copy()
        issue_dict['fixVersions'] = [{'name': self.release.fix_version}]
        issue_dict['summary'] = self.summary
        issue_dict['description'] = description
        issue_dict['assignee'] = {'login': self.assignee_login}
        issue_dict['components'] = self.package_info['components']
        return self.st_client.create_issue(**issue_dict)


class HotFixTicket(ReleaseTicket):
    """
    Хот фикс тикет
    """
    is_uniq = False
    packages_info = {
        'disk': {
            'summary': 'Протестировать mpfs для Диска, версия %(full_version)s (hotfix)',
            'search_query': 'Queue: DISKBACK Summary: "Протестировать mpfs для Диска, версия %(version)s" Summary: "hotfix"',
            'assignee_login': getpass.getuser(),
            'components': {'name': 'mpfs'},
        },
        'platform': {
            'summary': 'Протестировать mpfs для Платформы, версия %(full_version)s (hotfix)',
            'search_query': 'Queue: DISKBACK Summary: "Протестировать mpfs для Платформы, версия %(version)s" Summary: "hotfix"',
            'assignee_login': getpass.getuser(),
            'components': {'name': 'REST API'},
        },
    }


class LoadTicket(ReleaseTicket):
    """
    Тикет на нагрузочное тестирование
    """
    packages_info = {
        'disk': {
            'summary': 'Нагрузочное тестирование mpfs для Диска, версия %(full_version)s',
            'search_query': 'Queue: DISKBACK Summary: "Нагрузочное тестирование mpfs для Диска, версия %(version)s" Summary: !"hotfix"',
            'assignee_login': 'akinfold',
            'components': {'name': 'Тестирование'}
        },
    }

    def create(self, description=''):
        release_ticket = ReleaseTicket(self.release)
        try:
            # получаем релизный тикет на тестирование
            main_issue = release_ticket.issues[0]
        except NotExistsError:
            try:
                # Если оказалось так, что таск, к которому хотим привязаться, не успел создасться,
                # то ждем и пробуем еще раз
                time.sleep(2)
                main_issue = release_ticket.issues[0]
            except NotExistsError as e:
                # если не найден базовый таск на тестирование, то не создаем
                # нагрузочный тикет
                raise RelatedNotExistsError, e, sys.exc_info()[2]
        full_description = 'Тикет на тестирование: %s' % main_issue.key
        if description:
            full_description = "%s\n%s" % (description, full_description)
        return super(LoadTicket, self).create(description=full_description)


class TestsReviewTicket(LoadTicket):
    """
    Тикет на ревью тестов
    """
    packages_info = {
        'disk': {
            'summary': 'Отревьювить тесты для Диска, версия %(full_version)s',
            'search_query': 'Queue: DISKBACK Summary: "Отревьювить тесты для Диска, версия %(version)s" Summary: !"hotfix"',
            'assignee_login': 'akinfold',
            'components': {'name': 'Тестирование'}
        },
        'platform': {
            'summary': 'Отревьювить тесты для Платформы, версия %(full_version)s',
            'search_query': 'Queue: DISKBACK Summary: "Отревьювить тесты для Платформы, версия %(version)s" Summary: !"hotfix"',
            'assignee_login': 'akinfold',
            'components': {'name': 'Тестирование'}
        },
    }


class ReleaseNoteHelper(object):
    DEFAULT_QUEUE = 'DISKBACK'
    rn_patter = re.compile(r'//RN\. ?(.+)//')
    rn_body_tmpl = "  * %s(https://st.yandex-team.ru/%s)"

    def __init__(self, issue, release_fix_version):
        self.issue = issue
        self.release_fix_version = release_fix_version
        self.fix_version = self.get_mpfs_fix_version(issue)
        self.release_note = self.get_release_note(issue)

    @staticmethod
    def get_mpfs_fix_version(issue):
        for fix_version in issue.fixVersions:
            if re.search(r'^mpfs (disk|platform)', fix_version.name):
                return fix_version
        return None

    @classmethod
    def get_release_note(cls, issue):
        text_gen = (c.text for c in issue.comments)
        rn_sources = itertools.chain([issue.description], text_gen)
        for src in rn_sources:
            if not src:
                continue
            result = cls.rn_patter.search(src)
            if result:
                return result.group(1)
        return None

    def has_release_note(self):
        return self.release_note is not None

    def is_resolved(self):
        if self.issue.queue != self.DEFAULT_QUEUE:
            return True
        return self.issue.resolution is not None

    def is_correct_fix_version(self):
        if self.issue.queue.key != self.DEFAULT_QUEUE:
            return True
        if {x for x in self.issue.components if x.name == 'djfs'}:
            return True
        return self.fix_version and self.fix_version.name == self.release_fix_version.name

    def get_group(self):
        return (self.has_release_note(), self.is_correct_fix_version(), self.is_resolved())

    def is_correct(self):
        return all(self.get_group())

    def format_to_rn(self):
        return self.rn_body_tmpl % (self.release_note, self.issue.key)


###################################


def _print_issues_list(issues, prefix=''):
    for issue in issues:
        assignee = issue.assignee.display if issue.assignee else '-'
        print "%shttps://st.yandex-team.ru/%s | %16.15s | %s | %s" % (prefix, issue.key, assignee, issue.status.name, issue.summary)


def user_yes_no_query(question):
    msg = '%s [y/n] ' % question
    sys.stdout.write(msg)
    while True:
        try:
            return strtobool(raw_input().lower())
        except ValueError:
            print "Please respond with 'y' or 'n'."
            sys.stdout.write(msg)


def my_exit(code):
    color = Color.OK_GREEN if code == 0 else Color.FAIL
    Color.ppaint('Выход...', color)
    exit(code)


def _generate_tickets(ticket_type, version, hotfix):
    if ticket_type is None or version is None:
        raise TypeError()

    available_packages = Release.package_component_map.keys()

    if hotfix:
        if ticket_type not in available_packages:
            raise NotImplementedError('Not implemented hotfix for "%s"' % ticket_type)
        release = Release(ticket_type, version)
        return [HotFixTicket(release)]
    else:
        if ticket_type == 'all':
            packages = available_packages
        elif ticket_type in available_packages:
            packages = [ticket_type]
        else:
            packages = []

        result = []
        for package in packages:
            release = Release(package, version)
            ticket = ReleaseTicket(release)
            ticket.assert_can_be_created()
            result.append(ticket)

        # нагрузочный тикет
        if ticket_type in ('all', 'disk_load'):
            release = Release('disk', version)
            result.append(LoadTicket(release))
        return result


DESC_TEMPLATE = """
((%(branch_url)s Релизная ветка %(version)s))
((%(diff_url)s DIFF (trunk:%(trunk_revision)s release-%(version)s:HEAD)))
((https://teamcity.yandex-team.ru/buildConfiguration/Mpfs_Arcadia_PythonMpfsDiskApiQueue Сборка релиза на TC))
"""

def _get_description_for_ticket(ticket):
    try:
        log_line = subprocess.check_output(
            'svn log --stop-on-copy --limit 1 -r0:HEAD ^/branches/disk/mpfs/releases/release-%s' % ticket.release.version,
            shell=True)
        trunk_revision = re.search('\nr(\d{6,10})', log_line).groups()[0]
    except Exception:
        trunk_revision = 'HEAD'

    params = {
        'version': ticket.release.version,
        'diff_url': "https://arc.yandex-team.ru/wsvn/arc?op=comp&compare[0]=/trunk/arcadia/disk/mpfs&compare_rev[0]=%s&compare[1]=/branches/disk/mpfs/releases/release-%s&compare_rev[1]=HEAD" % (trunk_revision, ticket.release.version),
        'branch_url': "https://a.yandex-team.ru/arc/history/branches/disk/mpfs/releases/release-%s" % ticket.release.version,
        'trunk_revision': trunk_revision,
    }
    return DESC_TEMPLATE % params


def do_create(ticket_type=None, version=None, hotfix=False):
    # Подготавливаем тикеты, которые пользователь хочет создать
    try:
        tickets = _generate_tickets(ticket_type, version, hotfix)
    except AlreadyExistsError as e:
        msg = 'Релиз: "%s". Тикет: "%s". Такой таск уже существует: https://st.yandex-team.ru/%s' % (e.ticket.release, e.ticket.__class__.__name__, e.ticket.issues[0].key)
        Color.ppaint(msg, Color.FAIL)
        my_exit(1)

    # Выводим на экран, что будет создано
    prefix = ' '
    Color.ppaint("Будет создано:", Color.HEADER)
    for ticket in tickets:
        description = _get_description_for_ticket(ticket)
        Color.ppaint(ticket.summary, Color.OK_BLUE)
        print description

    # Ждем подтверждения о создании
    if not user_yes_no_query("Продолжить?"):
        my_exit(0)

    # Обновляем у релиз кандидатов fix versions, а затем создаем релизный тикет
    Color.ppaint("Создание:", Color.HEADER)
    for ticket in tickets:
        # создаем релизный таск
        description = _get_description_for_ticket(ticket)
        try:
            release_issue = ticket.create(description=description)
        except RelatedNotExistsError:
            msg = '%sРелиз: "%s". Тикет: "%s". Не могу найти связанный таск по запросу' % (prefix, ticket.release, ticket.__class__.__name__)
            Color.ppaint(msg, Color.WARNING)
        except AlreadyExistsError:
            msg = '%sРелиз: "%s". Тикет: "%s". Такой таск уже существует: https://st.yandex-team.ru/%s' % (prefix, ticket.release, ticket.__class__.__name__, ticket.issues[0].key)
            Color.ppaint(msg, Color.WARNING)
        else:
            msg = '%sСоздан тикет для "%s". https://st.yandex-team.ru/%s' % (prefix, ticket.release, release_issue.key)
            Color.ppaint(msg, Color.OK_GREEN)

    my_exit(0)


def do_release_notes(ticket_key=None):
    if ticket_key is None:
        raise TypeError()
    if not re.search(r'^DISKBACK-\d+$', ticket_key):
        raise ValueError()

    release_issue = ST_CLIENT.issues[ticket_key]
    release_fix_version = ReleaseNoteHelper.get_mpfs_fix_version(release_issue)

    correct_issue_map = collections.defaultdict(list)
    other_issues = collections.defaultdict(list)
    for link in release_issue.links:
        issue = link.object
        helper = ReleaseNoteHelper(issue, release_fix_version)
        group_key = helper.get_group()
        if helper.is_correct():
            type_name = helper.issue.type.name
            correct_issue_map[type_name].append(helper)
        else:
            other_issues[helper.get_group()].append(helper.issue)

    rn_body = ''
    for type_name, helpers in sorted(correct_issue_map.items()):
        rn_body += "%s:\n" % type_name
        for helper in helpers:
            rn_body += "%s\n" % helper.format_to_rn()
        rn_body += "\n"
    rn_body = rn_body.rstrip()

    release_notes = RN_TMPL % {
        'for_what': release_issue.summary.split(' ', 1)[1],
        'testing_date': release_issue.createdAt.split('T')[0],
        'production_date': release_issue.updatedAt.split('T')[0],
        'release_type': 'Hotfix' if 'hotfix' in release_issue.summary else 'Плановый релиз',
        'body': rn_body,
        'release_key': release_issue.key,
    }
    Color.ppaint(release_notes, Color.OK_GREEN)

    if other_issues:
        Color.ppaint('Тикеты ниже привязаны к релизному таску, но не попадают в RN, т.к. не все 3 условия True:', Color.WARNING)
        Color.ppaint('(has_release_note, is_correct_fix_version, is_resolved)', Color.WARNING)
        for group, issues in other_issues.iteritems():
            print group
            _print_issues_list(issues, prefix='  ')


def main():
    command_map = {
        'create': do_create,
        'rn': do_release_notes,
    }
    parser = ArgumentParser(description=__doc__)
    sub_parser = parser.add_subparsers(dest='cmd_name')

    # create
    tasks_types = ['all', 'disk', 'platform', 'disk_load']
    action, help_text = 'create', 'Создать релизный таск.'
    sub_cmd_parser = sub_parser.add_parser(action, help=help_text)
    sub_cmd_parser.add_argument('--hotfix', action='store_true', help='hotfix таск. Тело тикета заполнять руками')
    sub_cmd_parser.add_argument('ticket_type', choices=tasks_types, help='Тип тикета. all - все типы тикетов.')
    sub_cmd_parser.add_argument('version', help='Версия пакета. Пример: 2.30 или 2.30-3')

    # rn
    sub_cmd_parser = sub_parser.add_parser('rn', help='Генерация release notes')
    sub_cmd_parser.add_argument('ticket_key', help='Номер задачи из ST. DISKBACK-NNN')

    # parse and launch
    namespace = vars(parser.parse_args())
    cmd_name = namespace.pop('cmd_name')
    command_map[cmd_name](**namespace)


if __name__ == "__main__":
    main()
