#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
MPFS Tools
Release Builder
"""
import ConfigParser
import os
import re
import subprocess

from time import time

import builder_base

from build_common import copy_deb_related_files, remove_deb_related_files
from util import dist


class RemoteMPFSRepo(object):
    """
    Получение различных вещей из remotes/origin
    """
    release_branch_tmpl = re.compile(r'refs/heads/release-([0-9]+).([0-9]+)$')

    @classmethod
    def releases(cls):
        """
        Получить все релизные версии в формете [(<major>, <minor>), ...]
        """
        releases = []
        raw_branches = subprocess.check_output('git ls-remote --heads', shell=True).splitlines()
        for raw_branch in raw_branches:
            res = cls.release_branch_tmpl.search(raw_branch)
            if res:
                releases.append(tuple([int(p) for p in res.groups()]))
        return releases

    @classmethod
    def last_release(cls):
        """
        Получить последний релиз в формате (<major>, <minor>)
        """
        return sorted(cls.releases())[-1]

    @classmethod
    def get_release_last_build(cls, major, minor):
        """
        Получить последнюю сборку релиза
        """
        version = '%i.%i' % (major, minor)
        if version != cls.get_current_branch_version():
            # нужно на случай, если выкачали только одну ветку сборки, и при этом есть более новая, из которой хотим достать версию
            cmd = 'git fetch origin release-%s:release-%s' % (version, version)
            subprocess.check_call(cmd, shell=True)
        builder_conf_line = subprocess.check_output('git show release-%i.%i:tools/builder.conf | grep "release"' % (major, minor), shell=True)
        return int(builder_conf_line.split('=')[1].strip())

    @classmethod
    def last_release_build(cls):
        """
        Получить последнюю сборку последнего релиза в формате: (<major>, <minor>, <build>)
        """
        major, minor = cls.last_release()
        build = cls.get_release_last_build(major, minor)
        return (major, minor, build)

    @classmethod
    def get_current_branch_version(cls):
        version = builder_base.get_branch_name().replace('release-', '')
        version = version.replace('-hotfix', '')
        return version

    @staticmethod
    def get_last_commit_hash():
        return subprocess.check_output('git rev-parse HEAD', shell=True).strip()


class YaUpload(object):
    """
    Отправка ресурса в Sandbox
    """

    TTL_INF = 'inf'

    def __init__(self, source_dir):
        self.source_dir = source_dir

    def get_ya(self):
        ya_path = '/tmp/ya'
        if not os.path.isfile(ya_path):
            subprocess.check_call(
                'svn export svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya %s' % ya_path,
                shell=True
            )
        return ya_path

    def pack(self, package_name):
        """
        Запаковать tar.gz
        """
        subprocess.check_call(
            'tar -C %s/debian/%s/ -cvzf %s/debian/data.tar.gz --exclude=DEBIAN .'
            % (self.source_dir, package_name, self.source_dir),
            shell=True
        )

    def upload(self, package_name, package_version, days_to_live):
        """Загрузка в sandbox.

        :param package_name: Имя пакета
        :param package_version: Версия пакета
        :param days_to_live: Сколько дней хранить ресурс ("inf" - вечно)
        """
        package_full_name = "%s=%s" % (package_name, package_version)
        token = os.getenv('SANDBOX_TOKEN')
        if not token:
            raise ValueError('SANDBOX_TOKEN env var required')
        # Время существования ресурса (в днях)
        time_to_live_option = '--ttl=%s' % days_to_live
        command = [
            self.get_ya(),
            'upload',
            '-T=DISK_COMPRESSED_RESOURCE_APPLICATION',  # Общий тип для приложений в Диске
            '-a=linux',
            '--owner=DISK-ADMIN',
            '--token=%s' % token,
            time_to_live_option,
            '-d="%s"' % package_full_name,  # Описание ресурса в виде package_full_name
            '-A', package_full_name,  # Заводим аттрибут <package_name> со значением <package_version> для фильтрации
            '%s/debian/data.tar.gz' % self.source_dir,
        ]
        subprocess.check_call(command)

    def pack_and_upload(self, package_name, package_version, days_to_live=TTL_INF):
        self.pack(package_name)
        self.upload(package_name, package_version, days_to_live)


def main_build():
    builder_base.add_backup()
    version = RemoteMPFSRepo.get_current_branch_version()

    config = ConfigParser.ConfigParser()
    config.optionxform = str
    config.read(builder_base.CONFIG_FILE)

    builder_base.set_config(config)

    last_release_on_dist = dist.get_last_disk_release_on_dist(version)
    # Если на dist загружена более новая версия, чем по нашему конфигу сборки, то
    # возьмем более актуальную версию
    release = max(int(config.get('global', 'release')),
                  last_release_on_dist)
    packages = config.items('packages')

    print 'Last git commit hash: %s' % RemoteMPFSRepo.get_last_commit_hash()
    print '"##teamcity[buildNumber \'%s-%s\']"' % (version, release + 1)
    builder_base.update_config(release, config=config)
    builder_base.update_mpfs_version(version, release=release)

    subprocess.check_call('cp apps/setup.py .', shell=True)

    new_release = release + 1

    source_dir = os.path.abspath(os.curdir)
    ya_uploader = YaUpload(source_dir)

    for package_type, package_name in packages:
        copy_deb_related_files(source_dir, package_type)
        builder_base.build_pkg(package_type, package_name, version, new_release)

        package_version = "%s-%s" % (version, new_release)
        os.chdir('../')
        ya_uploader.pack_and_upload(package_name, package_version)

        for to in ('yandex-precise', 'yandex-trusty', 'yandex-disk-common'):
            cmd = "dupload --to %s %s_amd64.changes --nomail" % (to, "%s_%s" % (package_name, package_version))
            subprocess.check_call(cmd, shell=True)

        remove_deb_related_files(source_dir, package_type)
        os.chdir(source_dir)

    subprocess.check_call('rm setup.py', shell=True)

    subprocess.check_call('git add apps/{disk,browser,api,queue}/deploy/debian/changelog lib/mpfs/__init__.py tools/builder.conf', shell=True)
    subprocess.check_call('git commit -m "building all: version %s release %s platform deb"' % (version, new_release), shell=True)
    subprocess.check_call('git push', shell=True)
    return "%s-%s" % (version, new_release)


if __name__ == "__main__":
    print "\tStart buildrelease\n"
    start = time()
    main_build()
    print "\n\tBuildrelease finished in %.3f seconds.\n" % (time() - start)
