package ru.yandex.chemodan.app.notes.api;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notes.core.NotesManager;
import ru.yandex.chemodan.app.notes.core.ResultWithRevision;
import ru.yandex.chemodan.app.notes.core.model.notes.Note;
import ru.yandex.chemodan.app.notes.core.model.notes.NoteCreation;
import ru.yandex.chemodan.app.notes.core.model.notes.NotePatch;
import ru.yandex.chemodan.app.notes.core.model.notes.NoteWithMdsKey;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByBender;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestHeader;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.result.pojo.ActionResultPojo;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 * @author vpronto
 */
@ActionContainer
@AllArgsConstructor
public class NotesActions {
    private final NotesManager notesManager;

    @Path(value = "/notes", methods = HttpMethod.GET)
    public NotesActions.NotesList listNotes(@RequestParam("__uid") DataApiUserId uid) {
        return new NotesActions.NotesList(notesManager.listNotes(uid));
    }

    @Path(value = "/notes/{id}", methods = HttpMethod.GET)
    public ResultWithRevision<NoteWithMdsKey> getNote(
            @RequestParam("__uid") DataApiUserId uid,
            @PathParam("id") String id,
            @RequestParam("mdsKey") Option<Boolean> getMdsKey)
    {
        return notesManager.findNote(uid, id, getMdsKey.getOrElse(false));
    }

    @Path(value = "/notes", methods = HttpMethod.POST)
    public ResultWithRevision<Note> createNote(
            @RequestParam("__uid") DataApiUserId uid,
            @RequestHeader(value = "X-Request-Id") Option<String> requestId,
            @BoundByBender NoteCreation noteCreation)
    {
        return notesManager.createNote(uid, noteCreation.withoutInnerFields(), requestId);
    }

    @Path(value = "/notes", methods = HttpMethod.PUT)
    public ResultWithRevision<Note> createNoteWithContentAction(
            @RequestParam("__uid") DataApiUserId uid,
            @RequestHeader(value = "X-Request-Id", required = false) Option<String> requestId,
            @RequestHeader(value = "X-Mtime", required = false) Option<Instant> clientTime,
            @SpecialParam HttpServletRequestX req)
    {
        return notesManager.createNoteWithContent(
                uid, Option.empty(), requestId, clientTime.getOrElse(Instant.now()), req);
    }

    @Path(value = "/notes/{id}", methods = HttpMethod.DELETE)
    public ResultWithRevision<Void> deleteNote(
            @RequestParam("__uid") DataApiUserId uid,
            @PathParam("id") String id,
            @SpecialParam HttpServletRequestX req)
    {
        return notesManager.deleteNote(uid, id, req);
    }

    @Path(value = "/notes/{id}", methods = HttpMethod.PATCH)
    public ResultWithRevision<?> patchNote(
            @RequestParam("__uid") DataApiUserId uid,
            @RequestHeader(value = "X-Request-Id") Option<String> requestId,
            @PathParam("id") String id,
            @BoundByBender NotePatch patch)
    {
        return notesManager.patchNote(uid, id, patch.withoutContentRevision(), requestId, false);
    }

    //for test purposes only
    @Path(value = "/notes/initial", methods = HttpMethod.POST)
    public ResultWithRevision<Note> createInitialNote(
            @RequestParam("__uid") DataApiUserId uid)
    {
        return notesManager.createInitialNote(uid);
    }

    @ActionResultPojo
    @BenderBindAllFields
    private static class NotesList {
        public final ListF<Note> items;

        public NotesList(ListF<Note> items) {
            this.items = items;
        }
    }
}
