package ru.yandex.chemodan.app.notes.api;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notes.core.NotesManager;
import ru.yandex.chemodan.app.notes.core.ResultWithRevision;
import ru.yandex.chemodan.app.notes.core.model.notes.Attachment;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentAddition;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentCreation;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentCreationResult;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentQuery;
import ru.yandex.chemodan.app.notes.core.model.notes.Attachments;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByBender;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestHeader;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;

/**
 * @author tolmalev
 * @author vpronto
 */
@ActionContainer
@AllArgsConstructor
public class NotesAttachmentActions {
    private final NotesManager notesManager;

    @Path(value = "/notes/{note_id}/attachments", methods = HttpMethod.GET)
    public ResultWithRevision<Attachments> getAttachmentsAction(
            @RequestParam("__uid") DataApiUserId uid,
            @RequestParam(value = "limit", required = false) Option<Integer> limit,
            @RequestParam(value = "offset", required = false) Option<Integer> offset,
            @PathParam("note_id") String noteId,
            @RequestHeader(value = "X-Actual-Revision", required = false) Option<Long> revision)
    {
        return notesManager.getAttachments(AttachmentQuery.builder().uid(uid).noteId(noteId).rev(revision)
                .limit(limit).offset(offset).build());
    }

    @Path(value = "/notes/{note_id}/attachments/{resource_id}", methods = HttpMethod.GET)
    public ResultWithRevision<Attachment> getAttachmentsActionById(
            @RequestParam("__uid") DataApiUserId uid,
            @PathParam("note_id") String noteId,
            @PathParam("resource_id") String resourceId)
    {
        return notesManager.getAttachment(uid, noteId, resourceId);
    }

    @Path(value = "/notes/{note_id}/attachments/{resource_id}", methods = HttpMethod.DELETE)
    public ResultWithRevision<Void> deleteAttachmentsAction(
            @RequestParam("__uid") DataApiUserId uid,
            @PathParam("note_id") String noteId,
            @PathParam("resource_id") String resourceId,
            @RequestHeader(value = "X-Mtime", required = false) Option<Instant> clientTime,
            @RequestHeader(value = "X-Actual-Revision", required = false) Option<Long> revision,
            @RequestHeader(value = "User-Agent", required = false) Option<String> userAgent)
    {
        return notesManager.deleteAttachment(
                uid, noteId, resourceId, revision, clientTime.getOrElse(Instant.now()), userAgent);
    }

    @Path(value = "/notes/{note_id}/attachments", methods = HttpMethod.PUT)
    public ResultWithRevision<AttachmentCreationResult> putAttachmentsAction(
            @RequestParam("__uid") DataApiUserId uid,
            @PathParam("note_id") String noteId,
            @BoundByBender AttachmentCreation creation,
            @RequestHeader(value = "X-Mtime", required = false) Option<Instant> clientTime,
            @RequestHeader(value = "X-Actual-Revision", required = false) Option<Long> revision)
    {
        return notesManager.addAttachments(AttachmentAddition.builder()
                .uid(uid).noteId(noteId).name(creation.getName()).id(creation.getId()).revision(revision).build(),
                clientTime.getOrElse(Instant.now()));
    }

    @Path(value = "/notes/{note_id}/attachments/synchronize", methods = {HttpMethod.GET, HttpMethod.POST})
    public Object synchronizeAttachmentsAction(
            @RequestParam("__uid") DataApiUserId uid,
            @PathParam("note_id") String noteId)
    {
        notesManager.synchronizeAttachments(uid, noteId, Option.empty());
        return new OkPojo();
    }
}
