package ru.yandex.chemodan.app.notes.core;

import java.util.Arrays;
import java.util.Collection;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notes.core.model.notes.NoteTag;
import ru.yandex.chemodan.app.notes.core.model.tag.Tag;
import ru.yandex.chemodan.app.notes.core.model.tag.TagType;
import ru.yandex.chemodan.app.notes.dao.NotesRequestsDao;
import ru.yandex.chemodan.app.notes.dao.model.RequestRecord;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author vpronto
 */
@RequiredArgsConstructor
public class AbstractManager {

    protected static final String DS_APP = "yanotes";
    protected static final String DB_ID = "notes";

    protected static final String TAGS_COLLECTION = "tags";
    protected static final String NOTES_COLLECTION = "notes";

    protected static final DatabaseRef DATABASE_REF = DatabaseRef.cons(Option.of(DS_APP), DB_ID);

    protected final DataApiManager dataApiManager;
    protected final NotesRequestsDao requestsDao;

    protected UserDatabaseSpec dbSpec(DataApiUserId uid) {
        return new UserDatabaseSpec(uid, DATABASE_REF);
    }

    protected Option<Database> findDb(DataApiUserId uid) {
        return dataApiManager.getDatabaseO(dbSpec(uid));
    }

    protected <T> T withExistingDbOrThrow(DataApiUserId uid, Function<Database, T> fn) {
        return findDb(uid).map(fn).getOrThrow(() -> new NotFoundException("User has no database"));
    }

    protected <T> T withExistingDbOrElse(DataApiUserId uid, Function<Database, T> fn, Function0<T> elseFn) {
        return findDb(uid).map(fn).getOrElse(elseFn);
    }

    protected DataApiRecord toDataRecord(TagType tagType) {
        return toDataRecord(new Tag(tagType.getPredefinedId(), tagType, Option.empty()));
    }

    protected DataApiRecord toDataRecord(Tag tag) {
        MapF<String, DataField> fields = Cf.hashMap();

        fields.put("type", DataField.string(tag.type.toString().toLowerCase()));
        tag.value.forEach(value -> fields.put("value", DataField.string(value)));

        return new SimpleDataRecord(
                recordId(tag.id),
                new DataFields(fields)
        );
    }

    protected Option<RequestRecord> checkExists(DataApiUserId uid, Option<String> requestId) {
        if (requestId == null || !requestId.isPresent() || StringUtils.isBlank(requestId.get())) {
            return Option.empty();
        }
        return requestsDao.find(RequestRecord.builder().uid(uid).requestId(requestId.get()).build());
    }

    protected Delta initTags() {
        return Delta.fromNewRecords(Arrays.stream(TagType.values()).filter(TagType::isSystem)
                .map(this::toDataRecord).toArray(DataApiRecord[]::new));
    }

    protected boolean containsTag(Collection<NoteTag> tags, Long predefinedId) {
        return getTag(tags, predefinedId).isPresent();
    }

    protected Option<NoteTag> getTag(Collection<NoteTag> tags, Long predefinedId) {
        return Option.x(tags.stream().filter(t -> t.id.equals(predefinedId)).findFirst());
    }

    protected Delta createDeltaTag(Tag tag) {
        return Delta.fromNewRecords(toDataRecord(tag));
    }

    protected RecordId recordId(long id) {
        return new SimpleRecordId(TAGS_COLLECTION, Long.toString(id));
    }

    protected RecordId recordId(String id) {
        return new SimpleRecordId(NOTES_COLLECTION, id);
    }

}
