package ru.yandex.chemodan.app.notes.core;

import java.util.concurrent.Callable;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notes.core.model.notes.Attachment;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentAddition;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentCreationResult;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentQuery;
import ru.yandex.chemodan.app.notes.core.model.notes.Attachments;
import ru.yandex.chemodan.app.notes.dao.NotesDao;
import ru.yandex.chemodan.app.notes.dao.model.NoteRecord;
import ru.yandex.chemodan.mpfs.MpfsCallbackResponse;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMeta;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsStoreOperation;
import ru.yandex.chemodan.mpfs.MpfsStoreOperationContext;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author vpronto
 */
@AllArgsConstructor
public class NotesAttachmentsManagerImpl implements NotesAttachmentsManager {

    private final static Logger logger = LoggerFactory.getLogger(NotesAttachmentsManagerImpl.class);

    private final MpfsClient mpfsClient;
    private final NotesDao notesDao;
    private final NotesAttachmentsProperties properties;
    private final String callbackUrl;

    @Override
    public boolean needInit(DataApiUserId uid) {
        return mpfsClient.needInit(uid.forMpfs());
    }

    @Override
    public void userInit(DataApiUserId uid, Language locale) {
        mpfsClient.userInit(uid.forMpfs(), locale, NOTES_SOURCE, Option.empty());
    }

    @Override
    public ResultWithRevision<AttachmentCreationResult> addAttachments(AttachmentAddition addition, Instant mtime) {
        Long revision = addition.getRevision()
                .getOrElse(() -> getOrThrow(addition.getUid(), addition.getNoteId()).lastRevision);
        mkdir(addition.getUid(), generatePath(addition.getNoteId()));
        MpfsStoreOperation store;
        try {
            store = mpfsClient.store(MpfsStoreOperationContext.builder()
                            .uid(addition.getUid().forMpfs())
                            .callback(buildCallbackUrl(addition))
                            .addParam(NOTE_REVISION_CREATED, revision)
                            .addParam(NOTE_NAME, addition.getName())
                            .addParam(NOTE_ID, checkId(addition.getId()))
                            .addParam(NOTE_ATTACHMENT_MTIME, mtime)
                            .path(generatePath(addition.getNoteId(), addition.getId()))
                            .force(true)
                            .build(),
                    Tuple2List.tuple2List());
        } catch (PermanentHttpFailureException e) {
            throw new A3ExceptionWithStatus("add-attachment-error", e.getStatusCode().get());
        }

        String url = store.getUploadUrl().getOrThrow(() -> {
            logger.error("can't upload {}", store);
            return new A3ExceptionWithStatus("revision-conflict",
                    store.getStatus().getOrElse("can't create file"),
                    HttpStatus.SC_409_CONFLICT);
        });

        return new ResultWithRevision<>(new AttachmentCreationResult(url, store.getOid()), revision);
    }

    private Option<String> buildCallbackUrl(AttachmentAddition addition) {
        return Option.of(callbackUrl +
                "/api/notes/" + addition.getNoteId() + "/attachments/synchronize?__uid=" + addition.getUid());
    }

    @Override
    public ResultWithRevision<Void> deleteAttachment(
            DataApiUserId uid, String noteId, String resourceId, Option<Long> rev, Instant mtime)
    {
        String id = getIdByResourceId(uid, noteId, resourceId);
        Long revision = rev.getOrElse(() -> getOrThrow(uid, noteId).lastRevision);
        wrapNotFound(() -> {
            mpfsClient.setprop(uid.forMpfs(), generatePath(noteId, id),
                    Cf.map(NOTE_REVISION_DELETED, revision.toString()));
            return null;
        });
        return new ResultWithRevision<>(revision);
    }

    @Override
    public ResultWithRevision<Attachment> getAttachment(DataApiUserId uid, String noteId, String resourceId) {
        NoteRecord note = getOrThrow(uid, noteId);
        MpfsFileInfo fileInfoByFileId;
        try {
            fileInfoByFileId = getMpfsFileInfo(uid, noteId, resourceId);
        } catch (PermanentHttpFailureException e) {
            if (e.getStatusCode().isPresent()) {
                throw new A3ExceptionWithStatus("get-attachment-error", e.getStatusCode().get());
            } else {
                throw e;
            }
        }
        return new ResultWithRevision<>(Attachment.builder()
                .id(fileInfoByFileId.name)
                .name(fetchStringFromMeta(fileInfoByFileId, NOTE_NAME))
                .mediaType(fileInfoByFileId.getMeta().getMediaType())
                .mtime(fileInfoByFileId.getTimes().mtime)
                .createdRev(fetchLongFromMeta(fileInfoByFileId, NOTE_REVISION_CREATED).get())
                .deletedRev(fetchLongFromMeta(fileInfoByFileId, NOTE_REVISION_DELETED))
                .resourceId(fileInfoByFileId.getMeta().getResourceId().get().serialize())
                .build(), note.lastRevision);
    }

    @Override
    public ResultWithRevision<Attachments> getAttachments(AttachmentQuery query) {
        Long revision = query.getRev().getOrElse(() -> {
            NoteRecord record = getOrThrow(query.getUid(), query.getNoteId());
            return record.lastRevision;
        });
        Integer lim = query.getLimit().getOrElse(properties::getRequestLimit);
        ListF<Attachment> attachments = Cf.arrayList();
        String fullPath = generatePath(query.getNoteId());
        try {
            mpfsClient.streamingListByUidAndPath(query.getUid().forMpfs(), fullPath, Option.of(0), Option.of(1000),
                    Option.of("mtime"),
                    false,
                    mpfsFileInfoIterator -> {
                        while (mpfsFileInfoIterator.hasNext()) {
                            MpfsFileInfo next = mpfsFileInfoIterator.next();
                            Option<Long> created = fetchLongFromMeta(next, NOTE_REVISION_CREATED);
                            Option<Long> deleted = fetchLongFromMeta(next, NOTE_REVISION_DELETED);
                            Option<String> noteName = fetchStringFromMeta(next, NOTE_NAME);

                            if (infoMatchesRevision(next, revision)) {
                                attachments.add(Attachment.builder()
                                        .createdRev(created.get())
                                        .deletedRev(deleted)
                                        .name(noteName)
                                        .id(next.name)
                                        .mediaType(next.getMeta().getMediaType())
                                        .mtime(next.getTimes().mtime)
                                        .resourceId(next.getMeta().getResourceId().get().serialize())
                                        .build());
                            }
                        }
                    });
        } catch (HttpException e) {
            if (e.getStatusCode().isSome(HttpStatus.SC_404_NOT_FOUND)) {
                logger.info("Skipping 404");
            }
        }

        Integer off = query.getOffset().getOrElse(0);
        Integer start = Math.min(off, attachments.size());
        Integer end = Math.min(off + lim, attachments.size());
        ListF<Attachment> sublist = attachments.subList(start, end);
        return new ResultWithRevision<>(new Attachments(sublist, lim, off), revision);
    }

    @Override
    public ListF<MpfsFileInfo> getAttachmentsInfos(DataApiUserId uid, String noteId, Option<Long> revO) {
        long rev = revO.getOrElse(() -> getOrThrow(uid, noteId).getLastRevision());
        return getAttachmentsInfos(uid, noteId, i -> infoMatchesRevision(i, rev));
    }

    private ListF<MpfsFileInfo> getAttachmentsInfos(
            DataApiUserId uid, String noteId, Function1B<MpfsFileInfo> matcher)
    {
        String fullPath = generatePath(noteId);
        try {
            return mpfsClient
                    .listByUidAndPath(uid.forMpfs(), fullPath, Option.of(0), Option.of(1000), Option.of("mtime"), false)
                    .getChildren().filter(matcher);
        } catch (PermanentHttpFailureException e) {
            if (e.getStatusCode().isSome(HttpStatus.SC_404_NOT_FOUND)) {
                logger.info("Skipping 404");
            }
        }
        return Cf.list();
    }

    @Override
    public void copyAttachments(
            DataApiUserId uid, String srcId, String dstId, long dstRev, Option<ListF<String>> resourceIds)
    {
        MpfsUser mpfsUser = uid.forMpfs();
        String dstRootPath = generatePath(dstId);

        Function1B<MpfsFileInfo> matcher = resourceIds.isPresent()
                ? i -> i.getMeta().getResourceId().exists(r -> resourceIds.get().containsTs(r.serialize()))
                : i -> infoMatchesRevision(i, getOrThrow(uid, srcId).getLastRevision());

        ListF<MpfsFileInfo> attachmentsInfos = getAttachmentsInfos(uid, srcId, matcher);

        if (attachmentsInfos.isEmpty()) {
            return;
        }

        mpfsClient.mkdir(mpfsUser, dstRootPath);
        attachmentsInfos.forEach(mpfsFileInfo -> {
            if (!mpfsFileInfo.getPath().isPresent() || !mpfsFileInfo.getName().isPresent()) {
                return;
            }
            String srcPath = mpfsFileInfo.getPath().get();
            String dstPath = dstRootPath + "/" + mpfsFileInfo.getName().get();

            throwExceptionIfNotOk(
                    mpfsClient.copy(mpfsUser, srcPath, dstPath, true),
                    "copy-problem",
                    "Failed to copy " + mpfsFileInfo.getName().get() + " attachment to " + dstId
            );

            throwExceptionIfNotOk(
                    mpfsClient.setProp(mpfsUser, dstPath,
                            Tuple2List.fromPairs(NOTE_REVISION_CREATED, String.valueOf(dstRev)),
                            Cf.list(NOTE_REVISION_DELETED)),
                    "sep-prop-problem",
                    "Failed to reset revisions for " + dstPath);
        });
    }

    @Override
    public void removeAttachmentsFromMpfs(DataApiUserId uid, String noteId) {
        try {
            mpfsClient.rm(uid.forMpfs(), generatePath(noteId));
        } catch (PermanentHttpFailureException e) {
            if (!e.getStatusCode().isSome(HttpStatus.SC_404_NOT_FOUND)) {
                throw e;
            }
        }
    }

    @Override
    public ListF<String> copyInitialAttachments(
            DataApiUserId uid, String noteId, Language lang, Instant mtime, long rev)
    {
        String srcPath = INITIAL_NOTE_PREFIX + lang.value();
        MapF<String, Object> metaParams = Cf.map(
                NOTE_REVISION_CREATED, rev,
                NOTE_ATTACHMENT_MTIME, mtime);

        throwExceptionIfNotOk(
                mpfsClient.initNotes(uid.forMpfs(), srcPath, noteId, metaParams),
                "attach-init-problems", "Failed to copy initial attachments");

        return getAttachmentsInfos(uid, noteId, i -> true)
                .map(MpfsFileInfo::getMeta)
                .filterMap(MpfsFileMeta::getResourceId)
                .map(MpfsResourceId::serialize);
    }

    private void throwExceptionIfNotOk(MpfsCallbackResponse response, String name, String message) {
        if (response.getStatusCode() != HttpStatus.SC_200_OK) {
            throw new A3ExceptionWithStatus(name, message + ". Got " + response,
                    HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
        }
    }

    private boolean infoMatchesRevision(MpfsFileInfo info, long revision) {
        Option<Long> created = fetchLongFromMeta(info, NOTE_REVISION_CREATED);
        Option<Long> deleted = fetchLongFromMeta(info, NOTE_REVISION_DELETED);
        return (!deleted.isPresent() || deleted.isMatch(d -> d > revision)) && (created.isMatch(c -> c <= revision));
    }

    private Option<String> fetchStringFromMeta(MpfsFileInfo next, String key) {
        return getNode(next, key).flatMapO(node -> Option.of(node.getString()));
    }

    private Option<Long> fetchLongFromMeta(MpfsFileInfo next, String key) {
        return getNode(next, key).flatMapO(node -> Option.of(Long.valueOf(node.getString())));
    }

    private Option<BenderJsonNode> getNode(MpfsFileInfo next, String key) {
        return next.getMeta().getMetaJsonField(key);
    }

    private NoteRecord getOrThrow(DataApiUserId uid, String id) {
        Option<NoteRecord> note = notesDao.findNote(uid, id);
        return note.getOrThrow(() -> new NotFoundException("Note doesn't exists: " + id));
    }

    private String generatePath(String noteId, String fileName) {
        return generatePath(noteId) + "/" + fileName;
    }

    private String generatePath(String noteId) {
        return properties.getNotesRoot() + "/" + noteId;
    }

    private boolean isMpfsResourceId(String resourceId) {
        return resourceId.contains(":");
    }

    private String checkId(String id) {
        if (isMpfsResourceId(id) || id.contains("/") || id.contains("\\")) {
            throw new IllegalArgumentException("Illegal symbols [:/\\] in : " + id);
        }
        return id;
    }

    private void mkdir(DataApiUserId uid, String dir) {
        try {
            mpfsClient.mkdir(uid.forMpfs(), dir);
        } catch (PermanentHttpFailureException e) {
            if (e.getStatusCode().isMatch(c ->
                    c == HttpStatus.SC_409_CONFLICT || c == HttpStatus.SC_405_METHOD_NOT_ALLOWED))
            {
                logger.warn("Can't create dir: {}, for: {}", dir, uid, e);
            } else {
                throw e;
            }
        }
    }

    private String getIdByResourceId(DataApiUserId uid, String noteId, String resourceId) {
        if (isMpfsResourceId(resourceId)) {
            ResultWithRevision<Attachment> attachment = getAttachment(uid, noteId, resourceId);
            return attachment.getResult().id.getOrThrow("Can't get id for " + attachment);
        }
        return resourceId;
    }

    private <T> T wrapNotFound(Callable<T> function) {
        try {
            return function.call();
        } catch (HttpException e) {
            if (e.getStatusCode().isSome(HttpStatus.SC_404_NOT_FOUND)) {
                throw new NotFoundException("Resource doesn't exists " + e.getMessage());
            } else {
                throw e;
            }
        } catch (PermanentHttpFailureException e) {
            if (e.getStatusCode().isSome(HttpStatus.SC_404_NOT_FOUND)) {
                throw new NotFoundException("Resource doesn't exists " + e.getMessage());
            } else {
                throw e;
            }
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private MpfsFileInfo getMpfsFileInfo(DataApiUserId uid, String noteId, String resourceId) {
        return wrapNotFound(getMpfsFileInfoF(uid, noteId, resourceId));
    }

    private Callable<MpfsFileInfo> getMpfsFileInfoF(DataApiUserId uid, String noteId, String resourceId) {
        if (isMpfsResourceId(resourceId)) {
            return () -> mpfsClient
                    .bulkInfoByResourceIds(uid.forMpfs(), META_FIELDS, Cf.list(resourceId), Cf.list("/notes"))
                    .firstO()
                    .getOrThrow(() -> new NotFoundException(resourceId));
        } else {
            return () -> mpfsClient.getFileInfoByUidAndPath(uid.forMpfs(), generatePath(noteId, resourceId), Cf.list(NOTE_NAME, NOTE_REVISION_CREATED,
                    NOTE_REVISION_DELETED, "mediatype", "resource_id"));
        }
    }
}
