package ru.yandex.chemodan.app.notes.core;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ExecutorService;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.apache.http.entity.ContentType;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.internal.Validate;
import ru.yandex.chemodan.app.dataapi.ErrorNames;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPublicUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notes.api.AddRevisionInterceptor;
import ru.yandex.chemodan.app.notes.core.model.notes.Attachment;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentAddition;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentCreationResult;
import ru.yandex.chemodan.app.notes.core.model.notes.AttachmentQuery;
import ru.yandex.chemodan.app.notes.core.model.notes.Attachments;
import ru.yandex.chemodan.app.notes.core.model.notes.Note;
import ru.yandex.chemodan.app.notes.core.model.notes.NoteCreation;
import ru.yandex.chemodan.app.notes.core.model.notes.NotePatch;
import ru.yandex.chemodan.app.notes.core.model.notes.NoteTag;
import ru.yandex.chemodan.app.notes.core.model.notes.NoteWithMdsKey;
import ru.yandex.chemodan.app.notes.core.model.tag.TagType;
import ru.yandex.chemodan.app.notes.dao.NotesDao;
import ru.yandex.chemodan.app.notes.dao.NotesRequestsDao;
import ru.yandex.chemodan.app.notes.dao.NotesRevisionsDao;
import ru.yandex.chemodan.app.notes.dao.model.NoteRecord;
import ru.yandex.chemodan.app.notes.dao.model.RequestRecord;
import ru.yandex.chemodan.app.notes.dao.model.RevisionRecord;
import ru.yandex.chemodan.app.notes.tasks.RemoveDeletedNotesTask;
import ru.yandex.chemodan.app.notes.tasks.SyncAttachResourceIdsTask;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMeta;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.util.bender.ISOInstantUnmarshaller;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.chemodan.util.http.RequestUtils;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsFileNotFoundException;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.web.servlet.HttpServletResponseX;

/**
 * @author tolmalev
 */
public class NotesManagerImpl extends AbstractManager implements NotesManager {

    private final static Logger logger = LoggerFactory.getLogger(NotesManagerImpl.class);

    private final static byte[] EMPTY_CONTENT = "{}".getBytes();

    private final static Pattern2 oldClientPattern = Pattern2.compile("^Yandex\\.Disk.*\"vsn\":\"3\\.0\\..*");

    public final DynamicProperty<Boolean> createInitialNote = new DynamicProperty<>("notes.create-initial-note", false);
    public final DynamicProperty<String> defaultInitialNoteLang =
            new DynamicProperty<>("notes.initial-note-default-lang", "ru");
    public final DynamicProperty<Integer> removeNotesDelay = new DynamicProperty<Integer>("notes.remove-delay[d]", 30);

    private final NotesDao notesDao;
    private final NotesRevisionsDao revisionsDao;
    private final NotesRequestsDao requestsDao;
    private final NotesContentManager contentManager;
    private final NotesAttachmentsManager notesAttachmentsManager;
    private final Blackbox2 blackbox2;
    private final BazingaTaskManager bazingaTaskManager;
    private final ExecutorService syncAttachmentsResourcesService;
    private final Duration attachmentsSyncDelay;
    private final int retiresOnConflictCount;

    public NotesManagerImpl(DataApiManager dataApiManager, NotesDao notesDao, NotesRevisionsDao notesRevisionsDao,
            NotesRequestsDao requestsDao, NotesContentManager contentManager,
            NotesAttachmentsManager notesAttachmentsManager,
            BazingaTaskManager bazingaTaskManager,
            Blackbox2 blackbox2,
            ExecutorService syncAttachmentsResourcesService,
            Duration attachmentsSyncDelay,
            int retiresOnConflictCount)
    {
        super(dataApiManager, requestsDao);
        this.notesDao = notesDao;
        this.revisionsDao = notesRevisionsDao;
        this.requestsDao = requestsDao;
        this.contentManager = contentManager;
        this.notesAttachmentsManager = notesAttachmentsManager;
        this.bazingaTaskManager = bazingaTaskManager;
        this.blackbox2 = blackbox2;
        this.syncAttachmentsResourcesService = syncAttachmentsResourcesService;
        this.attachmentsSyncDelay = attachmentsSyncDelay;
        this.retiresOnConflictCount = retiresOnConflictCount;
    }

    private ListF<Note> findAllNotes(Database db) {
        ListF<DataRecord> records = dataApiManager
                .getRecords(UserDatabaseSpec.fromDatabase(db),
                        RecordsFilter.DEFAULT
                                .withCollectionId(AbstractManager.NOTES_COLLECTION));

        return records.map(this::parseFromDs);
    }

    @Override
    public ResultWithRevision<Void> init(DataApiUserId uid, Option<String> requestId,
            Option<Language> locale, Option<String> userAgent)
    {
        Language lang = locale.getOrElse(resolveUserLanguage(uid));
        Option<InitialNoteData> initialNoteData = Option.when(
                createInitialNote.get() && !isFromOldClient(userAgent), new InitialNoteData(lang, requestId));

        if (notesAttachmentsManager.needInit(uid)) {
            notesAttachmentsManager.userInit(uid, lang);
        }

        Database db = findOrCreateDb(uid, initialNoteData);

        return new ResultWithRevision<>(db.rev);
    }

    public ResultWithRevision<Void> init(DataApiUserId uid) {
        return new ResultWithRevision<>(findOrCreateDb(uid).rev);
    }

    public Database initInternal(DataApiUserId uid, Option<InitialNoteData> initialNoteData) {
        Database db = dataApiManager.getOrCreateDatabase(dbSpec(uid));
        if (db.isNew || db.rev == 0) {
            try {
                db = dataApiManager.applyDelta(db.spec(), db.rev, RevisionCheckMode.WHOLE_DATABASE, initTags());
            } catch (OutdatedChangeException ignore) {
                return dataApiManager.getOrCreateDatabase(dbSpec(uid));
            }

            if (initialNoteData.isPresent()) {
                try {
                    createInitialNote(uid,
                            initialNoteData.get().lang, initialNoteData.get().requestId, Option.of(db.rev));
                    return dataApiManager.getOrCreateDatabase(dbSpec(uid));
                } catch (Exception e) {
                    ExceptionUtils.throwIfUnrecoverable(e);
                    logger.error("Failed to create initial not for {}: {}", uid, e);
                }
            }

            return db;
        }
        //already inited, maybe need check
        return db;
    }

    @Override
    public Database findOrCreateDb(DataApiUserId uid) {
        return findOrCreateDb(uid, Option.empty());
    }

    private Database findOrCreateDb(DataApiUserId uid, Option<InitialNoteData> initialNoteData) {
        return findDb(uid).getOrElse(() -> initInternal(uid, initialNoteData));
    }

    @Override
    public ListF<Note> listNotes(DataApiUserId uid) {
        return withExistingDbOrElse(uid, this::findAllNotes, Cf::list);
    }

    @Override
    public ResultWithRevision<NoteWithMdsKey> findNote(DataApiUserId uid, String id, boolean getMdsKey) {
        Database db = findDb(uid).getOrThrow(() -> new NotFoundException("no notes for user"));

        DataRecord record = recordById(db, id).getOrThrow(() -> new NotFoundException("note doesn't exists"));

        Note note = parseFromDs(record);

        Option<MdsFileKey> mdsKey = note.contentRevision.filterMap(contentRevision -> getMdsKey
                ? revisionsDao.findRevision(uid, id, contentRevision)
                : Option.empty()
        ).map(revisionRecord -> revisionRecord.snapshotMdsKey);

        return new ResultWithRevision<>(new NoteWithMdsKey(note, mdsKey), record.rev);
    }

    private Note findNote(DataApiUserId uid, String id) {
        return findNote(uid, id, false).result.getNote();
    }

    @Override
    public ResultWithRevision<Note> createNote(DataApiUserId uid, NoteCreation base, Option<String> requestId) {
        return createNote(uid, Option.empty(), base, requestId, Option.empty());
    }

    public ResultWithRevision<Note> createNote(
            DataApiUserId uid, Option<String> noteIdO, NoteCreation base, Option<String> requestId,
            Option<Long> expectedDbRevision)
    {
        Database db = findOrCreateDb(uid);
        Option<Note> actualNote = resolveNoteByRequest(uid, requestId, db);
        if (actualNote.isPresent()) {
            return new ResultWithRevision<>(actualNote.get(), db.rev).wasCreated();
        } else {
            long lastRevision = 0;
            Note note = base.toNote(noteIdO.getOrElse(createNoteId(uid, db)));
            notesDao.create(NoteRecord.builder()
                    .id(note.id).uid(uid).mtime(note.mtime).lastRevision(lastRevision).build());
            long rev;
            if (expectedDbRevision.isPresent()) {
                rev = dataApiManager.applyDelta(
                        db.spec(), expectedDbRevision.get(), RevisionCheckMode.WHOLE_DATABASE, createDelta(note)).rev;
            } else {
                rev = dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, createDelta(note)).rev;
            }
            storeRequestId(uid, requestId, note.id, lastRevision);
            return new ResultWithRevision<>(note, rev).wasCreated();
        }
    }

    private String createNoteId(DataApiUserId uid, Database db) {
        return uid.toString() + "_" + db.rev + "_" + Random2.R.nextString(3, "0123456789");
    }

    @Override
    public ResultWithRevision<Void> deleteNote(DataApiUserId uid, String id, HttpServletRequestX req) {
        Option<Long> clientRevision = RequestUtils.getIfMatchAsLong(req);
        if (clientRevision.isPresent()) {
            long lastRevision = notesDao.findNote(uid, id).map(NoteRecord::getLastRevision).getOrElse(0L);
            if (clientRevision.get() != lastRevision) {
                return new ResultWithRevision<>(findOrCreateDb(uid).rev);
            }
        }

        return deleteNote(uid, id);
    }

    private ResultWithRevision<Void> deleteNote(DataApiUserId uid, String id) {
        return withRetriesOnRevConflict(() -> withExistingDbOrThrow(uid, db -> {
            Option<DataRecord> records = recordById(db, id);

            if (!records.isPresent()) {
                return new ResultWithRevision<Void>(db.rev);
            }

            Note actualNote = parseFromDs(records.get());

            if (containsTag(actualNote.tagsWithMeta, TagType.DELETED.getPredefinedId())) {
                return new ResultWithRevision<Void>(db.rev);
            }

            FieldChange fieldChange = FieldChange.insertListItem("tags",
                    actualNote.tagsWithMeta.size(),
                    toDataRecord(NoteTag.builder().id(TagType.DELETED.getPredefinedId())
                            .mtime(Option.of(Instant.now())).build()));


            long finalRev = dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                    new Delta(RecordChange.update(records.get().id, Cf.list(fieldChange)))
            ).rev;
            notesDao.softDelete(uid, id);

            scheduleNotesRemoval(uid, Instant.now());

            return new ResultWithRevision<Void>(finalRev);
        }));
    }

    @Override
    public ResultWithRevision<?> patchNote(
            DataApiUserId uid, String id, NotePatch patch, Option<String> requestId, boolean forceMtimeUpdate)
    {
        return withRetriesOnRevConflict(() -> withExistingDbOrThrow(uid, db -> {

            Option<DataRecord> recordO = recordById(db, id);
            if (!recordO.isPresent()) {
                throw new NotFoundException("note doesn't exists: " + id);
            }

            Option<RequestRecord> requestRecordO = checkExists(uid, requestId);
            if (requestRecordO.isPresent()) {
                return new ResultWithRevision<Void>(db.rev);
            }

            Note actualNote = parseFromDs(recordO.get());

            ListF<NoteTag> addTags = patch.getAllAddTagsMeta().filter(t -> !containsTag(actualNote.tagsWithMeta, t.id));

            //todo: move to better place
            ListF<FieldChange> fieldChanges = Cf.arrayList();

            fieldChanges.addAll(patch.title.map(title -> FieldChange.put("title", DataField.string(title))));
            fieldChanges.addAll(patch.snippet.map(snippet -> FieldChange.put("snippet", DataField.string(snippet))));
            fieldChanges.addAll(patch.contentRevision.map(
                    rev -> FieldChange.put("content_revision", DataField.integer(rev))));

            if (forceMtimeUpdate || fieldChanges.isNotEmpty() ||
                    patch.removeTags.filterNot(t -> t == TagType.PIN.getPredefinedId()).isNotEmpty() ||
                    addTags.filterNot(t -> t.id == TagType.PIN.getPredefinedId()).isNotEmpty())
            {
                Instant mtime = patch.mtime.getOrElse(Instant.now());
                fieldChanges.add(FieldChange.put("mtime", DataField.timestamp(mtime)));
                notesDao.updateMtime(uid, id, mtime);
            }

            fieldChanges.addAll(addTags
                    .zipWithIndex()
                    .map((tag, idx) -> FieldChange.insertListItem("tags",
                            actualNote.tags.size() + idx,
                            toDataRecord(tag))
                    )
            );

            ListF<NoteTag> tagsMutable = Cf.toArrayList(actualNote.tagsWithMeta.plus(addTags));

            fieldChanges.addAll(patch.removeTags
                    .filter(t -> containsTag(tagsMutable, t))
                    .map(tag -> {
                        int index = tagsMutable.indexOfTs(getTag(tagsMutable, tag).get());
                        tagsMutable.remove(index);

                        return FieldChange.deleteListItem("tags", index);
                    })
            );

            long finalRev = dataApiManager.applyDelta(db,
                    RevisionCheckMode.PER_RECORD,
                    new Delta(RecordChange.update(recordO.get().id, fieldChanges))).rev;

            requestId.ifPresent(r -> requestsDao.create(RequestRecord.builder()
                    .requestId(r).uid(uid).entityId(id).revision(finalRev).build()));

            return new ResultWithRevision<>(findNote(uid, id), finalRev);

        }));
    }

    @Override
    public void updateMtimeAndAttachResourceIds(
            DataApiUserId uid, String noteId, Option<Instant> attachMtime, ListF<String> resourceIds)
    {
        updateMtimeAndAttachResourceIdsInner(uid, noteId, attachMtime, Option.of(resourceIds));
    }

    @Override
    public void updateMtime(DataApiUserId uid, String noteId, Instant mtime) {
        updateMtimeAndAttachResourceIdsInner(uid, noteId, Option.of(mtime), Option.empty());
    }

    private void updateMtimeAndAttachResourceIdsInner(
            DataApiUserId uid, String noteId, Option<Instant> attachMtime, Option<ListF<String>> resourceIdsO)
    {
        withRetriesOnRevConflict(() -> withExistingDbOrThrow(uid, db -> {
            Option<DataRecord> recordO = recordById(db, noteId);
            if (!recordO.isPresent()) {
                throw new NotFoundException("note doesn't exists: " + noteId);
            }

            Option<Instant> currentMTime = recordO.get().getData().getO("mtime").map(DataField::timestampValue);
            Option<Instant> newMTime = currentMTime.plus(attachMtime).maxO();

            ListF<FieldChange> fieldChanges = Cf.arrayList();
            fieldChanges.addAll(resourceIdsO.map(resourceIds ->
                    FieldChange.put("attach_resource_ids", DataField.list(resourceIds.map(DataField::string)))));
            fieldChanges.addAll(newMTime.map(mtime -> FieldChange.put("mtime", DataField.timestamp(mtime))));
            newMTime.ifPresent(mtime -> notesDao.updateMtime(uid, noteId, mtime));

            dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                    new Delta(RecordChange.update(recordO.get().id, fieldChanges)));

            return null;
        }));
    }

    @Override
    public void getContent(DataApiUserId uid, String id, HttpServletResponseX responseX, Option<Long> clientRevision) {
        Option<NoteRecord> note = notesDao.findNote(uid, id);
        NoteRecord noteRecord = note.getOrThrow(() -> new NotFoundException("note doesn't exists"));
        setRevision(responseX, noteRecord.lastRevision);
        if (clientRevision.isPresent() && clientRevision.isMatch(aLong -> noteRecord.lastRevision == aLong)) {
            responseX.setStatus(HttpStatus.SC_304_NOT_MODIFIED);
            return;
        }
        Option<RevisionRecord> revision = revisionsDao.findRevision(noteRecord.uid, noteRecord.id, noteRecord.lastRevision);
        if (!revision.isPresent()) {
            responseX.writeContent(EMPTY_CONTENT, ContentType.APPLICATION_JSON.getMimeType());
            return;
        }
        RevisionRecord revisionRecord = revision.get();
        setRevision(responseX, revisionRecord.revision);
        try {
            contentManager.get(revisionRecord.snapshotMdsKey, responseX.getOutputStream());
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Override
    public void putContent(DataApiUserId uid, String id, Option<String> requestId,
            Instant clientTime, boolean extendedContent, HttpServletRequestX req, HttpServletResponseX resp)
    {
        Option<Long> clientRevision = RequestUtils.getIfMatchAsLong(req);
        NoteRecord note = notesDao.findNote(uid, id).getOrElse(
                () -> notesDao.create(NoteRecord.builder().id(id).uid(uid).mtime(Instant.now()).lastRevision(0).build()));
        //Collision resolving?
        if (clientRevision.isPresent() && !clientRevision.isMatch(aLong -> note.lastRevision == aLong)) {
            setRevision(resp, note.lastRevision);
            Option<RequestRecord> requestRecordO = checkExists(uid, requestId);
            if (requestRecordO.isPresent()) {
                return;
            }

            throw new A3ExceptionWithStatus("revision-conflict", "Expected revision:"
                    + note.lastRevision, HttpStatus.SC_409_CONFLICT);
        }
        long newRevision = note.lastRevision + 1;

        ExtendedContent content = new ExtendedContent(req.getInputStreamX(), extendedContent);
        MdsFileKey put = contentManager.put(id, uid, newRevision, content.content);
        revisionsDao.create(
                RevisionRecord.builder()
                        .noteId(id)
                        .uid(uid)
                        .snapshotMdsKey(put)
                        .authorUid(uid)
                        .clientTime(clientTime)
                        .revision(newRevision)
                        .build());
        notesDao.updateRevision(uid, id, newRevision);

        NotePatch patch = new NotePatch(content.title, content.snippet, content.mtime.plus(clientTime).firstO(),
                content.addTags, content.addTagsWithMeta, content.removeTags, Option.of(newRevision));
        patchNote(uid, id, patch, requestId, true);

        requestId.ifPresent(r -> requestsDao.create(RequestRecord.builder().requestId(r).uid(uid)
                .entityId(note.id).revision(newRevision).build()));
        setRevision(resp, newRevision);
    }

    @Override
    public ResultWithRevision<Note> createNoteWithContent(DataApiUserId uid, Option<String> sourceIdO,
            Option<String> requestId, Instant clientTime, HttpServletRequestX req)
    {
        Option<RequestRecord> request = checkExists(uid, requestId);
        Database db = findOrCreateDb(uid);
        if (request.isPresent()) {
            Option<DataRecord> records = recordById(db, request.get().entityId);
            Note actualNote = parseFromDs(records.get());
            return new ResultWithRevision<>(actualNote, db.rev).wasCreated();
        }

        Option<Note> sourceNote = sourceIdO.map(sourceId -> findNote(uid, sourceId));
        ExtendedContent extendedContent = new ExtendedContent(req.getInputStreamX(), true);

        // generate future note id
        String noteId = createNoteId(uid, db);

        long initialRevision = 0;

        ListF<Long> excludedTagsIds = Cf.list(TagType.TRASH, TagType.DELETED).map(TagType::getPredefinedId);
        ListF<NoteTag> sourceNoteTags = sourceNote.map(Note::getTagsWithMeta).getOrElse(Cf.list())
                .filterNot(tag -> excludedTagsIds.containsTs(tag.id));
        ListF<NoteTag> noteTags = extendedContent.tagsWithMeta.getOrElse(sourceNoteTags);

        ListF<String> resourceIds;

        try {
            // copy attachments
            if (sourceIdO.isPresent()) {
                notesAttachmentsManager.copyAttachments(
                        uid, sourceIdO.get(), noteId, initialRevision, extendedContent.attachResourceIds);
                resourceIds = notesAttachmentsManager.getAttachmentsInfos(uid, noteId, Option.of(initialRevision))
                        .map(MpfsFileInfo::getMeta)
                        .filterMap(MpfsFileMeta::getResourceId).map(MpfsResourceId::serialize);
            } else {
                resourceIds = Cf.list();
            }

            // create note with pregenerated id
            NoteCreation base = new NoteCreation(
                    extendedContent.title.getOrElse(sourceNote.map(Note::getTitle).getOrElse("")),
                    extendedContent.ctime.plus(clientTime).firstO(),
                    extendedContent.mtime.plus(clientTime).firstO(),
                    noteTags,
                    Cf.list(),
                    extendedContent.snippet.plus(sourceNote.filterMap(Note::getSnippet)).firstO(),
                    Option.of(resourceIds),
                    Option.of(initialRevision));

            executeAsync(() -> synchronizeAttachmentsLater(uid, noteId, attachmentsSyncDelay));

            return createNoteWithContentInner(
                    uid, noteId, base, extendedContent.content, requestId, initialRevision, Option.empty());

        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            if (sourceIdO.isPresent()) {
                logger.error("Failed to clone note {} into {}: {}. Trying to remove attachments",
                        sourceIdO.get(), noteId, e);
                executeAsync(() -> executeSafe(() -> notesAttachmentsManager.removeAttachmentsFromMpfs(uid, noteId)));
            } else {
                logger.error("Failed to create note with content {}", e);
            }
            throw e;
        }
    }

    private void executeSafe(Runnable action) {
        try {
            action.run();
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            logger.error(t);
        }
    }

    @Override
    public ResultWithRevision<Attachment> getAttachment(DataApiUserId uid, String noteId, String resourceId) {
        return notesAttachmentsManager.getAttachment(uid, noteId, resourceId);
    }

    @Override
    public ResultWithRevision<Attachments> getAttachments(AttachmentQuery query) {
        ResultWithRevision<Attachments> result = notesAttachmentsManager.getAttachments(query);
        if (result.result.offset.equals(0) && (result.result.items.size() < result.result.limit)) {
            executeAsync(() -> synchronizeAttachmentsIfDiffers(
                    query.getUid(), query.getNoteId(), Cf.x(result.result.items).map(a -> a.resourceId)));
        }
        return result;
    }

    @Override
    public ResultWithRevision<AttachmentCreationResult> addAttachments(AttachmentAddition addition, Instant mtime) {
        ResultWithRevision<AttachmentCreationResult> result = notesAttachmentsManager.addAttachments(addition, mtime);
        executeAsync(() -> synchronizeAttachmentsLater(addition.getUid(), addition.getNoteId(), attachmentsSyncDelay));
        return result;
    }

    @Override
    public ResultWithRevision<Void> deleteAttachment(
            DataApiUserId uid, String noteId, String resourceId, Option<Long> revision, Instant mtime,
            Option<String> userAgent)
    {
        if (isFromOldClient(userAgent)) {
            logger.info("Attachment deletion skipped due to old client version");
            long rev = findDb(uid).getOrThrow(() -> new NotFoundException("no notes for user")).rev;
            return new ResultWithRevision<>(rev);
        }

        ResultWithRevision<Void> result =
                notesAttachmentsManager.deleteAttachment(uid, noteId, resourceId, revision, mtime);
        executeAsync(() -> synchronizeAttachmentsWithBazingaFallback(uid, noteId, Option.of(mtime)));
        return result;
    }

    @Override
    public void scheduleNotesRemoval(DataApiUserId uid, Instant noteDeletionTime) {
        bazingaTaskManager.schedule(new RemoveDeletedNotesTask(uid), noteDeletionTime.plus(getRemoveDelay()));
    }

    @Override
    public void removeDeletedNotes(DataApiUserId uid, boolean withReschedule) {
        removeDeletedNotes(uid, Instant.now().minus(getRemoveDelay()), withReschedule);
    }

    public void removeDeletedNotes(DataApiUserId uid, Instant deadline, boolean withReschedule) {
        Tuple2List<Note, NoteTag> deletedNotes =
                listNotes(uid).zipWith(this::getDeletedTag).filterBy2(Option::isPresent).map2(Option::get);

        deletedNotes.filterBy2Not(tag -> tag.getMtime().exists(time -> time.isAfter(deadline)))
                .get1().forEach(note -> removeNoteSafe(uid, note));

        if (withReschedule) {
            deletedNotes.get2().filterMap(NoteTag::getMtime).filter(time -> time.isAfter(deadline)).minO()
                    .ifPresent(time -> scheduleNotesRemoval(uid, time));
        }
    }

    private Option<NoteTag> getDeletedTag(Note note) {
        return note.getTagsWithMeta()
                .filter(tag -> tag.getId() == TagType.DELETED.getPredefinedId())
                .firstO();
    }

    private void removeNoteSafe(DataApiUserId uid, Note note) {
        try {
            removeNote(uid, note);
        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            logger.error("Failed to remove note. uid {} note_id {}. Reason:", uid, note.id, e);
        }
    }

    private void removeNote(DataApiUserId uid, Note note) {
        Check.isTrue(note.getTags().containsTs(TagType.DELETED.getPredefinedId()));
        String noteId = note.getId();
        logger.info("removing note {}", noteId);
        if (note.getAttachResourceIds().isNotEmpty()) {
            notesAttachmentsManager.removeAttachmentsFromMpfs(uid, noteId);
        }

        ListF<RevisionRecord> revisions = revisionsDao.findRevisions(uid, noteId);
        revisions.forEach(revision -> {
            MdsFileKey key = revision.getSnapshotMdsKey();
            try {
                contentManager.deleteData(key);
            } catch (MdsFileNotFoundException ignore) {
            }
            revisionsDao.delete(uid, noteId, revision.getRevision());
        });

        notesDao.delete(uid, noteId);

        Database db = findDb(uid).get();

        RecordId recordId = new SimpleRecordId(NOTES_COLLECTION, noteId);

        dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, new Delta(RecordChange.delete(recordId)));
    }

    private Duration getRemoveDelay() {
        return Duration.standardDays(removeNotesDelay.get());
    }

    private void executeAsync(Runnable action) {
        syncAttachmentsResourcesService.submit(action);
    }

    private void synchronizeAttachmentsIfDiffers(DataApiUserId uid, String noteId, ListF<String> actualResourceIds) {
        ListF<String> storedResourceIds = findNote(uid, noteId).getAttachResourceIds();
        if (!storedResourceIds.unique().equals(actualResourceIds.unique())) {
            synchronizeAttachmentsWithBazingaFallback(uid, noteId, Option.empty());
        }
    }

    private void synchronizeAttachmentsLater(DataApiUserId uid, String noteId, Duration delay) {
        try {
            bazingaTaskManager.schedule(new SyncAttachResourceIdsTask(uid, noteId), Instant.now().plus(delay));
        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            logger.warn("Failed to schedule attachments synchronization: {}", e);
        }
    }

    private void synchronizeAttachmentsWithBazingaFallback(
            DataApiUserId uid, String noteId, Option<Instant> actionMtime)
    {
        try {
            synchronizeAttachments(uid, noteId, actionMtime);
        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            logger.warn("Failed to synchronize attachments, will try later. Error message {}", e);
            synchronizeAttachmentsLater(uid, noteId, attachmentsSyncDelay);
            actionMtime.ifPresent(mtime -> updateMtime(uid, noteId, mtime));
        }
    }

    @Override
    public void synchronizeAttachments(DataApiUserId uid, String noteId, Option<Instant> actionMtime) {
        ListF<MpfsFileMeta> metas = notesAttachmentsManager.getAttachmentsInfos(uid, noteId, Option.empty()).map(MpfsFileInfo::getMeta);

        ListF<String> resourceIds = metas.filterMap(MpfsFileMeta::getResourceId).map(MpfsResourceId::serialize);

        Option<Instant> mtime = metas.filterMap(MpfsFileMeta::getNoteAttachmentMtime)
                .map(Instant::parse).plus(actionMtime).maxO();

        updateMtimeAndAttachResourceIds(uid, noteId, mtime, resourceIds);
    }

    private void setRevision(HttpServletResponseX resp, long revision) {
        resp.setHeader(AddRevisionInterceptor.X_ACTUAL_REVISION, Long.toString(revision));
        resp.setHeader(CommonHeaders.ACCESS_CONTROL_EXPOSE_HEADERS,
                "Cache-Control, Content-Language, Content-Type, Expires, Last-Modified, Pragma, "
                        + AddRevisionInterceptor.X_ACTUAL_REVISION);
    }

    private Option<DataRecord> recordById(Database db, String id) {
        RecordId dataRecordId = recordId(id);
        return dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT.withRecordRef(dataRecordId)).singleO();
    }

    private Note parseFromDs(DataRecord tagRecord) {
        MapF<String, DataField> data = tagRecord.getData();

        ListF<NoteTag> tags = data.getO("tags").map(DataField::listValue).getOrElse(Cf.list()).map(this::parseTagFromDs);
        ListF<String> resourceIds = data.getO("attach_resource_ids")
                .map(DataField::listValue).getOrElse(Cf.list()).map(DataField::stringValue);
        return Note.builder()
                .id(tagRecord.id.recordId())
                .title(data.getO("title").map(DataField::stringValue).getOrElse(""))
                .ctime(data.getTs("ctime").timestampValue())
                .mtime(data.getTs("mtime").timestampValue())
                .tags(tags.map(noteTag -> noteTag.id))
                .tagsWithMeta(tags)
                .snippet(data.getO("snippet").map(DataField::stringValue))
                .attachResourceIds(resourceIds)
                .contentRevision(data.getO("content_revision").map(DataField::integerValue))
                .build();
    }

    private NoteTag parseTagFromDs(DataField t) {
        if (t.fieldType == DataFieldType.LIST) {
            ListF<DataField> dataFields = t.listValue();
            Validate.isTrue(dataFields.size() == 2, "Size should be 2 for " + t);
            return NoteTag.builder().id(dataFields.get(0).integerValue()).mtime(Option.of(dataFields.get(1).timestampValue()))
                    .build();
        } else {
            return NoteTag.builder().id(t.integerValue()).mtime(Option.empty()).build();
        }
    }

    private DataField toDataRecord(NoteTag t) {
        return DataField.list(DataField.integer(t.id), DataField.timestamp(t.mtime.getOrElse(Instant.now())));
    }

    private DataApiRecord toDataRecord(Note note) {
        Tuple2List<String, DataField> fields = Cf.Tuple2List.fromPairs(
                "title", DataField.string(note.title),
                "mtime", DataField.timestamp(note.mtime),
                "ctime", DataField.timestamp(note.ctime),
                "tags", DataField.list(note.tagsWithMeta.map(this::toDataRecord)),
                "attach_resource_ids", DataField.list(note.attachResourceIds.map(DataField::string)));
        if (note.snippet.isPresent()) {
            fields.add("snippet", DataField.string(note.snippet.get()));
        }
        if (note.contentRevision.isPresent()) {
            fields.add("content_revision", DataField.integer(note.contentRevision.get()));
        }
        return new SimpleDataRecord(recordId(note.id),
                new DataFields(fields.toMap()));
    }

    private Delta createDelta(Note note) {
        return Delta.fromNewRecords(toDataRecord(note));
    }

    private  <T> T withRetriesOnRevConflict(Function0<T> action) {
        for (int i = 0; i <= retiresOnConflictCount; i++) {
            try {
                return action.apply();
            } catch (OutdatedChangeException ignore) {
            }
        }
        throw new A3ExceptionWithStatus(ErrorNames.OUTDATED_CHANGE,
                "Failed to execute action within " + retiresOnConflictCount + " retries due to rev conflicts",
                HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
    }

    public ResultWithRevision<Note> createInitialNote(
            DataApiUserId uid, Language lang, Option<String> requestId, Option<Long> expectedDbRevision)
    {
        Database db = findOrCreateDb(uid);
        Option<Note> actualNote = resolveNoteByRequest(uid, requestId, db);
        if (actualNote.isPresent()) {
            return new ResultWithRevision<>(actualNote.get(), db.rev).wasCreated();
        }

        DataApiUserId sourceUid = new DataApiPublicUserId();

        Note initialNote;
        Language initialNoteLang;

        try {
            initialNote = findNote(sourceUid, getPublicNoteId(lang));
            initialNoteLang = lang;
        } catch (NotFoundException ignore) {
            initialNote = findNote(sourceUid, getPublicNoteId(getDefaultLang()));
            initialNoteLang = getDefaultLang();
        }

        // resolve initial note's content
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        MdsFileKey initialContentKey = notesDao.findNote(sourceUid, initialNote.id)
                .filterMap(note -> revisionsDao.findRevision(note.uid, note.id, note.lastRevision))
                .map(RevisionRecord::getSnapshotMdsKey)
                .getOrThrow("Failed to resolve initial note content. noteId: " + initialNote.id);
        contentManager.get(initialContentKey, outputStream);
        byte[] content = outputStream.toByteArray();

        // generate initial note's id
        String noteId = createNoteId(uid, db);
        long initialRevision = 0;
        Instant now = Instant.now();

        try {
            // copy attachments
            ListF<String> resourceIds =
                    notesAttachmentsManager.copyInitialAttachments(uid, noteId, initialNoteLang, now, initialRevision);

            // create note
            NoteCreation base = new NoteCreation(
                    initialNote.title,
                    Option.of(now),
                    Option.of(now),
                    initialNote.tagsWithMeta,
                    Cf.list(),
                    initialNote.snippet,
                    Option.of(resourceIds),
                    Option.of(initialRevision));

            executeAsync(() -> synchronizeAttachmentsLater(uid, noteId, attachmentsSyncDelay));

            return createNoteWithContentInner(
                    uid, noteId, base, content, requestId, initialRevision, expectedDbRevision);

        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            logger.error("Failed to create initial note for {}. Trying to remove attachments", uid);
            executeAsync(() -> executeSafe(() -> notesAttachmentsManager.removeAttachmentsFromMpfs(uid, noteId)));
            throw e;
        }
    }

    private String getPublicNoteId(Language lang) {
        return new DataApiPublicUserId().stringValue() + "_" + lang.value();
    }

    private Option<Note> resolveNoteByRequest(DataApiUserId uid, Option<String> requestId, Database db) {
        return checkExists(uid, requestId).map(request -> {
            Option<DataRecord> records = recordById(db, request.entityId);
            return parseFromDs(records.get());
        });
    }

    private void storeRequestId(DataApiUserId uid, Option<String> requestId, String noteId, long revision) {
        requestId.ifPresent(r -> requestsDao.create(RequestRecord.builder().requestId(r).uid(uid)
                .entityId(noteId).revision(revision).build()));
    }

    private ResultWithRevision<Note> createNoteWithContentInner(
            DataApiUserId uid, String noteId, NoteCreation base, byte[] content, Option<String> requestId, long rev,
            Option<Long> expectedDbRev)
    {
        MdsFileKey contentKey = null;
        try {
            // store content
            contentKey = contentManager.put(noteId, uid, rev, content);

            // create note
            ResultWithRevision<Note> creationResult = createNote(
                    uid, Option.of(noteId), base, Option.empty(), expectedDbRev);

            // store content revision
            revisionsDao.create(
                    RevisionRecord.builder()
                            .noteId(noteId)
                            .uid(uid)
                            .snapshotMdsKey(contentKey)
                            .authorUid(uid)
                            .clientTime(Instant.now())
                            .revision(rev)
                            .build());

            storeRequestId(uid, requestId, noteId, creationResult.rev);

            return creationResult;

        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            MdsFileKey keyForRollback =
                    contentKey == null ? null : new MdsFileKey(contentKey.getGroup(), contentKey.getFilename());
            logger.error("Failed to create note with content for {}. Trying to remove content", uid);
            executeAsync(() -> {
                Option.ofNullable(keyForRollback).ifPresent(key -> executeSafe(() -> contentManager.deleteData(key)));
                executeSafe(() -> deleteNote(uid, noteId));
            });
            throw e;
        }
    }

    @Override
    public ResultWithRevision<Note> createPublicNote(Language lang, String title, String snippet, String content) {
        Instant now = Instant.now();
        long rev = 0;
        NoteCreation base = new NoteCreation(title, Option.of(now), Option.of(now),
                Cf.list(), Cf.list(), Option.of(snippet), Option.empty(), Option.of(rev));
        return createNoteWithContentInner(
                new DataApiPublicUserId(), getPublicNoteId(lang), base, content.getBytes(StandardCharsets.UTF_8), Option.empty(), rev, Option.empty());
    }

    @Override
    public ResultWithRevision<Note> createInitialNote(DataApiUserId uid) {
        return createInitialNote(uid, resolveUserLanguage(uid), Option.empty(), Option.empty());
    }

    private Language resolveUserLanguage(DataApiUserId uid) {
        try {
            return blackbox2.query()
                    .userInfo(LocalhostUtils.localAddress(), uid.toPassportUid(),
                            Cf.list(BlackboxDbFields.LANG), Cf.list())
                    .getDbFields().getO(BlackboxDbFields.LANG).map(Language.R::fromValue).get();
        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            logger.error("Failed to resolve language for {}: {}", uid, e);
            return getDefaultLang();
        }
    }

    private Language getDefaultLang() {
        return Language.R.fromValue(defaultInitialNoteLang.get());
    }

    private boolean isFromOldClient(Option<String> userAgent) {
        return userAgent.exists(oldClientPattern::matches);
    }

    public static class ExtendedContent {
        private static final BenderParser<Pojo> parser = new BenderMapper(BenderConfiguration.cons(
                MembersToBind.WITH_ANNOTATIONS, false,
                CustomMarshallerUnmarshallerFactoryBuilder.cons()
                        .add(Instant.class, new ISOInstantUnmarshaller())
                        .build()
        )).createParser(Pojo.class);

        public final Option<String> title;
        public final Option<String> snippet;
        public final Option<Instant> ctime;
        public final Option<Instant> mtime;
        public final Option<ListF<NoteTag>> tagsWithMeta;
        public final SetF<Long> addTags;
        public final SetF<NoteTag> addTagsWithMeta;
        public final SetF<Long> removeTags;
        public final Option<ListF<String>> attachResourceIds;

        public final byte[] content;

        public ExtendedContent(InputStreamX stream, boolean isExtended) {
            if (isExtended) {
                Pojo extendedContentPojo = parser.parseJson(stream.readBytes());
                title = extendedContentPojo.title;
                snippet = extendedContentPojo.snippet;
                ctime = extendedContentPojo.ctime;
                mtime = extendedContentPojo.mtime;
                tagsWithMeta = extendedContentPojo.tagsWithMeta;
                addTags = toSet(extendedContentPojo.addTags);
                addTagsWithMeta = toSet(extendedContentPojo.addTagsWithMeta);
                removeTags = toSet(extendedContentPojo.removeTags);
                attachResourceIds = extendedContentPojo.attachResourceIds;
                content = extendedContentPojo.content.getBytes(StandardCharsets.UTF_8);
            } else {
                title = Option.empty();
                snippet = Option.empty();
                ctime = Option.empty();
                mtime = Option.empty();
                tagsWithMeta = Option.empty();
                addTags = Cf.set();
                addTagsWithMeta = Cf.set();
                removeTags = Cf.set();
                attachResourceIds = Option.empty();
                content = stream.readBytes();
            }
        }

        private <T> SetF<T> toSet(Option<ListF<T>> optionalList) {
            return optionalList.map(CollectionF::unique).getOrElse(Cf.set());
        }

        @BenderBindAllFields
        @AllArgsConstructor
        private static class Pojo {
            public final String content;

            public final Option<String> title;
            public final Option<String> snippet;

            public final Option<Instant> ctime;
            public final Option<Instant> mtime;

            @BenderPart(name = "add_tags", strictName = true, wrapperName = "add_tags")
            public final Option<ListF<Long>> addTags;
            @BenderPart(name = "add_tags_with_meta", strictName = true, wrapperName = "add_tags_with_meta")
            public final Option<ListF<NoteTag>> addTagsWithMeta;
            @BenderPart(name = "remove_tags", strictName = true, wrapperName = "remove_tags")
            public final Option<ListF<Long>> removeTags;

            @BenderPart(name = "tags_with_meta", strictName = true, wrapperName = "tags_with_meta")
            public final Option<ListF<NoteTag>> tagsWithMeta;

            @BenderPart(name = "attach_resource_ids", strictName = true, wrapperName = "attach_resource_ids")
            public final Option<ListF<String>> attachResourceIds;
        }
    }

    @Data
    private static class InitialNoteData {
        public final Language lang;
        public final Option<String> requestId;
    }
}
