package ru.yandex.chemodan.app.notes.core;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.InvalidInputException;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notes.core.model.notes.Note;
import ru.yandex.chemodan.app.notes.core.model.tag.Tag;
import ru.yandex.chemodan.app.notes.core.model.tag.TagBase;
import ru.yandex.chemodan.app.notes.core.model.tag.TagPatch;
import ru.yandex.chemodan.app.notes.core.model.tag.TagType;
import ru.yandex.chemodan.app.notes.dao.NotesRequestsDao;
import ru.yandex.chemodan.app.notes.dao.model.RequestRecord;

/**
 * @author vpronto
 */
public class TagsManagerImpl extends AbstractManager implements TagsManager {

    private static final long TAG_PADDING = 100;

    private final NotesManager notesManager;

    public TagsManagerImpl(DataApiManager dataApiManager, NotesManager notesManager, NotesRequestsDao requestsDao) {
        super(dataApiManager, requestsDao);
        this.notesManager = notesManager;
    }

    private ListF<Tag> findAllTags(Database db) {
        ListF<DataRecord> records = dataApiManager
                .getRecords(UserDatabaseSpec.fromDatabase(db), RecordsFilter.DEFAULT.withCollectionId(TAGS_COLLECTION));

        return records.map(this::parseTagFromDs);
    }

    @Override
    public ListF<Tag> listTags(DataApiUserId uid) {
        return withExistingDbOrElse(uid, this::findAllTags, Cf::list);
    }

    @Override
    public ResultWithRevision<Void> deleteTag(DataApiUserId uid, long id) {
        return withExistingDbOrThrow(uid, db -> {

            if (id < TAG_PADDING) {
                throw new InvalidInputException("can't delete system tag");
            }
            //todo: find notes with such tag in db
            ListF<Note> allNotes = notesManager.listNotes(uid);
            long finalRev = deleteTag(db, id, allNotes);

            return new ResultWithRevision<Void>(finalRev);
        });
    }

    @Override
    public ResultWithRevision<Tag> createTag(DataApiUserId uid, TagBase tagBase) {
        if (tagBase.type.isSystem()) {
            throw new IllegalArgumentException("Invalid tag create request: " + tagBase);
        }

        Database db = notesManager.findOrCreateDb(uid);
        ListF<Tag> allTags = findAllTags(db);
        ListF<Tag> filteredTags = allTags.filter(tag -> tag.type.equals(tagBase.type));
        return filteredTags
                .find(tag -> tag.value.equals(tagBase.value))
                .map(tag -> new ResultWithRevision<>(tag, db.rev))
                .getOrElse(() -> {
                    Tag tag = tagBase.toTag(calcId(db));
                    long finalRev = dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, createDeltaTag(tag)).rev;
                    return new ResultWithRevision<>(tag, finalRev).wasCreated();
                });
    }

    private long calcId(Database db) {
        return db.rev + TAG_PADDING + 1;
    }

    private long deleteTag(Database db, long id, ListF<Note> allNotes) {
            RecordId dataRecordId = recordId(id);
            Option<DataRecord> records = recordById(db, dataRecordId);

            if (!records.isPresent()) {
                //tag is already deleted
                return db.rev;
            }
            ListF<RecordChange> changes = allNotes
                    .filter(note -> containsTag(note.tagsWithMeta, id))
                    .map(note -> RecordChange.update(
                            recordId(note.id),
                            Cf.list(FieldChange.deleteListItem("tags", note.tagsWithMeta.indexOfTs(getTag(note.tagsWithMeta, id).get())))
                            )
                    );

            Delta delta = new Delta(changes).plusDeleted(dataRecordId);

        return dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, delta).rev;
    }

    @Override
    public ResultWithRevision<?> patchTag(DataApiUserId uid, long id, TagPatch tagPatch, Option<String> requestId) {
        //TODO check there is no such tag yet
        return withExistingDbOrThrow(uid, db -> {
            Option<RequestRecord> requestRecords = checkExists(uid, requestId);
            if (requestRecords.isPresent()) {
                return new ResultWithRevision<Void>(db.rev);
            } else {
                return patchTag(uid, db, id, tagPatch, requestId);
            }
        });
    }

    private ResultWithRevision<?> patchTag(DataApiUserId uid, Database db, long id, TagPatch tagPatch,
                                           Option<String> requestId)
    {
        RecordId dataRecordId = recordId(id);
        Option<DataRecord> recordO = recordById(db, dataRecordId);

        if (!recordO.isPresent()) {
            //tag is deleted
            return new ResultWithRevision<Void>(db.rev);
        }
        Tag actualTag = parseTagFromDs(recordO.get());

        if (tagPatch.needPatch(actualTag)) {
            requestId.ifPresent(r -> requestsDao.create(RequestRecord.builder().requestId(r).uid(uid)
                    .entityId(Long.toString(id)).build()));
            return patchTag(actualTag, tagPatch, db, dataRecordId);
        } else {
            return new ResultWithRevision<Void>(db.rev);
        }
    }

    private ResultWithRevision<Tag> patchTag(Tag actualTag, TagPatch tagPatch, Database db, RecordId dataRecordId) {
        Database database = dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                new Delta(tagPatch.toRecordChange(dataRecordId)));
        return new ResultWithRevision<>(actualTag.toBuilder().value(tagPatch.value).build(), database.rev);
    }

    private Option<DataRecord> recordById(Database db, RecordId dataRecordId) {
        return dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT.withRecordRef(dataRecordId)).singleO();
    }

    private Tag parseTagFromDs(DataRecord tagRecord) {
        return Tag.builder()
                .id(Long.parseLong(tagRecord.id.recordId()))
                .type(TagType.R.valueOf(tagRecord.data().get("type").stringValue()))
                .value(tagRecord.data().getO("value").map(DataField::stringValue))
                .build();
    }

}
