package ru.yandex.chemodan.app.notes.dao;

import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notes.dao.model.ModelHelper;
import ru.yandex.chemodan.app.notes.dao.model.NoteRecord;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author vpronto
 */
public class NotesDaoImpl extends BaseDao<NoteRecord> implements NotesDao {

    private final static RowMapper<NoteRecord> M = (RowMapper<NoteRecord>) (rs, rowNum) -> NoteRecord.builder()
            .uid(ModelHelper.toUserId(rs.getString("uid")))
            .id(rs.getString("id"))
            .mtime(new Instant(rs.getDate("mtime")))
            .lastRevision(rs.getLong("last_revision"))
            .build();

    public NotesDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate, M);
    }

    @Override
    public NoteRecord create(NoteRecord record) {
        String q = "INSERT INTO note(uid, id, mtime, last_revision) values (?, ?, ?, ?) ON CONFLICT DO NOTHING";
        update(q, record.uid, record.id, record.mtime, record.lastRevision);
        return record;
    }

    @Override
    public boolean softDelete(DataApiUserId uid, String id) {
        String q = "UPDATE note SET delete_date = now() WHERE uid = ? and id = ?";
        return update(q, uid, id);
    }


    @Override
    public boolean delete(DataApiUserId uid, String id) {
        String q = "DELETE FROM note WHERE uid = ? and id = ?";
        return update(q, uid, id);
    }

    @Override
    public Option<NoteRecord> findNote(DataApiUserId uid, String id) {
        String q = "SELECT uid, id, mtime, last_revision FROM note WHERE uid = ? and id = ? and delete_date is null";
        return queryForOption(q, uid, id);
    }

    @Override
    public ListF<NoteRecord> findNotes(DataApiUserId uid) {
        String q = "SELECT uid, id, mtime, last_revision FROM note WHERE uid = ? and delete_date is null";
        return query(q, uid);
    }

    @Override
    public boolean update(NoteRecord record) {
        String q = "UPDATE note SET mtime = ?, last_revision = ? WHERE uid = ? and id = ?";
        return update(q, record.mtime, record.lastRevision, record.uid, record.id);
    }

    @Override
    public void updateMtime(DataApiUserId uid, String id, Instant mtime) {
        String q = "UPDATE note SET mtime = ? WHERE uid = ? and id = ?";
        update(q, mtime, uid, id);
    }

    @Override
    public void updateRevision(DataApiUserId uid, String id, Long lastRevision) {
        String q = "UPDATE note SET last_revision = ? WHERE uid = ? and id = ?";
        update(q, lastRevision, uid, id);
    }
}
