package ru.yandex.chemodan.app.notes.dao;

import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notes.dao.model.ModelHelper;
import ru.yandex.chemodan.app.notes.dao.model.RevisionRecord;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author vpronto
 */
public class NotesRevisionsDaoImpl extends BaseDao<RevisionRecord> implements NotesRevisionsDao {

    private final static RowMapper<RevisionRecord> M = (RowMapper<RevisionRecord>) (rs, rowNum) -> RevisionRecord.builder()
            .noteId(rs.getString("note_id"))
            .uid(ModelHelper.toUserId(rs.getString("uid")))
            .authorUid(ModelHelper.toUserId(rs.getString("author_uid")))
            .serverTime(new Instant(rs.getDate("c_server_time")))
            .clientTime(new Instant(rs.getDate("c_client_time")))
            .revision(rs.getLong("revision"))
            .delta(rs.getString("delta"))
            .deltaMdsKey(ModelHelper.toMds(rs.getString("delta_mds_key")))
            .originalDeltaMdsKey(ModelHelper.toMds(rs.getString("original_delta_mds_key")))
            .snapshotMdsKey(ModelHelper.toMds(rs.getString("snapshot_mds_key")))
            .build();

    public NotesRevisionsDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate, M);
    }

    @Override
    public RevisionRecord create(RevisionRecord record) {
        String q = "INSERT INTO revision(uid, note_id, revision, c_client_time, author_uid, delta, delta_mds_key"
                + ", original_delta_mds_key, snapshot_mds_key) "
                + "values (?, ?, ?, ?, ?, (to_json(?::json)), ?, ?, ?) ON CONFLICT DO NOTHING";
        update(q, record.uid, record.noteId, record.revision, record.clientTime, record.authorUid,
                record.delta, record.deltaMdsKey, record.originalDeltaMdsKey, record.snapshotMdsKey);
        return record;
    }

    @Override
    public boolean delete(DataApiUserId uid, String noteId, long revision) {
        String q = "DELETE FROM revision WHERE uid = ? and note_id = ? and revision = ?";
        return update(q, uid, noteId, revision);
    }

    @Override
    public Option<RevisionRecord> findRevision(DataApiUserId uid, String noteId, long revision) {
        String q = "SELECT uid, note_id, revision, c_client_time, c_server_time, author_uid, delta, delta_mds_key"
                + ", original_delta_mds_key, snapshot_mds_key "
                + "FROM revision WHERE uid = ? and note_id = ? and revision = ?";
        return queryForOption(q, uid, noteId, revision);
    }

    @Override
    public ListF<RevisionRecord> findRevisions(DataApiUserId uid, String noteId) {
        String q = "SELECT uid, note_id, revision, c_client_time, c_server_time, author_uid, delta, delta_mds_key"
                + ", original_delta_mds_key, snapshot_mds_key "
                + "FROM revision WHERE uid = ? and note_id = ?";
        return query(q, uid, noteId);
    }

    @Override
    public boolean update(RevisionRecord record) {

        String q = "UPDATE revision SET c_client_time = ?, author_uid  = ?, delta  = (to_json(?::json)), delta_mds_key  = ?"
                + ", original_delta_mds_key = ?, snapshot_mds_key = ? "
                + "WHERE uid = ? and note_id = ? and revision = ?";
        return update(q, record.clientTime, record.authorUid, record.delta, record.deltaMdsKey,
                record.originalDeltaMdsKey, record.snapshotMdsKey, record.uid,
                record.noteId, record.revision);

    }

}
