package ru.yandex.chemodan.app.notifier;

import java.io.File;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.ToString;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockMeta;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderInvitationBlockFields;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.cool.utils.CoolTitlesUtils;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TermDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TextProcessor;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.reminder.CoolLentaReminder;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoReminderFields;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoSelectionFields;
import ru.yandex.chemodan.app.lentaloader.reminder.ReminderNotificationType;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.AbstractCoolLentaBlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.SpecialDateTitleConfiguration;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.SpecialDateTitleConfigurationRegistry;
import ru.yandex.chemodan.app.notifier.locale.LocaleManager;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplate;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplateResolver;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplateSelectionReason;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.notification.toggling.NotificationToggleRegistry;
import ru.yandex.chemodan.app.notifier.push.NotificationPushInfo;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.DiskMetadataProcessorContext;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.DiskMetadataProcessorManager;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorContext;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.eventlog.events.StandardMediaType;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockFullDescription;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.chemodan.util.TimeUtils;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author buberman
 */
@AllArgsConstructor
public class DiskNotificationManager {

    private static final Logger logger = LoggerFactory.getLogger(DiskNotificationManager.class);

    private final DiskMetadataProcessorManager diskMetadataProcessorManager;
    private final LocaleManager localeManager;
    private final NotificationPushManager notificationPushManager;
    private final LentaManager lentaManager;
    private final MpfsClient mpfsClient;
    private final NotificationRecordTypeManager notificationRecordTypeManager;
    private final NotificationTemplateResolver notificationTemplateResolver;
    private final ExperimentsManager experimentsManager;
    private final NotificationToggleRegistry notificationToggleRegistry;
    private final ThemeDefinitionRegistry themeDefinitionRegistry;
    private final UserTimezoneHelper userTimezoneHelper;
    private final SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry;

    public final DynamicProperty<Boolean> geoPushesEnabled = new DynamicProperty<>("cool-lenta-geo-in-push", false);

    public final DynamicProperty<Boolean> specialDatesPhotoSelectionPushesEnabled = new DynamicProperty<>("special_dates_photo_selection_pushes_enabled", false);

    public final DynamicProperty<Boolean> specialDatesAutouploadPushesEnabled = new DynamicProperty<>("special_dates_autoupload_pushes_enabled", false);

    private final MapF<SpecialDateBlockTypes, DynamicProperty<Boolean>> SPECIAL_DATE_ENABLING_FLAGS_FOR_BLOCK_TYPES =
            Cf.map(SpecialDateBlockTypes.PHOTO_SELECTION, specialDatesPhotoSelectionPushesEnabled,
                    SpecialDateBlockTypes.AUTOUPLOAD, specialDatesAutouploadPushesEnabled);

    public static final SetF<LentaRecordType> SUPPORTED_LENTA_BLOCKS_FOR_GNC = Cf.set(
            LentaRecordType.SHARED_FOLDER_INVITE,
            LentaRecordType.SHARED_FOLDER);

    public static final SetF<LentaRecordType> SUPPORTED_LENTA_BLOCKS = SUPPORTED_LENTA_BLOCKS_FOR_GNC
            .plus(LentaRecordType.PHOTO_REMIND_BLOCK)
            .plus(LentaRecordType.CONTENT_BLOCK)
            .plus(LentaRecordType.PHOTO_REMIND_HIDDEN_BLOCK)
            .plus(LentaRecordType.PHOTO_SELECTION_BLOCK);

    private static final MapF<ReminderNotificationType, ServiceAndType> PHOTO_REMINDER_NOTIFICATION_TYPES = Cf.map(
            ReminderNotificationType.REMINDER_NOTIFICATION_WORKDAY, DiskNotifications.PHOTO_REMINDER_WORKDAY,
            ReminderNotificationType.REMINDER_NOTIFICATION_WEEKEND, DiskNotifications.PHOTO_REMINDER_WEEKEND,
            ReminderNotificationType.REMINDER_NOTIFICATION_WEEKLY, DiskNotifications.PHOTO_REMINDER_WEEKLY,
            ReminderNotificationType.REMINDER_NOTIFICATION_BEAUTY, DiskNotifications.PHOTO_REMINDER_BEAUTY
    );

    private static final MapF<IntervalType, ServiceAndType> COOL_LENTA_NOTIFICATION_TYPE_BY_INTERVAL = Cf
            .map(IntervalType.ONE_DAY, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_ONE_DAY)
            .plus1(IntervalType.WEEKEND, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEKEND)
            .plus1(IntervalType.WEEK, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK)
            .plus1(IntervalType.MONTH, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_MONTH)
            .plus1(IntervalType.SEASON, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_SEASON)
            .plus1(IntervalType.YEAR, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_YEAR);

    private static final MapF<IntervalType, ServiceAndType> THEMATIC_NOTIFICATION_TYPES_FOR_INTERVALS =
            Cf.map(IntervalType.MONTH, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_THEMATIC_MONTH,
                    IntervalType.SEASON, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_THEMATIC_SEASON,
                    IntervalType.YEAR, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_THEMATIC_YEAR,
                    IntervalType.WHOLE_TIME, DiskNotifications.PHOTO_SELECTION_COOL_LENTA_THEMATIC_WHOLE_TIME);

    private final DynamicProperty<ListF<String>> disabledLentaBlocks =
            DynamicProperty.cons("notifier-disabled-lenta-blocks", Cf.list());

    public void addLentaBlock(DataApiUserId uid, String blockId) {
        addLentaBlock(uid, blockId, Instant.now());
    }

    public void addLentaBlock(DataApiUserId uid, String blockId, Instant now) {
        lentaManager.findBlock(uid, blockId, ActionInfo.internal(ActionSource.notifier())).orElse(() -> {
            logger.info("About to process: uid: {}, lentaBlockO: {}", uid, Option.empty());
            logger.info("Couldn't find lenta block by uid: {}, id: {}", uid, blockId);
            return Option.empty();
        }).ifPresent(lentaBlock -> {
            logger.info("About to process: uid: {}, lentaBlockO: {}", uid, Option.of(lentaBlock));
            //todo: what are supported block types
            if (!isSupportedBlock(lentaBlock.type)) {
                logger.info("Unsupported lenta block type: {}, uid: {}", lentaBlock.type, uid);
                return;
            }

            LentaBlockExData exData = getLentaBlockExtraData(uid, lentaBlock);
            NotificationType notificationType = notificationRecordTypeManager.resolveRecordType(exData.notificationType);

            if (filterLentaBlock(notificationType, exData, uid)) {
                logger.info("Lenta block filtered: {}, uid: {}", exData, uid);
                return;
            }

            NotificationTemplate template = notificationTemplateResolver.consTemplate(notificationType, uid);
            NotificationTemplate mainTemplate = notificationType.consMainTemplate();

            SetF<String> placeholders = localeManager.getAllMessagesPlaceholders(template);

            DiskMetadataProcessorContext context = new DiskMetadataProcessorContext(
                    new MetadataProcessorContext(), exData, placeholders);

            MetadataWrapper metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock, context);

            SetF<String> unresolved = metadata.findTextMissingEntities(placeholders);


            if (unresolved.isNotEmpty() && !mainTemplate.equals(template)) {
                logger.debug("Placeholders {} are unresolved for template {}, will use main template instead",
                        unresolved, template.getName());

                template = mainTemplate.withReason(NotificationTemplateSelectionReason.fallbackUnresolved(mainTemplate, notificationType));

                context = new DiskMetadataProcessorContext(
                        context.getContext(), exData, localeManager.getAllMessagesPlaceholders(template));

                metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock, context);
            }

            Option<Long> count = getCountValueForLentaBlock(template, metadata);

            NotificationPushInfo info = new NotificationPushInfo(template, exData.actor, now, Option.empty(), count);
            if (exData.pushPlatforms.isPresent()) {
                info = info.withPlatformFilter(exData.pushPlatforms.get());
            }

            notificationPushManager.pushNotificationWithDelay(uid, info, metadata, now);
        });
    }

    Option<Long> getCountValueForLentaBlock(NotificationTemplate template, MetadataWrapper metadata) {
        String attributeName;
        boolean mapOneToZero = false;

        if (template.getType().equals(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_N_YEARS_AGO.getType())) {
            attributeName = TextProcessor.ATTRIBUTES_PREFIX + CoolLentaReminder.YEARS_ATTRIBUTE_NAME;
            mapOneToZero = true;
        } else {
            attributeName = MetadataEntityNames.COUNT;
        }

        boolean finalMapOneToZero = mapOneToZero;
        return metadata.meta.getO(attributeName)
                .map(metadataEntity -> metadataEntity.get("text"))
                .flatMapO(Cf.Long::parseSafe)
                .map(number -> (number == 1L && finalMapOneToZero) ? 0L : number);
    }

    private boolean filterLentaBlock(NotificationType type, LentaBlockExData exData, DataApiUserId uid) {
        return !exData.createNotification
                || !notificationToggleRegistry.isNotificationEnabled(type, uid.toString());
    }

    private boolean isSupportedBlock(LentaRecordType lentaBlockType) {
        return SUPPORTED_LENTA_BLOCKS.minus(getDisabledBlocks()).containsTs(lentaBlockType);
    }

    private SetF<LentaRecordType> getDisabledBlocks() {
        return disabledLentaBlocks.get().filterMap(s -> {
            try {
                return Option.of(LentaRecordType.R.valueOf(s));
            } catch (Exception e) {
                logger.warn("Unknown lenta record type: " + s);
                return Option.empty();
            }
        }).unique();
    }

    private LentaBlockExData getLentaBlockExtraData(DataApiUserId uid, LentaBlockRecord lentaBlock) {
        switch (lentaBlock.type) {
            case CONTENT_BLOCK: {
                NotificationActor actor = NotificationActor.YA_DISK;
                ContentBlockMeta meta = ContentBlockMeta.fromBlock(uid, lentaBlock);
                ServiceAndType recordType;

                if (meta.action.isSome(ContentBlockAction.AUTOSAVE)) {
                    recordType = getNotificationTypeForSpecialDate("autoupload_sd_", uid, SpecialDateBlockTypes.AUTOUPLOAD, Option.empty())
                            .getOrElse(DiskNotifications.AUTOUPLOAD);
                    Option<LentaBlockExData> autoSaveBlock = getAutoSaveBlock(recordType, meta.mediaType);
                    if (autoSaveBlock.isPresent()) {
                        return autoSaveBlock.get();
                    }
                } else if (meta.action.isSome(ContentBlockAction.PHOTO_UNLIM)) {
                    recordType = getNotificationTypeForSpecialDate("autoupload_unlim_sd_", uid, SpecialDateBlockTypes.AUTOUPLOAD, Option.empty())
                            .getOrElse(DiskNotifications.UNLIM_AUTOUPLOAD);
                    Option<LentaBlockExData> autoSaveBlock = getAutoSaveBlock(recordType, meta.mediaType);
                    if (autoSaveBlock.isPresent()) {
                        return autoSaveBlock.get();
                    }
                } else {
                    recordType = meta.action.isSome(ContentBlockAction.ADDITION)
                            ? DiskNotifications.SHARED_FOLDER_FILE_CREATE
                            : DiskNotifications.SHARED_FOLDER_FILE_UPDATE;

                    if (uid.toString().equals(meta.modifierUid)) {
                        logger.debug("Not sending shared folder self notifications for " + meta.modifierUid);
                        return LentaBlockExData.createIgnoredData(recordType);
                    }
                    actor = NotificationActor.consFromSerialized(meta.modifierUid);
                }

                Option<MpfsLentaBlockFullDescription> lentaBlockInfo = getLentaBlockInfo(uid, lentaBlock);
                if (!lentaBlockInfo.isPresent() || lentaBlockInfo.get().files.isEmpty()) {
                    logger.debug("Couldn't find changed files in mpfs for block with id: " + lentaBlock.id);
                    return LentaBlockExData.createIgnoredData(recordType);
                }

                Option<PreviewInfo> preview = Option.empty();

                try {
                    preview = getFilePreview(uid, MpfsResourceId.parse(lentaBlockInfo.get().files.first().resourceId));
                } catch (StringIndexOutOfBoundsException e) {
                    //not resource_id
                }

                return LentaBlockExData.createContentData(Option.of((long) lentaBlockInfo.get().filesCount),
                        preview, recordType, actor, lentaBlockInfo);
            }
            case PHOTO_REMIND_BLOCK:
            case PHOTO_REMIND_HIDDEN_BLOCK: {
                ListF<String> resourceIds = PhotoReminderFields.RESOURCE_IDS.get(lentaBlock);

                int yearsAgo = PhotoReminderFields.YEARS_AGO.get(lentaBlock);
                Lazy<Option<Instant>> periodStart = Lazy.withValue(
                        PhotoReminderFields.BEAUTY_PERIOD_START.getO(lentaBlock));

                Lazy<ListF<LocalDate>> dates = Lazy.withSupplier(() ->
                        getFilesEtime(uid, resourceIds).map(ts -> new LocalDate(ts, DateTimeZone.UTC)));

                Lazy<Option<Integer>> month = Lazy.withSupplier(() -> dates.get()
                        .countBy(LocalDate::getMonthOfYear).entries().maxByO(Tuple2::get2).map(Tuple2::get1));

                Lazy<Option<Integer>> year = Lazy.withSupplier(() -> dates.get()
                        .countBy(LocalDate::getYear).entries().maxByO(Tuple2::get2).map(Tuple2::get1));

                PhotoReminderDate date = new PhotoReminderDate(month, year, yearsAgo, periodStart);

                Option<PreviewInfo> preview = getFilePreview(uid, MpfsResourceId.parse(resourceIds.first()));

                ServiceAndType serviceAndType = PhotoReminderFields.NOTIFICATION_TYPE.getO(lentaBlock)
                        .filterMap(PHOTO_REMINDER_NOTIFICATION_TYPES::getO)
                        .getOrElse(DiskNotifications.PHOTO_REMINDER);

                if (preview.isPresent()) {
                    return LentaBlockExData.createPhotoReminderData(preview, serviceAndType, date, NotificationActor.YA_DISK);
                } else {
                    return LentaBlockExData.createIgnoredData(serviceAndType);
                }
            } case PHOTO_SELECTION_BLOCK: {
                String bestResourceId = PhotoSelectionFields.BEST_RESOURCE_ID.get(lentaBlock);
                Option<PreviewInfo> preview = getFilePreview(uid, MpfsResourceId.parse(bestResourceId));

                LentaBlockExData ignoredData = LentaBlockExData.createIgnoredData(DiskNotifications.PHOTO_SELECTION_COOL_LENTA);
                if (!preview.isPresent()) {
                    return ignoredData;
                }

                String subtype = PhotoSelectionFields.SUBTYPE.getO(lentaBlock).getOrElse("");

                if (!subtype.startsWith("cool_")) {
                    logger.error("Unknown subtype {} for cool lenta block", subtype);
                    return LentaBlockExData.createIgnoredData(DiskNotifications.PHOTO_SELECTION_COOL_LENTA);
                }

                Option<DateTimeZone> timezoneIdO = PhotoSelectionFields.USER_TIMEZONE_ID.getO(lentaBlock).map(DateTimeZone::forID);
                if (!timezoneIdO.isPresent()) {
                    return ignoredData;
                }

                ListF<String> platforms = PhotoSelectionFields.ENABLED_PLATFORMS.getO(lentaBlock)
                        .map(str -> Cf.x(str.split(",")))
                        .getOrElse(Cf.list("ios", "android"));

                DateTime intervalStart = new DateTime(PhotoSelectionFields.INTERVAL_START.get(lentaBlock), timezoneIdO.get());
                DateTime intervalEnd = new DateTime(PhotoSelectionFields.INTERVAL_END.get(lentaBlock), timezoneIdO.get());
                Option<DateTime> generationTimeO = PhotoSelectionFields.USER_GENERATION_TIME.getO(lentaBlock)
                        .map(instant -> new DateTime(instant, timezoneIdO.get()));

                CoolLentaReminderDate remindDate = new CoolLentaReminderDate(intervalStart, intervalEnd, generationTimeO);

                Option<String> desktopUrl = PhotoSelectionFields.DESKTOP_URL.getO(lentaBlock);

                Option<ServiceAndType> specialDateServiceAndType = getNotificationTypeForSpecialDate("photo_selection_cool_lenta_",
                        uid, SpecialDateBlockTypes.PHOTO_SELECTION, Option.of(subtype));
                if (specialDateServiceAndType.isPresent()) {
                    return LentaBlockExData.coolLentaRemindData(preview, specialDateServiceAndType.get(), platforms, remindDate, Cf.list(),
                            Option.empty(), desktopUrl);
                }
                if (subtype.startsWith(CoolLentaReminder.N_YEARS_AGO_SUBTYPE_PREFIX)) {
                    return LentaBlockExData.coolLentaRemindData(preview,
                            DiskNotifications.PHOTO_SELECTION_COOL_LENTA_N_YEARS_AGO, platforms, remindDate, Cf.list(),
                            Option.empty(), desktopUrl);
                }
                if (blockIsThematic(subtype)) {
                    ServiceAndType notificationType = THEMATIC_NOTIFICATION_TYPES_FOR_INTERVALS
                            .getOrThrow(remindDate.getIntervalType(),
                                    String.format("No notification type for type `%s`", remindDate.getIntervalType()));
                    String themeId = getThemeIdFromBlockSubtype(subtype);
                    TermDefinition themeTerm = themeDefinitionRegistry.getO(themeId)
                            .getOrThrow(() ->
                                    new IllegalStateException(String.format("Theme with id `%s` has not been found",
                                            themeId)))
                            .getForms();
                    return LentaBlockExData.coolLentaRemindData(preview, notificationType, platforms, remindDate, Cf.list(),
                            Option.of(themeTerm), desktopUrl);
                }
                if (subtype.startsWith("cool_")) {
                    ListF<Integer> regionIds = PhotoSelectionFields.GEO_REGION_IDS.getO(lentaBlock).getOrElse(Cf::list);

                    Option<ServiceAndType> notificationTypeO = Option.empty();
                    if (remindDate.intervalType == IntervalType.WEEKEND) {
                        if (CoolTitlesUtils.isCrossYear(remindDate.intervalStart, remindDate.intervalType)) {
                            notificationTypeO = Option.of(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEKEND_CROSS_YEAR);
                        } else if (CoolTitlesUtils.isCrossMonth(remindDate.intervalStart, remindDate.intervalType)) {
                            notificationTypeO = Option.of(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEKEND_CROSS_MONTH);
                        }
                    } else if (remindDate.intervalType == IntervalType.SEASON) {
                        if(CoolTitlesUtils.isCrossYear(remindDate.intervalStart, remindDate.intervalType)) {
                            notificationTypeO = Option.of(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WINTER);
                        }
                    } if (remindDate.intervalType == IntervalType.WEEK) {
                        if (CoolTitlesUtils.isCrossYear(remindDate.intervalStart, remindDate.intervalType)) {
                            notificationTypeO = Option.of(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK_CROSS_YEAR);
                        } else if (CoolTitlesUtils.isCrossMonth(remindDate.intervalStart, remindDate.intervalType)) {
                            notificationTypeO = Option.of(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK_CROSS_MONTH);
                        }
                    }

                    notificationTypeO = notificationTypeO.orElse(COOL_LENTA_NOTIFICATION_TYPE_BY_INTERVAL.getO(remindDate.intervalType));
                    if (!notificationTypeO.isPresent()) {
                        return ignoredData;
                    }

                    ServiceAndType notificationType = notificationTypeO.get();

                    if (geoPushesEnabled.get() && regionIds.isNotEmpty()) {
                        int geoLength = diskMetadataProcessorManager.getGeoName(regionIds).length();

                        String geoSuffix = "_geo_small";

                        if (geoLength >= 10 && geoLength <= 14) {
                            geoSuffix = "_geo_medium";
                        } else if (geoLength > 14) {
                            geoSuffix = "_geo_long";
                        }

                        notificationType = new ServiceAndType(notificationType.getService(), notificationType.getType() + geoSuffix);
                    }

                    return LentaBlockExData.coolLentaRemindData(preview, notificationType, platforms, remindDate, regionIds,
                            Option.empty(), desktopUrl);
                }

                return LentaBlockExData.createIgnoredData(DiskNotifications.PHOTO_SELECTION_COOL_LENTA);
            } case SHARED_FOLDER_INVITE: {
                String groupId = SharedFolderInvitationBlockFields.GROUP_ID.get(lentaBlock);
                String inviteHash = SharedFolderInvitationBlockFields.INVITE_HASH.get(lentaBlock);
                String ownerUid = SharedFolderInvitationBlockFields.OWNER_UID.get(lentaBlock);

                if (StringUtils.isEmpty(groupId) || StringUtils.isEmpty(inviteHash) || StringUtils.isEmpty(ownerUid)) {
                    logger.debug("Lenta block doesn't have all required fields for shared folder invite: " + lentaBlock.id);
                    return LentaBlockExData.createIgnoredData(DiskNotifications.SHARED_FOLDER_INVITE_RECEIVED);
                }

                return LentaBlockExData.createSharedFolderData(DiskNotifications.SHARED_FOLDER_INVITE_RECEIVED,
                        NotificationActor.consFromUid(DataApiUserId.parse(ownerUid)),
                        createSharedFolderInviteBlockDescription(DataApiUserId.parse(ownerUid), groupId));
            }
            case SHARED_FOLDER: {
                String groupId = SharedFolderBlockFields.GROUP_ID.get(lentaBlock);
                String ownerUid = SharedFolderBlockFields.OWNER_UID.get(lentaBlock);

                if (StringUtils.isEmpty(groupId) || StringUtils.isEmpty(ownerUid)) {
                    logger.debug("Lenta block doesn't have all required fields for shared folder: " + lentaBlock.id);
                    return LentaBlockExData.createIgnoredData(DiskNotifications.SHARED_FOLDER_INVITE_ACCEPTED);
                }

                return LentaBlockExData.createSharedFolderData(DiskNotifications.SHARED_FOLDER_INVITE_ACCEPTED, NotificationActor.YA_DISK,
                        createSharedFolderInviteBlockDescription(DataApiUserId.parse(ownerUid), groupId));
            }
            default:
                throw new IllegalArgumentException("Unsupported lenta block type: " + lentaBlock.type);
        }
    }

    private Option<ServiceAndType> getNotificationTypeForSpecialDate(String prefixForType, DataApiUserId userId,
            SpecialDateBlockTypes specialDateBlockType, Option<String> subtypeO) {
        boolean isUserInOverride = experimentsManager.getFlags(userId.toPassportUid().toUidOrZero().getUid())
                .containsTs(AbstractCoolLentaBlockTitlesGenerator.SPECIAL_DATE_EXPERIMENT_FLAG);
        boolean isSpecialDateFeatureEnabledForBlock = SPECIAL_DATE_ENABLING_FLAGS_FOR_BLOCK_TYPES.getO(specialDateBlockType)
                .map(DynamicProperty::get).getOrElse(Boolean.FALSE);
        if (!isSpecialDateFeatureEnabledForBlock && !isUserInOverride) {
            logger.debug("The special dates feature is not enabled for uid={} specialDateFeatureEnabled={} userInOverride={} blockType={}",
                    userId, isSpecialDateFeatureEnabledForBlock, isUserInOverride, specialDateBlockType);
            return Option.empty();
        }
        DateTime userTimeNow = DateTime.now(userTimezoneHelper.getUserTimezone(userId.toPassportUid()));
        Option<SpecialDateTitleConfiguration> configurationO = specialDateTitleConfigurationRegistry
                .getSpecialDateConfigurationForDateO(new SpecialDateTitleConfiguration.DayWithMonth(userTimeNow.getMonthOfYear(), userTimeNow.getDayOfMonth()));
        if (!configurationO.isPresent()) {
            return Option.empty();
        }
        String specialDateName = configurationO.get().getSpecialDateName();
        ListF<String> notificationKeys = subtypeO.map(subtype -> getSpecialDateNotificationTypePostfixForSubtype(subtype)
                .map(postfix -> prefixForType + specialDateName + "_" + postfix))
                .getOrElse(Cf.list())
                .plus(prefixForType + specialDateName);
        for (String notificationKey : notificationKeys) {
            ServiceAndType serviceAndType = DiskNotifications.cons(notificationKey);
            Option<NotificationType> notificationTypeO = notificationRecordTypeManager.resolveRecordTypeO(serviceAndType);
            if (notificationTypeO.isPresent()) {
                return Option.of(serviceAndType);
            }
        }
        return Option.empty();
    }

    private boolean blockIsThematic(String subtype) {
        return subtype.startsWith(CoolLentaReminder.THEMATIC_SUBTYPE_PREFIX);
    }

    private ListF<String> getSpecialDateNotificationTypePostfixForSubtype(String subtype) {
        if (!blockIsThematic(subtype)) {
            return Cf.list();
        }
        String themeId = getThemeIdFromBlockSubtype(subtype);
        return Cf.list("thematic_" + themeId, "thematic");
    }

    private String getThemeIdFromBlockSubtype(String subtype) {
        String[] subtypeParts = subtype.split("_");
        return subtypeParts[subtypeParts.length - 1];
    }

    private Option<LentaBlockExData> getAutoSaveBlock(ServiceAndType recordType, String mediaType) {
        if (!StandardMediaType.IMAGE.value().equals(mediaType)) {
            return Option.of(LentaBlockExData.createIgnoredData(recordType));
        }
        return Option.empty();
    }

    private Option<PreviewInfo> getFilePreview(DataApiUserId uid, MpfsResourceId resourceId) {
        try {
            return mpfsClient.getFileInfoOByFileId(MpfsUser.of(uid.asString()), resourceId.owner.toString(), resourceId.fileId).flatMapO(fileInfo -> {
                if (!fileInfo.getMeta().getPreview().isPresent()) {
                    return Option.empty();
                }

                return Option.of(new PreviewInfo(
                        fileInfo.getMeta().getPreview().get().concat("&preview_type=jpg"),
                        fileInfo.path.get(),
                        resourceId.serialize()
                ));
            });
        } catch (HttpException | PermanentHttpFailureException e) {
            logger.warn("Unable to retrieve file preview for resource_id: {}, user: {}", resourceId, uid);
            return Option.empty();
        }
    }

    private ListF<Instant> getFilesEtime(DataApiUserId uid, ListF<String> resourceIds) {
        try {
            //etime in meta - just for proxying to djfs
            return mpfsClient.bulkInfoByResourceIds(uid.forMpfs(), Cf.set("etime"), resourceIds)
                    .map(file -> file.times.etime.map(ut -> new Instant(ut * 1000)).getOrElse(new Instant(file.times.ctime * 1000)));

        } catch (HttpException | PermanentHttpFailureException e) {
            logger.warn("Unable to retrieve files etime for user {}: {}", uid, ExceptionUtils.getAllMessages(e));

            return Cf.list();
        }
    }

    private Option<MpfsLentaBlockFullDescription> getLentaBlockInfo(DataApiUserId uid,
            LentaBlockRecord record)
    {
        ContentBlockMeta meta = ContentBlockMeta.fromBlock(uid, record);
        return mpfsClient.getLentaBlockFilesData(
                uid.forMpfs(), meta.folderId, meta.mediaType,
                MpfsUid.parse(meta.modifierUid),
                TimeUtils.unixTime(record.mFrom.get()), TimeUtils.unixTimeTill(record.mTill.get()), 1);
    }

    private Option<MpfsLentaBlockFullDescription> createSharedFolderInviteBlockDescription(DataApiUserId ownerUid,
            String groupId)
    {
        Option<String> pathO = mpfsClient.getSharedFolderPathByUidAndGroupId(ownerUid.forMpfs(), groupId);

        if (!pathO.isPresent() || !pathO.get().contains(File.separator)) {
            logger.warn("Invalid folder path from mpfs: {} for uid {} and groupId {}", pathO.get(), ownerUid, groupId);
            return Option.empty();
        }

        String path = pathO.get();
        String id = ownerUid + ":" + path;
        String folderName = path.substring(path.lastIndexOf(File.separator) + 1);

        MpfsLentaBlockItemDescription item =
                new MpfsLentaBlockItemDescription(id, id, id, Option.empty(), Option.empty(), Option.empty(),
                        Option.empty(), path, folderName, "dir", Option.empty(), Option.empty());
        return Option.of(new MpfsLentaBlockFullDescription(1, item, Cf.list()));
    }

    @AllArgsConstructor(access = AccessLevel.PRIVATE)
    @ToString
    public static class LentaBlockExData {
        public final Option<ListF<String>> pushPlatforms;
        public final Option<Long> filesCount;
        public final Option<PreviewInfo> preview;
        public final ServiceAndType notificationType;
        public final Option<PhotoReminderDate> reminderDate;
        public final Option<CoolLentaReminderDate> coolLentaReminderDate;
        public final ListF<Integer> geoRegionIds;
        public final NotificationActor actor;
        public final Option<MpfsLentaBlockFullDescription> lentaBlockDescription;
        public final Option<TermDefinition> themeTerm;
        public final boolean createNotification;
        public final Option<String> desktopUrl;

        public static LentaBlockExData createContentData(Option<Long> filesCount, Option<PreviewInfo> preview,
                ServiceAndType notificationType, NotificationActor actor,
                Option<MpfsLentaBlockFullDescription> lentaBlockDescription)
        {
            return new LentaBlockExData(Option.empty(), filesCount, preview, notificationType, Option.empty(), Option.empty(), Cf.list(), actor,
                    lentaBlockDescription, Option.empty(), true, Option.empty());
        }

        public static LentaBlockExData createPhotoReminderData(Option<PreviewInfo> preview,
                ServiceAndType notificationType, PhotoReminderDate reminderDate, NotificationActor actor)
        {
            return new LentaBlockExData(Option.empty(), Option.empty(), preview, notificationType,
                    Option.of(reminderDate), Option.empty(), Cf.list(), actor, Option.empty(), Option.empty(), true, Option.empty());
        }

        public static LentaBlockExData createSharedFolderData(ServiceAndType notificationType,
                NotificationActor actor, Option<MpfsLentaBlockFullDescription> lentaBlockDescription)
        {
            return new LentaBlockExData(Option.empty(), Option.empty(), Option.empty(), notificationType, Option.empty(), Option.empty(), Cf.list(),
                    actor, lentaBlockDescription, Option.empty(), true, Option.empty());
        }

        private static LentaBlockExData createIgnoredData(ServiceAndType notificationType) {
            return new LentaBlockExData(Option.empty(), Option.empty(), Option.empty(), notificationType, Option.empty(), Option.empty(),
                    Cf.list(), NotificationActor.YA_DISK, Option.empty(), Option.empty(), false, Option.empty());
        }

        public static LentaBlockExData coolLentaRemindData(Option<PreviewInfo> preview,
                                                           ServiceAndType notificationType,
                                                           ListF<String> platforms,
                                                           CoolLentaReminderDate coolLentaReminderDate,
                                                           ListF<Integer> geoRegionIds, Option<TermDefinition> themeTerm,
                Option<String> desktopUrl)
        {
            return new LentaBlockExData(Option.of(platforms), Option.empty(), preview, notificationType, Option.empty(),
                    Option.of(coolLentaReminderDate), geoRegionIds, NotificationActor.YA_DISK, Option.empty(), themeTerm,
                    true, desktopUrl);
        }
    }


    @AllArgsConstructor
    @ToString
    public static class PreviewInfo {
        public final String url;
        public final String filePath;
        public final String resourceId;
    }

    @Data
    @AllArgsConstructor
    public static class PhotoReminderDate {
        public final Lazy<Option<Integer>> month;
        public final Lazy<Option<Integer>> year;
        public final int yearsAgo;
        public final Lazy<Option<Instant>> periodStart;

        public PhotoReminderDate(int month, int year, int yearsAgo) {
            this(
                    Lazy.withValue(Option.of(month)),
                    Lazy.withValue(Option.of(year)),
                    yearsAgo,
                    Lazy.withValue(Option.empty()));
        }
    }

    @Data
    @AllArgsConstructor
    public static class CoolLentaReminderDate {
        public final DateTime intervalStart;
        public final DateTime intervalEnd;
        public final Option<DateTime> generationTime;
        public final IntervalType intervalType;

        public CoolLentaReminderDate(DateTime intervalStart, DateTime intervalEnd, Option<DateTime> generationTime) {
            this(intervalStart, intervalEnd, generationTime, TimeIntervalUtils.getMinimalInterval(intervalStart, intervalEnd));
        }
    }

    private enum SpecialDateBlockTypes {
        AUTOUPLOAD,
        PHOTO_SELECTION
    }
}
