package ru.yandex.chemodan.app.notifier;

import java.util.Arrays;
import java.util.concurrent.atomic.AtomicReference;

import lombok.SneakyThrows;
import org.joda.time.DateTime;
import org.joda.time.DateTimeConstants;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Years;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.internal.stubbing.defaultanswers.ForwardsInvocations;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function3V;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserType;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockMeta;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderInvitationBlockFields;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.GeoNamesSource;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.reminder.CoolLentaReminder;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoReminderFields;
import ru.yandex.chemodan.app.lentaloader.reminder.ReminderNotificationType;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.BlockTexts;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.SpecialDateTitleConfigurationRegistry;
import ru.yandex.chemodan.app.lentaloader.test.TestUtils;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.config.CounterConfiguration;
import ru.yandex.chemodan.app.notifier.config.CounterType;
import ru.yandex.chemodan.app.notifier.dao.NotificationBlockDao;
import ru.yandex.chemodan.app.notifier.dao.NotificationBlockDaoImpl;
import ru.yandex.chemodan.app.notifier.dao.NotificationCacheDao;
import ru.yandex.chemodan.app.notifier.dao.NotificationCacheDaoImpl;
import ru.yandex.chemodan.app.notifier.locale.LocaleManager;
import ru.yandex.chemodan.app.notifier.masters.MasterManager;
import ru.yandex.chemodan.app.notifier.masters.counter.CountMasterManager;
import ru.yandex.chemodan.app.notifier.masters.counter.IncrementCountMaster;
import ru.yandex.chemodan.app.notifier.masters.counter.PlusCountMaster;
import ru.yandex.chemodan.app.notifier.masters.counter.UniqueButOneCountMaster;
import ru.yandex.chemodan.app.notifier.masters.counter.UniqueCountMaster;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplate;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplateResolver;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.NotificationVariant;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskTankerMessages;
import ru.yandex.chemodan.app.notifier.notification.toggling.NotificationToggleRegistry;
import ru.yandex.chemodan.app.notifier.push.NotificationPushInfo;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.push.TextMessageGenerator;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.tanker.TankerManager;
import ru.yandex.chemodan.app.notifier.tanker.TankerMessageKey;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.ActionMetadataProcessor;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.DiskMetadataProcessorManager;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorManager;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorTestUtils;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.TextMetadataProcessor;
import ru.yandex.chemodan.app.notifier.worker.task.UpdateNotificationBlockTask;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.eventlog.events.StandardMediaType;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsLentaBlockFileIds;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsResourceTimes;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockFullDescription;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.chemodan.notifier.NotifierEnabledServicesProvider;
import ru.yandex.chemodan.notifier.NotifierUnreadCountProvider;
import ru.yandex.chemodan.util.ReflectionUtils;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.commune.bazinga.impl.BazingaTaskManagerImpl;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.storage.memory.InMemoryBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.tanker.model.FormWithCases;
import ru.yandex.inside.tanker.model.TankerTranslation;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.MoscowTime;

import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyInt;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.when;

/**
 * @author buberman
 */
@ActivateDataApiEmbeddedPg
@ActivateNotificationEmbeddedPg
@ContextConfiguration(classes = NotificationTypeManagerTestContextConfiguration.class)
public class DiskNotificationManagerTest extends DataApiTestSupport {
    @Autowired
    private DataApiManager dataApiManager;
    @Autowired
    private NotificationRecordTypeManager notificationRecordTypeManager;
    @Autowired
    private NotificationTemplateResolver notificationTemplateResolver;

    private DataApiUserId uid;

    private DiskNotificationManager diskNotificationManager;

    private NotificationBlockDao notificationBlockDao;

    @Mock
    private LocaleManager localeManager;
    @Mock
    private NotificationPushManager notificationPushManager;

    private NotificationRecordTypeManager recordTypeManager;

    @Mock
    private LentaManager lentaManager;
    @Mock
    private MpfsClientImpl mpfsClient;
    @Mock
    private NotificationToggleRegistry notificationToggleRegistry;
    @Mock
    private NotificationsGlobalSettingsManager notificationsGlobalSettingsManager;
    @Mock
    private TankerManager tankerManager;
    @Mock
    private NotifierEnabledServicesProvider notifierEnabledServicesProvider;
    @Mock
    private ExperimentsManager experimentsManager;

    @Mock
    private ThemeDefinitionRegistry themeDefinitionRegistry;

    @Mock
    private UserTimezoneHelper userTimezoneHelper;

    @Mock
    private SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry;

    private final BlockTitlesGenerator blockTitlesGenerator = new BlockTitlesGenerator(false, GeoNamesSource.EMPTY);

    @Value("${notifier.blocks.max-count}")
    private int maxBlocksCount;

    @Before
    public void setup() {
        uid = userInitializer.createRandomCleanUserInDefaultShard(DataApiUserType.PASSPORT);

        MockitoAnnotations.initMocks(this);

        recordTypeManager = Mockito.spy(notificationRecordTypeManager);
        Function3V<String, CounterType, String> mockCounters =
                (name, type, param) -> Mockito.when(recordTypeManager.resolveRecordType(name, "disk")).then(a -> {
                    Object r = a.callRealMethod();
                    ReflectionUtils.setFieldValue(r, "counterConfiguration",
                            new CounterConfiguration(CounterType.UNIQUE, Cf.list(param)));
                    return r;
                });
        mockCounters.apply("autoupload", CounterType.UNIQUE, "actor.type");
        mockCounters.apply("photo_reminder", CounterType.UNIQUE, "actor.type");
        mockCounters.apply("shared_folder_file_create", CounterType.UNIQUE, "actor.uid");
        mockCounters.apply("shared_folder_file_update", CounterType.UNIQUE, "actor.uid");

        notificationBlockDao = new NotificationBlockDaoImpl(
                dataApiManager, recordTypeManager,
                new NotifierUnreadCountProvider(dataApiManager, notifierEnabledServicesProvider), maxBlocksCount);

        MetadataProcessorManager metadataProcessorManager = new MetadataProcessorManager(Arrays.asList(
                new TextMetadataProcessor(-1, -1),
                new ActionMetadataProcessor(MetadataProcessorTestUtils.createBlackboxMock(), mpfsClient, 1))
        );

        TankerTranslation translation = new TankerTranslation();
        translation.form = Option.of(new FormWithCases("янв|фев|мар|апр|май|июн|июл|авг|сен|окт|ноя|дек", Cf.map()));

        when(tankerManager.getAllTranslations(eq(DiskTankerMessages.MONTHS_LOCATIVE)))
                .thenReturn(Cf.map("ru", translation));

        TankerTranslation coolTitle = new TankerTranslation();
        coolTitle.form =
                Option.of(new FormWithCases("%{date.start:d} %{month.start:gen} %{date.start:yyyy}", Cf.map()));
        coolTitle.form1 = coolTitle.form2 = coolTitle.form3 = coolTitle.form4 = Option.empty();

        TankerTranslation coolShort = new TankerTranslation();
        coolShort.form = Option.of(new FormWithCases("В тот день вы много снимали", Cf.map()));
        coolShort.form1 = coolShort.form2 = coolShort.form3 = coolShort.form4 = Option.empty();

        when(tankerManager.getAllTranslations(
                eq(new TankerMessageKey("notifier-notifications", "photo_selection_cool_lenta_month"))))
                .thenReturn(Cf.map());
        when(tankerManager.getAllTranslations(
                eq(new TankerMessageKey("notifier-notifications", "photo_selection_cool_lenta_month_title"))))
                .thenReturn(Cf.map("ru", coolTitle));
        when(tankerManager.getAllTranslations(
                eq(new TankerMessageKey("notifier-notifications", "photo_selection_cool_lenta_month_short"))))
                .thenReturn(Cf.map("ru", coolShort));

        TankerTranslation yearsAgoTitle = new TankerTranslation();
        yearsAgoTitle.form = Option.of(new FormWithCases("Помните этот момент?", Cf.map()));
        yearsAgoTitle.form1 = yearsAgoTitle.form2 = yearsAgoTitle.form3 = yearsAgoTitle.form4 = Option.empty();

        TankerTranslation yearsAgoShort = new TankerTranslation();
        yearsAgoShort.form = Option.empty();
        yearsAgoShort.form1 =
                Option.of(new FormWithCases("Посмотрите фото, сделанные %{attributes:n.years} год назад", Cf.map()));
        yearsAgoShort.form2 =
                Option.of(new FormWithCases("Посмотрите фото, сделанные %{attributes:n.years} года назад", Cf.map()));
        yearsAgoShort.form3 =
                Option.of(new FormWithCases("Посмотрите фото, сделанные %{attributes:n.years} лет назад", Cf.map()));
        yearsAgoShort.form4 = Option.of(new FormWithCases("Посмотрите фото, сделанные год назад", Cf.map()));

        when(tankerManager
                .getAllTranslations(eq(new TankerMessageKey("notifier-notifications", "day_in_history_1_android"))))
                .thenReturn(Cf.map("ru", yearsAgoShort));
        when(tankerManager.getAllTranslations(
                eq(new TankerMessageKey("notifier-notifications", "day_in_history_1_android_title"))))
                .thenReturn(Cf.map("ru", yearsAgoTitle));

        TankerTranslation thematicTitle = new TankerTranslation();
        thematicTitle.form = Option.of(new FormWithCases("%{term.nominative} на снимках", Cf.map()));
        thematicTitle.form1 = thematicTitle.form2 = thematicTitle.form3 = thematicTitle.form4 = Option.empty();

        TankerTranslation thematicShortSeason = new TankerTranslation();
        thematicShortSeason.form = Option.of(
                new FormWithCases("Посмотрите свои фото, сделанные %{season.start:abl} %{year.season}", Cf.map()));
        thematicShortSeason.form1 = thematicShortSeason.form2 = thematicShortSeason.form3 = thematicShortSeason.form4 =
                Option.empty();

        TankerTranslation thematicShortMonth = new TankerTranslation();
        thematicShortMonth.form = Option.of(
                new FormWithCases("Посмотрите свои фото за %{month.start:nom} %{date.start:yyyy} года", Cf.map()));
        thematicShortMonth.form1 = thematicShortMonth.form2 = thematicShortMonth.form3 = thematicShortMonth.form4 =
                Option.empty();

        when(tankerManager.getAllTranslations(eq(new TankerMessageKey("notifier-notifications", "thematic_title_v1"))))
                .thenReturn(Cf.map("ru", thematicTitle));
        when(tankerManager.getAllTranslations(
                eq(new TankerMessageKey("notifier-notifications", "thematic_short_title_season_v1"))))
                .thenReturn(Cf.map("ru", thematicShortSeason));
        when(tankerManager.getAllTranslations(
                eq(new TankerMessageKey("notifier-notifications", "thematic_short_title_month_v1"))))
                .thenReturn(Cf.map("ru", thematicShortMonth));

        when(localeManager.getAllMessagesPlaceholders(any())).thenReturn(Cf.set());
        when(localeManager.getAllLocalizations(any(), anyLong())).thenReturn(Cf.map());
        when(localeManager.getAllLocalizations(any(), any(Option.class), any(Option.class))).thenReturn(Cf.map());
        when(localeManager
                .getAllLocalizations(any(NotificationTemplate.class), any(Option.class), any(MetadataWrapper.class)))
                .thenReturn(Cf.map());
        when(localeManager.getAllLocalizations(any(NotificationTemplate.class), anyLong(), any())).thenReturn(Cf.map());
        when(localeManager.getAllLocalizations(any())).thenReturn(Cf.map());

        when(localeManager.getAllMessagesPlaceholders(argThat(template -> {
            return template != null && (
                    template.getType().startsWith("photo_selection_cool_lenta")
                            || template.getType().startsWith("day_in_history_1")
            );
        }))).then(new ForwardsInvocations(new LocaleManager(tankerManager, null)));
        when(localeManager.getAllLocalizations(argThat(template ->
                template != null && (
                        template.key.startsWith("photo_selection_cool_lenta")
                                || template.key.startsWith("day_in_history_1")
                                || template.key.startsWith("thematic_")
                )), any(Option.class), any(Option.class)))
                .then(new ForwardsInvocations(new LocaleManager(tankerManager, null)));

        DiskMetadataProcessorManager diskMetadataProcessorManager =
                new DiskMetadataProcessorManager(metadataProcessorManager, tankerManager, blockTitlesGenerator);

        when(notificationToggleRegistry.isNotificationEnabled(any(), any())).thenReturn(true);
        when(notificationsGlobalSettingsManager.isUserSubscribed(any(), any(), any())).thenReturn(true);

        MasterManager masterManager = new MasterManager(
                new CountMasterManager(Cf.list(
                        new IncrementCountMaster(),
                        new UniqueCountMaster(),
                        new PlusCountMaster(),
                        new UniqueButOneCountMaster()
                ))
        );

        NotificationCacheDao notificationCacheDao = new NotificationCacheDaoImpl(dataApiManager);

        AtomicReference<NotificationManager> notificationManagerReference = new AtomicReference<>();
        NotificationManager notificationManager = new NotificationManager(
                notificationCacheDao,
                notificationBlockDao,
                new BazingaTaskManagerImpl(new InMemoryBazingaStorage()) {
                    @Override
                    @SneakyThrows
                    public FullJobId schedule(OnetimeTask task) {
                        if (task instanceof UpdateNotificationBlockTask) {
                            ReflectionUtils
                                    .setFieldValue(task, "notificationManager", notificationManagerReference.get());
                        }
                        task.execute(null);
                        return super.schedule(task);
                    }
                },
                metadataProcessorManager,
                localeManager,
                Duration.standardMinutes(2),
                notificationPushManager,
                recordTypeManager,
                notificationTemplateResolver,
                masterManager,
                notificationsGlobalSettingsManager
        );
        notificationManagerReference.set(notificationManager);

        when(experimentsManager.getFlags(anyLong())).thenReturn(Cf.list());
        when(themeDefinitionRegistry.getO("nature")).thenReturn(Option.of(TestUtils.createNatureThemeDefinition()));

        diskNotificationManager = new DiskNotificationManager(
                diskMetadataProcessorManager,
                localeManager,
                notificationPushManager,
                lentaManager,
                mpfsClient,
                recordTypeManager,
                notificationTemplateResolver,
                experimentsManager,
                notificationToggleRegistry,
                themeDefinitionRegistry,
                userTimezoneHelper,
                specialDateTitleConfigurationRegistry
        );
    }

    @Test
    public void addAutouploadBlock() {
        LentaBlockRecord lentaBlock = createLentaAutouploadBlock(StandardMediaType.IMAGE);

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        when(mpfsClient.getLentaBlockFilesData(any(), any(), any(), any(), anyInt(), anyInt(), anyInt()))
                .thenReturn(Option.of(makeLentaBlockDescription()));
        mockFileInfo();


        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.none(notificationBlockDao.findNotificationBlock(uid, DiskNotifications.AUTOUPLOAD, lentaBlock.id));
    }

    @Test
    public void tryAddAutouploadBlockWithoutDataInMpfs() {
        LentaBlockRecord lentaBlock = createLentaAutouploadBlock(StandardMediaType.IMAGE);
        ContentBlockMeta meta = ContentBlockMeta.fromBlock(uid, lentaBlock);

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        when(mpfsClient.getLentaBlockFilesData(any(), any(), any(), any(), anyInt(), anyInt(), anyInt()))
                .thenReturn(Option.of(new MpfsLentaBlockFullDescription(2, null, Cf.list())));

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.none(notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.AUTOUPLOAD, meta.folderId));
    }

    @Test
    public void tryAddAutouploadBlockWithVideoMediaType() {
        String preview = "http://preview.ru/some.jpg";
        LentaBlockRecord lentaBlock = createLentaAutouploadBlock(StandardMediaType.VIDEO);

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        when(mpfsClient.getLentaBlocksFileIds(any(), any(), any(), any(), anyInt(), anyInt(), anyInt()))
                .thenReturn(Option.of(new MpfsLentaBlockFileIds(5, Cf.list("fileId"))));
        when(mpfsClient.getFileInfoByFileId(any(), any(), any()))
                .thenReturn(new MpfsFileInfo("", "", MpfsFileMetaDto.builder().preview(preview).build()));


        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.none(notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.AUTOUPLOAD, lentaBlock.id));
    }

    @Test
    public void addPhotoReminderBlock() {
        String preview = "http://preview.ru/some.jpg";
        LentaBlockRecord lentaBlock = createPhotoReminderLentaBlock();

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        mockFileInfo();

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.none(notificationBlockDao.findNotificationBlock(uid, DiskNotifications.PHOTO_REMINDER, lentaBlock.id));
    }

    @Test
    public void addNYearsAgoLentaBlock() {
        testNYearsAgo(1, "Посмотрите фото, сделанные год назад");
        testNYearsAgo(2, "Посмотрите фото, сделанные 2 года назад");
        testNYearsAgo(4, "Посмотрите фото, сделанные 4 года назад");
        testNYearsAgo(5, "Посмотрите фото, сделанные 5 лет назад");

        testNYearsAgo(21, "Посмотрите фото, сделанные 21 год назад");
        testNYearsAgo(11, "Посмотрите фото, сделанные 11 лет назад");
    }

    private void testNYearsAgo(int yearsAgo, String rightFinalText) {
        String preview = "http://preview.ru/some.jpg";
        DateTime intervalStart = IntervalType.ONE_DAY.getIntervalStart(DateTime.now().minus(Years.years(yearsAgo)));
        DateTime intervalEnd = IntervalType.ONE_DAY.getIntervalEnd(intervalStart);

        final DynamicLocalizedString title = new DynamicLocalizedString(Cf.map(
                Language.RUSSIAN, "Подборка. Сентябрь 2018",
                Language.ENGLISH, "Selection of photos. September 2018",
                Language.UKRAINIAN, "Добiрка. Вересень 2018"
        ));
        TitleGenerationContext titleContext =
                new TitleGenerationContext(Random2.R, IntervalType.ONE_DAY, DateTime.now())
                        .withAttribute(CoolLentaReminder.YEARS_ATTRIBUTE_NAME, yearsAgo);

        LentaBlockRecord lentaBlock = createCoolLentRemindBlock(
                intervalStart,
                intervalEnd,
                CoolLentaReminder.BlockGenerationType.N_YEARS, new BlockTexts(titleContext,
                        title, title, title, title, Cf.list())
        );

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        mockFileInfo();

        ArgumentCaptor<MetadataWrapper> metaCaptor = ArgumentCaptor.forClass(MetadataWrapper.class);
        ArgumentCaptor<NotificationPushInfo> pushInfoCaptor = ArgumentCaptor.forClass(NotificationPushInfo.class);
        doNothing().when(notificationPushManager)
                .pushNotificationWithDelay(eq(uid), pushInfoCaptor.capture(), metaCaptor.capture(), any());

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.notEmpty(pushInfoCaptor.getAllValues());
        Assert.notEmpty(metaCaptor.getAllValues());

        NotificationPushInfo pushInfo = pushInfoCaptor.getValue();
        MetadataWrapper metadata = metaCaptor.getValue();

        Assert.equals("day_in_history_1_android", pushInfo.template.getShortMessageKey().key);
        Assert.equals("day_in_history_1_android_title", pushInfo.template.getTitleMessageKey().key);

        Assert.in("attributes:n.years", metadata.meta.keys());

        Assert.equals("" + yearsAgo, metadata.meta.getTs("attributes:n.years").get("text"));

        LocalizedMessage message = localeManager.getAllLocalizations(pushInfo.template.getShortMessageKey(),
                diskNotificationManager.getCountValueForLentaBlock(pushInfo.template, metadata),
                Option.of(metadata))
                .getTs(NotifierLanguage.RUSSIAN);

        Assert.equals(rightFinalText,
                new TextMessageGenerator().generateSimpleTextMessage(message, metadata, Option.empty()));
    }

    private void mockFileInfo() {
        MpfsFileInfo fileInfo = new MpfsFileInfo(
                Option.of("file.jpg"),
                Option.of("file"),
                Option.of("/disk/file.jpg"),
                MpfsFileMetaDto.builder().preview("http://preview.ru/some.jpg").build(),
                MpfsResourceTimes.ZERO);

        when(mpfsClient.getFileInfoByFileId(eq(uid.forMpfs()), eq(uid.serialize()), anyString()))
                .thenReturn(fileInfo);

        when(mpfsClient.getFileInfoOByFileId(eq(uid.forMpfs()), eq(uid.serialize()), anyString()))
                .thenReturn(Option.of(fileInfo));
    }

    @Test
    public void testThematicSeason() {
        String preview = "http://preview.ru/some.jpg";
        DateTime intervalStart = IntervalType.SEASON.getIntervalStart(new DateTime(2019, 1, 15, 0, 0, 0));
        DateTime intervalEnd = IntervalType.SEASON.getIntervalEnd(intervalStart);

        final DynamicLocalizedString title = new DynamicLocalizedString(Cf.map(
                Language.RUSSIAN, "Подборка. Зима 2018-2019",
                Language.ENGLISH, "Selection of photos. The Winter 2018-2019",
                Language.UKRAINIAN, "Добiрка. Зима 2018-2019"
        ));
        TitleGenerationContext titleContext = new TitleGenerationContext(Random2.R, IntervalType.SEASON, DateTime.now())
                .withAttribute(CoolLentaReminder.THEME_ID_ATTRIBUTE_NAME, "nature");

        LentaBlockRecord lentaBlock = createCoolLentRemindBlock(
                intervalStart,
                intervalEnd,
                CoolLentaReminder.BlockGenerationType.THEMATIC, new BlockTexts(titleContext,
                        title, title, title, title, Cf.list())
        );

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        mockFileInfo();

        ArgumentCaptor<MetadataWrapper> metaCaptor = ArgumentCaptor.forClass(MetadataWrapper.class);
        ArgumentCaptor<NotificationPushInfo> pushInfoCaptor = ArgumentCaptor.forClass(NotificationPushInfo.class);
        doNothing().when(notificationPushManager)
                .pushNotificationWithDelay(eq(uid), pushInfoCaptor.capture(), metaCaptor.capture(), any());

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.notEmpty(pushInfoCaptor.getAllValues());
        Assert.notEmpty(metaCaptor.getAllValues());

        NotificationPushInfo pushInfo = pushInfoCaptor.getValue();
        MetadataWrapper metadata = metaCaptor.getValue();

        Assert.equals("thematic_short_title_season_v1", pushInfo.template.getShortMessageKey().key);
        Assert.equals("thematic_title_v1", pushInfo.template.getTitleMessageKey().key);

        Assert.in("term.nominative", metadata.meta.keys());
        Assert.in("season.start:abl", metadata.meta.keys());
        Assert.in("year.season", metadata.meta.keys());

        Assert.equals("природа", metadata.meta.getTs("term.nominative").get("text@ru"));
        Assert.equals("зимой", metadata.meta.getTs("season.start:abl").get("text@ru"));
        Assert.equals("2018–2019", metadata.meta.getTs("year.season").get("text"));

        LocalizedMessage message = localeManager.getAllLocalizations(pushInfo.template.getShortMessageKey(),
                Option.empty(),
                Option.of(metadata))
                .getTs(NotifierLanguage.RUSSIAN);
        Assert.equals("Посмотрите свои фото, сделанные зимой 2018–2019",
                new TextMessageGenerator().generateSimpleTextMessage(message, metadata, Option.empty()));

        LocalizedMessage titleMessage = localeManager.getAllLocalizations(pushInfo.template.getTitleMessageKey(),
                Option.empty(),
                Option.of(metadata))
                .getTs(NotifierLanguage.RUSSIAN);
        Assert.equals("Природа на снимках",
                new TextMessageGenerator().generateSimpleTextMessage(titleMessage, metadata, Option.empty()));
    }

    @Test
    public void testThematicMonth() {
        String preview = "http://preview.ru/some.jpg";
        DateTime intervalStart = IntervalType.MONTH.getIntervalStart(new DateTime(2019, 7, 15, 0, 0, 0));
        DateTime intervalEnd = IntervalType.MONTH.getIntervalEnd(intervalStart);

        final DynamicLocalizedString title = new DynamicLocalizedString(Cf.map(
                Language.RUSSIAN, "Подборка. Июль 2019",
                Language.ENGLISH, "Selection of photos. July 2019",
                Language.UKRAINIAN, "Добiрка. Липень 2019"
        ));
        TitleGenerationContext titleContext = new TitleGenerationContext(Random2.R, IntervalType.MONTH, DateTime.now())
                .withAttribute(CoolLentaReminder.THEME_ID_ATTRIBUTE_NAME, "nature");

        LentaBlockRecord lentaBlock = createCoolLentRemindBlock(
                intervalStart,
                intervalEnd,
                CoolLentaReminder.BlockGenerationType.THEMATIC, new BlockTexts(titleContext,
                        title, title, title, title, Cf.list())
        );

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        mockFileInfo();

        ArgumentCaptor<MetadataWrapper> metaCaptor = ArgumentCaptor.forClass(MetadataWrapper.class);
        ArgumentCaptor<NotificationPushInfo> pushInfoCaptor = ArgumentCaptor.forClass(NotificationPushInfo.class);
        doNothing().when(notificationPushManager)
                .pushNotificationWithDelay(eq(uid), pushInfoCaptor.capture(), metaCaptor.capture(), any());

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.notEmpty(pushInfoCaptor.getAllValues());
        Assert.notEmpty(metaCaptor.getAllValues());

        NotificationPushInfo pushInfo = pushInfoCaptor.getValue();
        MetadataWrapper metadata = metaCaptor.getValue();

        Assert.equals("thematic_short_title_month_v1", pushInfo.template.getShortMessageKey().key);
        Assert.equals("thematic_title_v1", pushInfo.template.getTitleMessageKey().key);

        Assert.in("term.nominative", metadata.meta.keys());
        Assert.in("month.start:nom", metadata.meta.keys());
        Assert.in("date.start:yyyy", metadata.meta.keys());

        Assert.equals("природа", metadata.meta.getTs("term.nominative").get("text@ru"));
        Assert.equals("июль", metadata.meta.getTs("month.start:nom").get("text@ru"));
        Assert.equals("2019", metadata.meta.getTs("date.start:yyyy").get("text"));

        LocalizedMessage message = localeManager.getAllLocalizations(pushInfo.template.getShortMessageKey(),
                Option.empty(),
                Option.of(metadata))
                .getTs(NotifierLanguage.RUSSIAN);
        Assert.equals("Посмотрите свои фото за июль 2019 года",
                new TextMessageGenerator().generateSimpleTextMessage(message, metadata, Option.empty()));

        LocalizedMessage titleMessage = localeManager.getAllLocalizations(pushInfo.template.getTitleMessageKey(),
                Option.empty(),
                Option.of(metadata))
                .getTs(NotifierLanguage.RUSSIAN);
        Assert.equals("Природа на снимках",
                new TextMessageGenerator().generateSimpleTextMessage(titleMessage, metadata, Option.empty()));
    }

    @Test
    public void addCoolLentaReminderBlock() {
        String preview = "http://preview.ru/some.jpg";
        DateTime intervalStart = DateTime.parse("2018-09-01T00:00:00.00+0200");
        DateTime intervalEnd = DateTime.parse("2018-10-01T00:00:00.00+0200");

        final DynamicLocalizedString title = new DynamicLocalizedString(Cf.map(
                Language.RUSSIAN, "Подборка. Сентябрь 2018",
                Language.ENGLISH, "Selection of photos. September 2018",
                Language.UKRAINIAN, "Добiрка. Вересень 2018"
        ));
        LentaBlockRecord lentaBlock = createCoolLentRemindBlock(
                intervalStart,
                intervalEnd,
                CoolLentaReminder.BlockGenerationType.USUAL,
                new BlockTexts(new TitleGenerationContext(Random2.R, IntervalType.ONE_DAY, DateTime.now()),
                        title, title, title, title, Cf.list())
        );

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        mockFileInfo();

        ArgumentCaptor<MetadataWrapper> metaCaptor = ArgumentCaptor.forClass(MetadataWrapper.class);
        ArgumentCaptor<NotificationPushInfo> pushInfoCaptor = ArgumentCaptor.forClass(NotificationPushInfo.class);
        doNothing().when(notificationPushManager)
                .pushNotificationWithDelay(eq(uid), pushInfoCaptor.capture(), metaCaptor.capture(), any());

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.notEmpty(pushInfoCaptor.getAllValues());
        Assert.notEmpty(metaCaptor.getAllValues());

        NotificationPushInfo pushInfo = pushInfoCaptor.getValue();
        MetadataWrapper metadata = metaCaptor.getValue();

        Assert.equals("photo_selection_cool_lenta_month_short", pushInfo.template.getShortMessageKey().key);
        Assert.equals("photo_selection_cool_lenta_month_title", pushInfo.template.getTitleMessageKey().key);

        Assert.in("date.start:yyyy", metadata.meta.keys());
        Assert.in("date.start:d", metadata.meta.keys());
        Assert.in("month.start:gen", metadata.meta.keys());

        Assert.equals("2018", metadata.meta.getTs("date.start:yyyy").get("text"));
        Assert.equals("1", metadata.meta.getTs("date.start:d").get("text"));

        Assert.equals("сентября", metadata.meta.getTs("month.start:gen").get("text@ru"));
        Assert.equals("September", metadata.meta.getTs("month.start:gen").get("text@en"));
        Assert.equals("вересня", metadata.meta.getTs("month.start:gen").get("text@uk"));

        LocalizedMessage message = localeManager.getAllLocalizations(pushInfo.template.getTitleMessageKey(),
                diskNotificationManager.getCountValueForLentaBlock(pushInfo.template, metadata),
                Option.of(metadata))
                .getTs(NotifierLanguage.RUSSIAN);

        Assert.equals("1 сентября 2018",
                new TextMessageGenerator().generateSimpleTextMessage(message, metadata, Option.empty()));
    }

    @Test
    public void addPhotoReminderHiddenBlock() {
        String preview = "http://preview.ru/some.jpg";
        LentaBlockRecord lentaBlock = createPhotoReminderHiddenLentaBlock();

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        mockFileInfo();

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.none(notificationBlockDao
                .findNotificationBlock(uid, DiskNotifications.PHOTO_REMINDER_WORKDAY, lentaBlock.id));
    }

    @Test
    public void tryAdddPhotoReminderBlockForRemovedFile() {
        LentaBlockRecord lentaBlock = createPhotoReminderLentaBlock();

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        when(mpfsClient.getFileInfoByFileId(eq(uid.forMpfs()), eq(uid.serialize()), eq("resource_id1")))
                .thenThrow(new PermanentHttpFailureException("", 404));

        mockFileInfo();

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.none(notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.PHOTO_REMINDER, lentaBlock.id), "Shouldn't create block for removed file");
    }

    @Test
    public void addSharedFolderCreateFileBlock() {
        addSharedFolderBlock(DiskNotifications.SHARED_FOLDER_FILE_CREATE, ContentBlockAction.ADDITION);
    }

    @Test
    public void addSharedFolderUpdateFileBlock() {
        addSharedFolderBlock(DiskNotifications.SHARED_FOLDER_FILE_UPDATE, ContentBlockAction.UPDATE);
    }

    private void addSharedFolderBlock(ServiceAndType notificationType, ContentBlockAction action) {
        PassportUid modifierUid = new PassportUid(123);
        LentaBlockRecord lentaBlock = createLentaContentBlockRecord(StandardMediaType.IMAGE, action);

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        when(mpfsClient.getLentaBlockFilesData(any(), any(), any(), any(), anyInt(), anyInt(), anyInt()))
                .thenReturn(Option.of(makeLentaBlockDescription()));

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.none(notificationBlockDao.findNotificationBlock(uid, notificationType, lentaBlock.id));
    }

    //https://st.yandex-team.ru/CHEMODAN-63914
    @Ignore
    @Test
    public void addSharedFolderInviteReceivedBlock() {
        String inviteHash = "3fccae609b19603c5022dd9605b3eeb1";
        String groupId = "56a4ecba94f386ff2fb6df32dd90e5c9";
        String ownerUid = "321";
        String folderName = "my_shared_folder";
        String folderPath = "/disk/" + folderName;

        LentaBlockRecord lentaBlock = createSharedFolderInviteReceivedLentaBlock(inviteHash, groupId, ownerUid);

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        when(mpfsClient.getSharedFolderPathByUidAndGroupId(eq(MpfsUser.of(ownerUid)), eq(groupId)))
                .thenReturn(Option.of(folderPath));

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        NotificationBlockRecord block = notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.SHARED_FOLDER_INVITE_RECEIVED, lentaBlock.id).get();

        Assert.equals(ownerUid, block.metadata.meta.getTs(block.actor).get("uid"));

        Assert.equals(LentaRecordType.SHARED_FOLDER_INVITE.value(),
                block.metadata.meta.getTs(block.action).get("block-type"));

        Assert.equals(groupId, block.metadata.meta.getTs(block.action).get("group_id"));
        Assert.equals(ownerUid, block.metadata.meta.getTs(block.action).get("owner_uid"));
        Assert.equals(inviteHash, block.metadata.meta.getTs(block.action).get("invite_hash"));

        Assert.equals(folderName, block.metadata.meta.getTs("folder").get("text"));
        Assert.equals(ownerUid + ":" + folderPath, block.metadata.meta.getTs("folder").get("id"));

        Assert.equals("dir", block.metadata.meta.getTs("entity").get("resource_type"));
        Assert.none(block.metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_URL));
    }

    @Test
    public void unresolvedPlaceholders() {
        String preview = "http://preview.ru/some.jpg";
        LentaBlockRecord lentaBlock = createPhotoReminderLentaBlock();

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any())).thenReturn(Option.of(lentaBlock));

        mockFileInfo();


        NotificationType type = NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS;

        type = type.withAlternativeVariants(Cf.list(new NotificationVariant(
                "alter", 100, 0, Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty())));

        Mockito.doReturn(type).when(recordTypeManager).resolveRecordType(Mockito.any());

        NotificationTemplate template = notificationTemplateResolver.consTemplate(type, uid);
        NotificationTemplate mainTemplate = type.consMainTemplate();

        Assert.notEquals(template, mainTemplate);

        ArgumentCaptor<NotificationPushInfo> pushCaptor = ArgumentCaptor.forClass(NotificationPushInfo.class);
        doNothing().when(notificationPushManager)
                .pushNotificationWithDelay(eq(uid), pushCaptor.capture(), any(), any());

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.equals(template.getName(), pushCaptor.getValue().template.getName());

        when(localeManager.getAllMessagesPlaceholders(any())).thenReturn(Cf.set("unresolvable"));

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.equals(mainTemplate.getName(), pushCaptor.getValue().template.getName());
    }

    @Test
    public void photoReminderDateMetadata() {
        String preview = "http://preview.ru/some.jpg";
        LentaBlockRecord lentaBlock = createPhotoReminderLentaBlock();

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any())).thenReturn(Option.of(lentaBlock));

        mockFileInfo();

        int daySec = DateTimeConstants.SECONDS_PER_DAY;
        long year = new DateTime(2018, 1, 1, 0, 0, DateTimeZone.UTC).getMillis() / 1000;

        when(mpfsClient.bulkInfoByResourceIds(any(), any(SetF.class), any())).thenReturn(Cf.range(-1, 2)
                .map(days -> new MpfsFileInfo(Option.empty(), Option.empty(), Option.empty(),
                        MpfsFileMetaDto.builder().build(),
                        new MpfsResourceTimes(0L, Option.of(year + days * daySec), 0L, 0L))));

        ArgumentCaptor<MetadataWrapper> metaCaptor = ArgumentCaptor.forClass(MetadataWrapper.class);
        doNothing().when(notificationPushManager)
                .pushNotificationWithDelay(eq(uid), any(), metaCaptor.capture(), any());

        when(localeManager.getAllMessagesPlaceholders(any())).thenReturn(Cf.set("month", "year"));

        diskNotificationManager.addLentaBlock(uid, lentaBlock.id);

        Assert.some("2018", metaCaptor.getValue().getEntityField("year", "text"));
        Assert.some("янв", metaCaptor.getValue().getEntityField("month", "text@ru"));

        Assert.none(notificationBlockDao.findNotificationBlock(uid, DiskNotifications.PHOTO_REMINDER, lentaBlock.id));
    }

    //https://st.yandex-team.ru/CHEMODAN-63914
    @Ignore
    @Test
    public void addSharedFolderInviteAcceptedBlock() {
        String groupId = "56a4ecba94f386ff2fb6df32dd90e5c9";
        String ownerUid = "321";
        String folderName = "my_shared_folder";
        checkSharedFolderInviteAcceptedNotification(groupId, ownerUid, folderName, 1);
    }

    //https://st.yandex-team.ru/CHEMODAN-63914
    @Ignore
    @Test
    public void addMultipleSharedFolderInviteAcceptedBlocks() {
        String groupId = "56a4ecba94f386ff2fb6df32dd90e5c9";
        String ownerUid = "321";
        String folderName = "my_shared_folder";
        int inviteeCount = 5;
        checkSharedFolderInviteAcceptedNotification(groupId, ownerUid, folderName, inviteeCount);
    }

    private void checkSharedFolderInviteAcceptedNotification(String groupId, String ownerUid, String folderName,
            int inviteeCount)
    {
        String folderPath = "/disk/" + folderName;

        LentaBlockRecord lentaBlock = createSharedFolderInviteAcceptedLentaBlock(groupId, ownerUid);

        when(lentaManager.findBlock(eq(uid), eq(lentaBlock.id), any()))
                .thenReturn(Option.of(lentaBlock));
        when(mpfsClient.getSharedFolderPathByUidAndGroupId(eq(MpfsUser.of(ownerUid)), eq(groupId)))
                .thenReturn(Option.of(folderPath));

        for (int i = 0; i < inviteeCount; i++) {
            diskNotificationManager.addLentaBlock(uid, lentaBlock.id);
        }

        NotificationBlockRecord block = notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.SHARED_FOLDER_INVITE_ACCEPTED, lentaBlock.id).get();

        Assert.equals(NotificationActor.YA_DISK_ACTOR, block.metadata.meta.getTs(block.actor).get("type"));

        Assert.equals(LentaRecordType.SHARED_FOLDER.value(),
                block.metadata.meta.getTs(block.action).get("block-type"));

        Assert.equals(groupId, block.metadata.meta.getTs(block.action).get("group_id"));
        Assert.equals(ownerUid, block.metadata.meta.getTs(block.action).get("owner_uid"));

        Assert.equals(folderName, block.metadata.meta.getTs("folder").get("text"));
        Assert.equals(ownerUid + ":" + folderPath, block.metadata.meta.getTs("folder").get("id"));

        Assert.equals("dir", block.metadata.meta.getTs("entity").get("resource_type"));
        Assert.none(block.metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_URL));
    }

    private LentaBlockRecord createLentaAutouploadBlock(StandardMediaType type) {
        return createLentaContentBlockRecord(type, ContentBlockAction.AUTOSAVE);
    }

    private LentaBlockRecord createLentaContentBlockRecord(StandardMediaType type, ContentBlockAction action) {
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                ContentBlockFields.MEDIA_TYPE.toData(type.value()),
                ContentBlockFields.FOLDER_ID.toData("folderId"),
                ContentBlockFields.MODIFIER_UID.toData("123"),
                ContentBlockFields.FILES_COUNT.toData(10),
                ContentBlockFields.AREA.toData("some_area"),
                ContentBlockFields.ACTION.toData(action)));

        Instant now = new Instant();
        return new LentaBlockRecord(
                "id", 1, LentaRecordType.CONTENT_BLOCK,
                "groupkey",
                Option.of(now.minus(Duration.standardHours(10))),
                now,
                Option.of(now),
                specific);
    }


    private LentaBlockRecord createPhotoReminderLentaBlock() {
        Instant now = new DateTime(2016, 10, 23, 14, 0, 0, MoscowTime.TZ).toInstant();
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                PhotoReminderFields.YEARS_AGO.toData(1),
                PhotoReminderFields.RESOURCE_IDS.toData(Cf.list(uid + ":resource_id1", uid + ":resource_id2")),
                PhotoReminderFields.INTERVAL_START.toData(now.minus(Duration.standardDays(100))),
                PhotoReminderFields.INTERVAL_END.toData(now),
                PhotoReminderFields.NOTIFICATION_TYPE.toData(ReminderNotificationType.REMINDER_NOTIFICATION_DEFAULT)));

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.PHOTO_REMIND_BLOCK,
                "groupkey",
                Option.of(now.minus(Duration.standardHours(10))),
                now,
                Option.of(now),
                specific);
    }

    private LentaBlockRecord createCoolLentRemindBlock(DateTime from, DateTime to,
            CoolLentaReminder.BlockGenerationType generationType, BlockTexts texts)
    {
        Instant now = new DateTime(2017, 9, 12, 19, 29, 0, MoscowTime.TZ).toInstant();

        MinimalCoolLentaBlock block = new MinimalCoolLentaBlock(
                "id",
                (generationType == CoolLentaReminder.BlockGenerationType.THEMATIC ? "thematic_" : "default_") +
                        TimeIntervalUtils.getMinimalInterval(from, to) +
                        (generationType == CoolLentaReminder.BlockGenerationType.THEMATIC ? "_nature" : ""),
                Cf.list(uid + ":resource_id1").map(MpfsResourceId::parse),
                MpfsResourceId.parse(uid + ":resource_id1"),
                from.getZone().getID(),
                Option.empty(),
                from.toInstant(),
                to.toInstant(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                false
        );

        MapF<String, DataField> specific = CoolLentaReminder.getSpecificLentaBlockFields(generationType, block,
                texts, Cf.list("ios", "android"), DateTime.now(), Option.empty());

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.PHOTO_SELECTION_BLOCK,
                "groupkey",
                Option.of(now.minus(Duration.standardHours(10))),
                now,
                Option.of(now),
                specific);
    }

    private LentaBlockRecord createPhotoReminderHiddenLentaBlock() {
        Instant now = new DateTime(2016, 10, 23, 14, 0, 0, MoscowTime.TZ).toInstant();
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                PhotoReminderFields.YEARS_AGO.toData(1),
                PhotoReminderFields.RESOURCE_IDS.toData(Cf.list(uid + ":resource_id1", uid + ":resource_id2")),
                PhotoReminderFields.INTERVAL_START.toData(now.minus(Duration.standardDays(100))),
                PhotoReminderFields.INTERVAL_END.toData(now),
                PhotoReminderFields.NOTIFICATION_TYPE.toData(ReminderNotificationType.REMINDER_NOTIFICATION_WORKDAY)));

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.PHOTO_REMIND_HIDDEN_BLOCK,
                "groupkey",
                Option.of(now.minus(Duration.standardHours(10))),
                now,
                Option.of(now),
                specific);
    }

    private MpfsLentaBlockFullDescription makeLentaBlockDescription() {
        String preview = "http://preview.ru/some.jpg";
        return new MpfsLentaBlockFullDescription(2,
                new MpfsLentaBlockItemDescription("", "", "",
                        Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                        "", "", "", Option.empty(), Option.empty()),
                Cf.list(new MpfsLentaBlockItemDescription("", "", "",
                                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                                "", "", "", Option.empty(), Option.of(preview)),
                        new MpfsLentaBlockItemDescription("", "", "",
                                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                                "", "", "", Option.empty(), Option.of(preview)))
        );
    }

    private LentaBlockRecord createSharedFolderInviteReceivedLentaBlock(String inviteHash, String groupId,
            String ownerUid)
    {
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                SharedFolderInvitationBlockFields.INVITE_HASH.toData(inviteHash),
                SharedFolderInvitationBlockFields.GROUP_ID.toData(groupId),
                SharedFolderInvitationBlockFields.OWNER_UID.toData(ownerUid)));

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.SHARED_FOLDER_INVITE,
                "groupkey",
                Option.of(now),
                now,
                Option.of(now),
                specific);
    }

    private LentaBlockRecord createSharedFolderInviteAcceptedLentaBlock(String groupId, String ownerUid) {
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                SharedFolderBlockFields.GROUP_ID.toData(groupId),
                SharedFolderBlockFields.OWNER_UID.toData(ownerUid)));

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.SHARED_FOLDER,
                "groupkey",
                Option.empty(),
                now,
                Option.empty(),
                specific);
    }
}
