package ru.yandex.chemodan.app.notifier;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.app.dataapi.DataApiBaseContextConfiguration;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.retry.RetryClauses;
import ru.yandex.chemodan.app.lentaloader.ThemeRegistryContextConfiguration;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManagerImpl;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.SpecialDateTitleConfigurationRegistry;
import ru.yandex.chemodan.app.notifier.admin.web.NotifierTypesAdminContextConfiguration;
import ru.yandex.chemodan.app.notifier.dao.NotificationBlockDao;
import ru.yandex.chemodan.app.notifier.dao.NotificationBlockDaoImpl;
import ru.yandex.chemodan.app.notifier.dao.NotificationCacheDao;
import ru.yandex.chemodan.app.notifier.dao.NotificationCacheDaoImpl;
import ru.yandex.chemodan.app.notifier.locale.LocaleContextConfiguration;
import ru.yandex.chemodan.app.notifier.locale.LocaleManager;
import ru.yandex.chemodan.app.notifier.masters.MasterManager;
import ru.yandex.chemodan.app.notifier.masters.MastersConfiguration;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplateResolver;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.toggling.NotificationToggleRegistry;
import ru.yandex.chemodan.app.notifier.promo.PromoActionManager;
import ru.yandex.chemodan.app.notifier.push.NotificationPushClientContext;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.settings.NotificationSettingsContextConfiguration;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.settings.NotifierWidgetSettingsManager;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.DiskMetadataProcessorManager;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorManager;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.notifier.NotifierUnreadCountProvider;
import ru.yandex.chemodan.notifier.NotifierUnreadCountProviderContextConfiguration;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.util.retry.RetryProxy;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistryContextConfiguration;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.inside.passport.blackbox2.Blackbox2;

/**
 * @author akirakozov
 */
@Configuration
@Import({
        DataApiBaseContextConfiguration.class,
        LocaleContextConfiguration.class,
        NotificationPushClientContext.class,
        NotificationSettingsContextConfiguration.class,
        MastersConfiguration.class,
        NotificationTypeManagerContextConfiguration.class,
        NotifierTypesAdminContextConfiguration.class,
        NotifierUnreadCountProviderContextConfiguration.class,
        YandexStaffUserRegistryContextConfiguration.class,
        ThemeRegistryContextConfiguration.class,
})
public class NotificationBaseContextConfiguration {
    @Value("${notifier.datasource.retry.count}")
    private int retryCount;

    private <T> T retryProxy(T impl, Class<T> iface) {
        return RetryProxy.proxy(impl, iface, retryCount,
                RetryClauses.dataApiAccess,
                RetryProxy.pushToNdcInterceptor());
    }

    @Bean
    public NotificationCacheDao notificationCacheDao(DataApiManager dataApiManager) {
        NotificationCacheDaoImpl impl = new NotificationCacheDaoImpl(dataApiManager);
        return retryProxy(impl, NotificationCacheDao.class);
    }

    @Bean
    public NotificationBlockDao notificationBlockDao(
            DataApiManager dataApiManager,
            NotificationRecordTypeManager notificationRecordTypeManager,
            NotifierUnreadCountProvider unreadCountProvider,
            @Value("${notifier.blocks.max-count}") int maxBlocksCount)
    {
        NotificationBlockDaoImpl impl = new NotificationBlockDaoImpl(
                dataApiManager, notificationRecordTypeManager, unreadCountProvider, maxBlocksCount);
        return retryProxy(impl, NotificationBlockDao.class);
    }

    @Bean
    public LentaManager lentaManager(DataApiManager dataApiManager, PgBazingaStorage bazingaStorage, ExperimentsManager experimentsManager) {
        return new LentaManagerImpl(dataApiManager, bazingaStorage, experimentsManager, MeteredCache.noCache(), MeteredCache.noCache());
    }

    @Bean
    public NotificationManager notificationManager(
            NotificationCacheDao notificationCacheDao,
            NotificationBlockDao notificationBlockDao,
            BazingaTaskManager bazingaTaskManager,
            MetadataProcessorManager metadataProcessorManager,
            LocaleManager localeManager,
            NotificationPushManager notificationPushManager,
            NotificationRecordTypeManager notificationRecordTypeManager,
            NotificationTemplateResolver notificationTemplateResolver,
            MasterManager masterManager,
            NotificationsGlobalSettingsManager notificationsGlobalSettingsManager,
            @Value("${notifier.update.block.delay}") Duration updateBlockDelay)
    {
        return new NotificationManager(
                notificationCacheDao,
                notificationBlockDao,
                bazingaTaskManager,
                metadataProcessorManager,
                localeManager,
                updateBlockDelay,
                notificationPushManager,
                notificationRecordTypeManager,
                notificationTemplateResolver,
                masterManager,
                notificationsGlobalSettingsManager
        );
    }

    @Bean
    @Qualifier("notification-client")
    @OverridableValuePrefix("notification-client")
    public HttpClientConfigurator httpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public NotificationClient notificationClient(NotificationManager manager, @Qualifier("notification-client")
            HttpClientConfigurator configurator, @Value("${notifier.intapi.url}") String url) {
        return new CloudNotificationClient(manager, configurator.configure(), url);
    }

    @Bean
    public DiskNotificationManager diskNotificationManager(
            DiskMetadataProcessorManager diskMetadataProcessorManager,
            LocaleManager localeManager,
            NotificationPushManager notificationPushManager,
            LentaManager lentaManager,
            MpfsClient mpfsClient,
            NotificationRecordTypeManager notificationRecordTypeManager,
            NotificationTemplateResolver notificationTemplateResolver,
            NotificationToggleRegistry notificationToggleRegistry,
            ExperimentsManager experimentsManager,
            ThemeDefinitionRegistry themeDefinitionRegistry, UserTimezoneHelper userTimezoneHelper,
            SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry)
    {
        return new DiskNotificationManager(
                diskMetadataProcessorManager,
                localeManager,
                notificationPushManager,
                lentaManager,
                mpfsClient,
                notificationRecordTypeManager,
                notificationTemplateResolver,
                experimentsManager,
                notificationToggleRegistry,
                themeDefinitionRegistry,
                userTimezoneHelper,
                specialDateTitleConfigurationRegistry
        );
    }

    @Bean
    public NotificationToggleRegistry notificationToggleRegistry(
            @Qualifier("zkRoot") ZkPath zkRoot,
            ZkManager zkManager)
    {
        NotificationToggleRegistry registry = new NotificationToggleRegistry(zkRoot.child("notification-toggle"));
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public PromoActionManager promoActionManager(
            NotificationPushManager pushManager,
            NotificationRecordTypeManager typeManager,
            BazingaTaskManager bazingaTaskManager)
    {
        return new PromoActionManager(pushManager, typeManager, bazingaTaskManager);
    }

    @Bean
    public NotifierWidgetSettingsManager notifierWidgetSettingsManager(DataApiManager dataApiManager) {
        return new NotifierWidgetSettingsManager(dataApiManager);
    }

    @Bean
    public SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry(@Qualifier("zkRoot") ZkPath zkRoot, ZkManager zkManager) {
        SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry =
                new SpecialDateTitleConfigurationRegistry(zkRoot.child("cool-lenta-reminder-special-dates"));
        zkManager.addClient(specialDateTitleConfigurationRegistry);
        return specialDateTitleConfigurationRegistry;
    }

    @Bean
    public UserTimezoneHelper userTimezoneHelper(Blackbox2 blackbox2) {
        return new UserTimezoneHelper(blackbox2);
    }
}
