package ru.yandex.chemodan.app.notifier;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.OrderedUUID;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.dao.NotificationBlockDao;
import ru.yandex.chemodan.app.notifier.dao.NotificationCacheDao;
import ru.yandex.chemodan.app.notifier.locale.LocaleManager;
import ru.yandex.chemodan.app.notifier.log.NotificationBlockUpdateMode;
import ru.yandex.chemodan.app.notifier.log.NotifierEvent;
import ru.yandex.chemodan.app.notifier.masters.MasterManager;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.chemodan.app.notifier.notification.NewNotificationData;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockUpdate;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplate;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplateResolver;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.push.NotificationPushInfo;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorManager;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorUtils;
import ru.yandex.chemodan.app.notifier.worker.task.UpdateNotificationBlockTask;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
@AllArgsConstructor
public class NotificationManager {
    private static final Logger logger = LoggerFactory.getLogger(NotificationManager.class);

    public static final String ACTOR_KEY = "actor";
    public static final String ENTITY_KEY = "entity";
    public static final String ACTION_KEY = "action";

    protected static final int MAX_ACTORS_COUNT = 100;

    private final NotificationCacheDao notificationCacheDao;
    private final NotificationBlockDao notificationBlockDao;
    private final BazingaTaskManager bazingaTaskManager;
    private final MetadataProcessorManager metadataProcessorManager;
    private final LocaleManager localeManager;
    private final Duration updateBlockDelay;
    private final NotificationPushManager notificationPushManager;
    private final NotificationRecordTypeManager notificationRecordTypeManager;
    private final NotificationTemplateResolver notificationTemplateResolver;
    private final MasterManager masterManager;
    private final NotificationsGlobalSettingsManager notificationsGlobalSettingsManager;

    public void createNewNotification(NewNotificationData data) {
        DataApiUserId uid = data.owner;
        MetadataWrapper metadata = preProccessMetadata(data.metadata, data.actor);
        NotificationRecord record = new NotificationRecord(
                OrderedUUID.generateOrderedUUID(),
                data.actor, data.subscriptionKey, data.groupKey,
                data.type, metadata, Instant.now());
        String serviceName = data.type.getService().name;
        notificationCacheDao.createNewNotification(uid, record);
        scheduleBlockUpdate(uid, serviceName, data.groupKey, data.type);
    }

    private MetadataWrapper preProccessMetadata(MetadataWrapper metadata, NotificationActor actor) {
        if (!metadata.getEntityFieldsO(MetadataEntityNames.ACTOR).isPresent()) {
            MetadataProcessorUtils.addActor(metadata, actor);
        }
        return metadata;
    }

    private void scheduleBlockUpdate(DataApiUserId uid, String service, String groupKey, NotificationType type) {
        UpdateNotificationBlockTask updateNotificationBlockTask =
                new UpdateNotificationBlockTask(uid, service, type.value(), groupKey);
        if (!bazingaTaskManager.isJobActive(updateNotificationBlockTask)) {
            bazingaTaskManager.schedule(updateNotificationBlockTask.instant());

            NotifierEvent.blockUpdateScheduled(uid, groupKey, type).log();
        } else {
            NotifierEvent.blockUpdateAlreadyScheduled(uid, groupKey, type).log();
        }
    }

    private void rescheduleBlockUpdate(DataApiUserId uid, String service, String groupKey, NotificationType type) {
        UpdateNotificationBlockTask updateNotificationBlockTask =
                new UpdateNotificationBlockTask(uid, service, type.value(), groupKey);
        bazingaTaskManager.schedule(updateNotificationBlockTask, Instant.now().plus(updateBlockDelay));
        NotifierEvent.blockUpdateScheduled(uid, groupKey, type).log();
    }

    public void updateOrCreateNotificationBlock(
            DataApiUserId userId, String service, String type, String groupKey)
    {
        updateOrCreateNotificationBlock(userId, service,
                notificationRecordTypeManager.resolveRecordType(type, service), groupKey);
    }


    public void updateOrCreateNotificationBlock(
            DataApiUserId userId, String service, NotificationType type, String groupKey)
    {
        ListF<NotificationRecord> newNotifications =
                notificationCacheDao.findNewNotifications(userId, type, groupKey);

        if (newNotifications.isEmpty()) {
            logger.debug("No new notifications found.");
            return;
        }

        if (!notificationsGlobalSettingsManager
                .isUserSubscribed(userId, GlobalSubscriptionChannel.ALL, type.getSettingsGroupName()))
        {
            logger.info("Didn't process notification of type {} for uid {} because it's disabled", type, userId);
            notificationCacheDao.removeNewNotifications(userId, type, groupKey, newNotifications.map(n -> n.id));
            return;
        }

        Option<NotificationBlockRecord> block =
                notificationBlockDao.findNotificationBlock(userId, type.getFullName(), groupKey);
        if (block.isPresent()) {
            updateNotificationBlock(userId, service, block.get(), newNotifications);
        } else {
            createNewAndSiftOldestBlock(userId, newNotifications);
        }

        notificationCacheDao.removeNewNotifications(userId, type, groupKey, newNotifications.map(n -> n.id));

        ListF<Tuple2<NotificationRecord, MetadataWrapper>> metadataForPush =
                metadataProcessorManager.createNotificationMetadataForPush(newNotifications);

        NotificationTemplate template = notificationTemplateResolver.consTemplate(type, userId);

        metadataForPush.forEach(pair -> notificationPushManager.pushNotificationWithDelay(
                userId,
                NotificationPushInfo.fromRecord(template, pair.get1()),
                pair.get2()
        ));

        rescheduleBlockUpdate(userId, service, groupKey, type);
    }

    public void markMessageAsRead(DataApiUserId userId, String groupKey, NotificationType recordType) {
        notificationBlockDao.markNotificationBlockAsRead(userId, groupKey, recordType);
    }

    private void updateNotificationBlock(DataApiUserId userId, String service,
            NotificationBlockRecord block, ListF<NotificationRecord> newNotifications)
    {
        ListF<NotificationRecord> sortedNotProcessed = newNotifications
                .filter(n -> n.ctime.isAfter(block.mtime))
                .sorted().reverse();

        if (sortedNotProcessed.isEmpty()) {
            logger.debug("No new notifications found.");
            return;
        }

        NotificationRecord newestNotification = sortedNotProcessed.first();

        NotificationBlockUpdate update =
                masterManager.countMasterManager.adjustCount(new NotificationBlockUpdate(), block, sortedNotProcessed);

        long count = update.count.getOrElse(0L);

        MetadataWrapper metadata = metadataProcessorManager
                .createNotificationMetadata(newestNotification, Option.of(count));

        // mainly migration purposes, used for syncing isRead/modification times with another system (CHEMODAN-47125)
        Instant mtime = metadata.getEntityField(MetadataEntityNames.OTHER, "mtime")
                .map(t -> new Instant(Long.parseLong(t)))
                .getOrElse(newestNotification.ctime);
        Boolean skipMtimeUpdate = metadata.getEntityField(MetadataEntityNames.OTHER, "skip_mtime_update")
                .map(Boolean::parseBoolean)
                .getOrElse(false);
        Boolean isRead = metadata.getEntityField(MetadataEntityNames.OTHER, "is_read")
                .map(Boolean::parseBoolean)
                .getOrElse(false);

        if (!skipMtimeUpdate) {
            update = update.withMtime(mtime);
        }
        update = isRead ? update.viewed() : update.unviewed();

        update = update.withMetadata(metadata);

        NotificationTemplate template = notificationTemplateResolver.consTemplate(block.type, userId);

        MapF<NotifierLanguage, LocalizedMessage> messages =
                localeManager.getAllLocalizations(template, count, metadata);

        notificationBlockDao.updateNotificationBlock(userId, service, block.id, update, messages);

        NotifierEvent
                .createOrUpdateNotificationBlock(userId, NotificationBlockUpdateMode.UPDATE,
                        newestNotification.groupKey, newestNotification.type,
                        update.counterUniques.getOrElse(Cf.list()), template.getName())
                .log();
    }

    private void createNewAndSiftOldestBlock(
            DataApiUserId userId, ListF<NotificationRecord> newNotifications)
    {
        if (newNotifications.isEmpty()) {
            return;
        }

        ListF<NotificationRecord> sortedNotifications = newNotifications.sorted().reverse().take(MAX_ACTORS_COUNT);
        NotificationRecord newestNotification = sortedNotifications.first();

        NotificationBlockUpdate update =
                masterManager.countMasterManager.startCount(new NotificationBlockUpdate(), newNotifications);
        Option<Long> count = newestNotification.metadata.getEntityField("count", "text").map(Long::parseLong).orElse(update.count);

        MetadataWrapper metadata = metadataProcessorManager
                .createNotificationMetadata(newestNotification, count);

        // this is used primarily for transition, allowing overriding of normally constant fields
        Option<Instant> ctime =
                metadata.getEntityField(MetadataEntityNames.OTHER, "ctime").map(t -> new Instant(Long.parseLong(t)));
        Option<Boolean> isRead =
                metadata.getEntityField(MetadataEntityNames.OTHER, "is_read").map(Boolean::parseBoolean);

        Option<String> previewKey = metadata.getEntityField(MetadataEntityNames.ENTITY, MetadataEntityNames.PREVIEW_URL)
                .filterNot(String::isEmpty).map(v -> ENTITY_KEY);

        // TODO: determine "actor" and "preview" objects according to notification type
        NotificationBlockRecord block = new NotificationBlockRecord(
                OrderedUUID.generateOrderedUUID(),
                update.counterUniques.getOrElse(Cf.list()),
                newestNotification.groupKey,
                newestNotification.type,
                count,
                metadata,
                ACTOR_KEY,
                previewKey,
                ACTION_KEY,
                ctime.getOrElse(newNotifications.map(n -> n.ctime).sorted().first()),
                ctime.getOrElse(newestNotification.ctime),
                isRead.getOrElse(false),
                newestNotification.subscriptionKey);

        NotificationTemplate template = notificationTemplateResolver.consTemplate(block.type, userId);

        MapF<NotifierLanguage, LocalizedMessage> messages =
                localeManager.getAllLocalizations(template, block.count, metadata);

        notificationBlockDao.createNewNotificationBlockAndSiftOldest(userId, block, messages);

        NotifierEvent.createOrUpdateNotificationBlock(userId, NotificationBlockUpdateMode.CREATE,
                newestNotification.groupKey, newestNotification.type, block.counterUniques, template.getName()).log();
    }

    public int getUnviewedBlocksCountForService(DataApiUserId userId, String service) {
        return notificationBlockDao.getUnviewedBlocksCountForService(userId, service);
    }

    public int getTotalUnviewedBlocksCountForEnabledServices(DataApiUserId userId) {
        return notificationBlockDao.getTotalUnviewedBlocksCountForEnabledServices(userId);
    }

    public void deleteNotificationBlock(DataApiUserId userId, NotificationType type, String groupKey) {
        Option<NotificationBlockRecord> block =
                notificationBlockDao.findNotificationBlock(userId, type.getFullName(), groupKey);

        if (!block.isPresent()) {
            throw new NotFoundException("Notification was not found");
        }

        notificationBlockDao.deleteBlock(userId, block.get());
    }
}
