package ru.yandex.chemodan.app.notifier;

import java.util.Arrays;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserType;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.config.CounterConfiguration;
import ru.yandex.chemodan.app.notifier.dao.NotificationBlockDao;
import ru.yandex.chemodan.app.notifier.dao.NotificationBlockDaoImpl;
import ru.yandex.chemodan.app.notifier.dao.NotificationCacheDao;
import ru.yandex.chemodan.app.notifier.dao.NotificationCacheDaoImpl;
import ru.yandex.chemodan.app.notifier.locale.LocaleManager;
import ru.yandex.chemodan.app.notifier.masters.MasterManager;
import ru.yandex.chemodan.app.notifier.masters.counter.CountMasterManager;
import ru.yandex.chemodan.app.notifier.masters.counter.IncrementCountMaster;
import ru.yandex.chemodan.app.notifier.masters.counter.PlusCountMaster;
import ru.yandex.chemodan.app.notifier.masters.counter.UniqueButOneCountMaster;
import ru.yandex.chemodan.app.notifier.masters.counter.UniqueCountMaster;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NewNotificationData;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplate;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplateResolver;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManagerTest;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.ActionMetadataProcessor;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.CommentMetadataProcessor;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorManager;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorTestUtils;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.ResourceMetadataProcessor;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.TextMetadataProcessor;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.notifier.NotifierEnabledServicesProvider;
import ru.yandex.chemodan.notifier.NotifierUnreadCountProvider;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

import static org.mockito.Mockito.any;
import static org.mockito.Mockito.anyLong;
import static org.mockito.Mockito.when;

/**
 * @author akirakozov
 */
@ActivateDataApiEmbeddedPg
@ActivateNotificationEmbeddedPg
@ContextConfiguration(classes = NotificationTypeManagerTestContextConfiguration.class)
public class NotificationManagerTest extends DataApiTestSupport {

    @Autowired
    private DataApiManager dataApiManager;

    private DataApiUserId uid;
    private NotificationManager notificationManager;
    private NotificationCacheDao notificationCacheDao;
    private NotificationBlockDao notificationBlockDao;
    private MetadataProcessorManager metadataProcessorManager;

    private static int COMMENT_MAX_LENGTH = 39;

    @Mock
    private LocaleManager localeManager;
    @Mock
    private NotificationPushManager notificationPushManager;
    @Mock
    private MpfsClient mpfsClient;
    @Mock
    private BazingaTaskManager bazingaTaskManager;
    @Mock
    private NotificationsGlobalSettingsManager notificationsGlobalSettingsManager;
    @Mock
    private NotifierEnabledServicesProvider notifierEnabledServicesProvider;

    @Autowired
    private NotificationRecordTypeManager notificationRecordTypeManager;
    @Autowired
    private NotificationTemplateResolver notificationTemplateResolver;

    @Value("${notifier.blocks.max-count}")
    private int maxBlocksCount;

    @Before
    public void setup() {
        uid = userInitializer.createRandomCleanUserInDefaultShard(DataApiUserType.PASSPORT);

        MockitoAnnotations.initMocks(this);
        notificationCacheDao = new NotificationCacheDaoImpl(dataApiManager);
        notificationBlockDao = new NotificationBlockDaoImpl(
                dataApiManager, notificationRecordTypeManager,
                new NotifierUnreadCountProvider(dataApiManager, notifierEnabledServicesProvider), maxBlocksCount);

        TextMetadataProcessor textProcessor = new TextMetadataProcessor(COMMENT_MAX_LENGTH, 3);
        metadataProcessorManager = new MetadataProcessorManager(Arrays.asList(
                textProcessor,
                new ActionMetadataProcessor(MetadataProcessorTestUtils.createBlackboxMock(), mpfsClient, 1),
                new ResourceMetadataProcessor(mpfsClient, 1),
                new CommentMetadataProcessor(mpfsClient, textProcessor, 1)));

        when(localeManager.getAllLocalizations(any(), anyLong())).thenReturn(Cf.map());
        when(localeManager.getAllLocalizations(any(), any(Option.class), any(Option.class))).thenReturn(Cf.map());
        when(localeManager.getAllLocalizations(any(NotificationTemplate.class), any(Option.class), any(MetadataWrapper.class))).thenReturn(Cf.map());
        when(localeManager.getAllLocalizations(any(NotificationTemplate.class), anyLong(), any())).thenReturn(Cf.map());
        when(localeManager.getAllLocalizations(any())).thenReturn(Cf.map());

        MasterManager masterManager = new MasterManager(
                new CountMasterManager(Cf.list(
                        new IncrementCountMaster(),
                        new UniqueCountMaster(),
                        new PlusCountMaster(),
                        new UniqueButOneCountMaster()
                ))
        );

        when(notificationsGlobalSettingsManager.isUserSubscribed(any(), any(), any())).thenReturn(true);

        notificationManager = new NotificationManager(
                notificationCacheDao,
                notificationBlockDao,
                bazingaTaskManager,
                metadataProcessorManager,
                localeManager,
                Duration.standardMinutes(2),
                notificationPushManager,
                notificationRecordTypeManager,
                notificationTemplateResolver,
                masterManager,
                notificationsGlobalSettingsManager
        );
    }

    @Test
    public void createNewNotificationBlock() {
        NotificationType type = NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS;
        Instant start = Instant.now().minus(Duration.standardHours(3));
        String group = "group1";
        long newCount = 3;
        ListF<String> actors = createNewNotificationsAndGetLastActors(start, group, newCount).reverse();

        notificationManager.updateOrCreateNotificationBlock(uid, DiskServices.DISK, type, group);
        NotificationBlockRecord block = notificationBlockDao
                .findNotificationBlock(uid, type.getFullName(), group)
                .getOrThrow("Notification block not created");

        Assert.some(newCount - 1, block.count);
        Assert.equals(start.plus(Duration.standardMinutes(0)), block.ctime);
        Assert.equals(start.plus(Duration.standardMinutes(newCount - 1)), block.mtime);
        Assert.assertListsEqual(actors, block.counterUniques);

        ListF<NotificationRecord> newNotifications =
                notificationCacheDao.findNewNotifications(uid, type, group);
        Assert.assertEmpty(newNotifications);
    }

    @Test
    public void createDefaultNotificationBlock() {
        NotificationType type = NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS
                .withCounterConfiguration(CounterConfiguration.NONE);

        String group = Random2.R.nextAlnum(12);
        NotificationRecord record = NotificationTestUtils.createDefaultNotificationRecord(group, type);

        notificationCacheDao.createNewNotification(uid, record);

        notificationManager.updateOrCreateNotificationBlock(
                uid, DiskServices.DISK, type, group);

        NotificationBlockRecord block = notificationBlockDao
                .findNotificationBlock(uid, type.getFullName(), group)
                .getOrThrow("Notification block not created");

        Assert.none(block.count);
        Assert.assertListsEqual(Cf.list(), block.counterUniques);

        ListF<NotificationRecord> newNotifications = notificationCacheDao.findNewNotifications(
                uid, type, group);
        Assert.assertEmpty(newNotifications);
    }

    @Test
    public void updateNotificationBlock() {
        NotificationType type = NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS;
        Instant start = Instant.now().minus(Duration.standardHours(3));
        String group = "group1";
        long newCount = 4;
        ListF<String> actors = createNewNotificationsAndGetLastActors(start, group, newCount);

        // mtime of this block is after of one new notification
        NotificationBlockRecord initialBlock = NotificationTestUtils.createTestNotificationBlockRecord(group,
                start.plus(Duration.standardMinutes(1).minus(Duration.standardSeconds(10))));

        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, initialBlock, Cf.hashMap());

        notificationManager.updateOrCreateNotificationBlock(uid, DiskServices.DISK, type, group);
        NotificationBlockRecord block = notificationBlockDao
                .findNotificationBlock(uid, type.getFullName(), group)
                .getOrThrow("Notification block not created");

        // -1 because one of the new notifications was outdated, another -1 because we count "actors but one"
        Assert.equals(newCount + initialBlock.count.get() - 2, block.count.get());
        Assert.equals(initialBlock.ctime, block.ctime);
        Assert.equals(start.plus(Duration.standardMinutes(newCount - 1)), block.mtime);

        ListF<String> realBlockActors = actors.take(3).plus(initialBlock.counterUniques);
        Assert.equals(realBlockActors, block.counterUniques);

        ListF<NotificationRecord> newNotifications = notificationCacheDao.findNewNotifications(
                uid, NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS, group);
        Assert.assertEmpty(newNotifications);
    }

    @Test
    public void updateNotificationBlockWithSameUser() {
        NotificationType type = NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS;
        Instant start = Instant.now().minus(Duration.standardHours(3));
        String group = "group1";

        NotificationBlockRecord initialBlock = NotificationTestUtils
                .createTestNotificationBlockRecord(group, start)
                .withCounterUniques(Cf.range(0, 2).map(i -> NotificationTestUtils.createRandomActor().serialize()));
        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, initialBlock, Cf.hashMap());

        NotificationRecord record = NotificationTestUtils
                .createTestNewNotificationRecord(group, start.plus(Duration.standardMinutes(10)))
                .withLastActor(NotificationActor.consFromSerialized(initialBlock.counterUniques.get(1)));
        notificationCacheDao.createNewNotification(uid, record);

        notificationManager.updateOrCreateNotificationBlock(uid, DiskServices.DISK, type, group);
        NotificationBlockRecord block = notificationBlockDao
                .findNotificationBlock(uid, type.getFullName(), group)
                .getOrThrow("Notification block not created");

        // Counter shouldn't change, because the same user acted
        Assert.equals(initialBlock.count, block.count);

        // First user should become first in result counterUniques list
        ListF<String> realBlockActors = initialBlock.counterUniques.reverse();
        Assert.equals(realBlockActors, block.counterUniques);
    }

    @Test
    public void createNewNotificationForMetadataWithoutActor() {
        NotificationType type = NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS;

        NewNotificationData data = new NewNotificationData(
                uid, NotificationActor.consFromUid(123), Option.empty(),
                "group1", type, MetadataWrapper.createEmpty());
        notificationManager.createNewNotification(data);
        NotificationRecord record = notificationCacheDao.findNewNotifications(uid, type, data.groupKey).single();

        Assert.some("123", record.metadata.getEntityField(MetadataEntityNames.ACTOR, "uid"));
        Assert.some(MetadataEntityType.USER, record.metadata.getEntityType(MetadataEntityNames.ACTOR));
    }

    @Test
    public void createNewNotificationForMetadataWithLongComment() {
        NotificationType type = NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS;
        String group = "group1";

        NewNotificationData data = new NewNotificationData(
                uid, NotificationActor.consFromUid(123), Option.empty(),
                group, type, createMetadataWithLongComment());
        notificationManager.createNewNotification(data);

        notificationManager.updateOrCreateNotificationBlock(uid, DiskServices.DISK, type, group);
        NotificationBlockRecord block = notificationBlockDao
                .findNotificationBlock(uid, type.getFullName(), group)
                .getOrThrow("Notification block not created");

        String expectedComment = StringUtils.repeat("a", COMMENT_MAX_LENGTH - 3) + "...";
        Assert.some(expectedComment, block.metadata.getEntityField(MetadataEntityNames.COMMENT, "text"));
    }

    @Test
    public void createUpgradeRequiredUnlim() {
        NotificationType type = notificationRecordTypeManager.resolveRecordType(
                DiskNotifications.UPGRADE_REQUIRED_2017_UNLIM_IOS);

        MetadataWrapper metadata = NotificationPushManagerTest.createUpgradeRequiredUnlimMetadata("mobile_link", true);
        addGoToTunePageAction(metadata);
        addResourcePreviewMeta(metadata);

        NewNotificationData data = new NewNotificationData(
                uid, NotificationActor.YA_DISK, Option.empty(),
                "group1", type, metadata);
        notificationManager.createNewNotification(data);
        notificationManager.updateOrCreateNotificationBlock(uid, DiskServices.DISK, type, data.groupKey);

        NotificationBlockRecord block = notificationBlockDao.findNotificationBlock(uid, type.getFullName(), data.groupKey).get();
        Assert.equals(type, block.type);
        Assert.some("http://preview.ru/prev.jpg",
                block.metadata.getEntityField(MetadataEntityNames.ENTITY, MetadataEntityNames.PREVIEW_URL));
    }

    private void addGoToTunePageAction(MetadataWrapper metadata) {
        MetadataEntity actorEntity = new MetadataEntity(
                MetadataEntityType.ACTION, Cf.map("action", "GO_TO_TUNE_PAGE"));
        metadata.meta.put(MetadataEntityNames.ACTION, actorEntity);
    }

    private void addResourcePreviewMeta(MetadataWrapper metadata) {
        MetadataEntity entity = new MetadataEntity(
                MetadataEntityType.PREVIEW, Cf.map(MetadataEntityNames.PREVIEW_URL, "http://preview.ru/prev.jpg"));
        metadata.meta.put(MetadataEntityNames.ENTITY, entity);
    }

    private ListF<String> createNewNotificationsAndGetLastActors(
            Instant start, String group, long newCount)
    {
        ListF<NotificationActor> actors = Cf.list();
        for (long i = 0; i < newCount; i++) {
            Instant ctime = start.plus(Duration.standardMinutes(i));
            NotificationRecord record = NotificationTestUtils
                    .createTestNewNotificationRecord(group, ctime);
            NotificationActor lastActor = record.actor;
                    notificationCacheDao.createNewNotification(uid, record);
            actors = actors.plus1(lastActor);
        }
        return actors.map(actor -> actor.serialize()).reverse();
    }

    private MetadataWrapper createMetadataWithLongComment() {
        MetadataWrapper metadata = MetadataWrapper.createEmpty();
        String comment = StringUtils.repeat("a", 100);
        metadata.meta.put(
                MetadataEntityNames.COMMENT, new MetadataEntity(MetadataEntityType.COMMENT, Cf.map("text", comment)));
        metadata.meta.put(MetadataEntityNames.ENTITY, new MetadataEntity(MetadataEntityType.RESOURCE, Cf.map("id", "id")));
        return metadata;
    }

}
