package ru.yandex.chemodan.app.notifier.actions;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.convert.ConverterToDataApiUserId;
import ru.yandex.chemodan.app.notifier.NotificationManager;
import ru.yandex.chemodan.app.notifier.actions.convert.ConverterToNotificationActor;
import ru.yandex.chemodan.app.notifier.grants.ServiceGrantsChecker;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NewNotificationData;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.utils.NotifierUtils;
import ru.yandex.chemodan.util.exception.AccessForbiddenException;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.commune.a3.action.CloneableAction;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;

/**
 * @author akirakozov
 */
@Path(value = "/add-notification", methods = {HttpMethod.POST})
@RequiredArgsConstructor
public class AddNotificationAction implements CloneableAction {
    @RequestParam(value = "uid", customConverter = ConverterToDataApiUserId.class)
    private DataApiUserId uid;
    // TODO: enable binding source app and user from request
    // @BindWith(DataApiUserIdBinder.class)
    // private DataApiUserId user;
    // @BindWith(DataApiAppBinder.class)
    private Option<String> sourceApp = Option.empty();

    @RequestParam(value = "actor", customConverter = ConverterToNotificationActor.class)
    private NotificationActor actor;
    @RequestParam
    private String type;
    @RequestParam
    private Option<String> service;
    @RequestParam(value = "groupKey")
    private Option<String> groupKey;
    @RequestParam(value = "subscriptionKey")
    private Option<String> subscriptionKey;
    @RequestParam(value = "meta")
    private Option<String> meta;

    private final NotificationManager notificationManager;
    private final NotificationRecordTypeManager notificationRecordTypeManager;
    private final ServiceGrantsChecker serviceGrantsChecker;

    @Override
    public Object execute() throws Exception {
        validateParams();
        String realService = service.isPresent() ? service.get() : sourceApp.get();

        checkGrants(realService);

        MetadataWrapper metadata = meta.map(MetadataWrapper::fromJsonString).getOrElse(MetadataWrapper.createEmpty());
        NotificationType recordType = notificationRecordTypeManager.resolveRecordType(type, realService);

        metadata = recordType.defaultMetadata.mergeWith(metadata);

        NewNotificationData newNotificationData = new NewNotificationData(
                uid,
                actor,
                subscriptionKey,
                groupKey.getOrElse(NotifierUtils.generateRandomGroupKey()),
                recordType,
                metadata);

        notificationManager.createNewNotification(newNotificationData);

        return new OkPojo();
    }

    private void checkGrants(String realService) {
        // TODO: set real sourceApp
        if (!serviceGrantsChecker.checkGrants(realService, sourceApp)) {
            throw new AccessForbiddenException(
                    "Application " + sourceApp + " hasn't grants to add notification into service " + realService);
        }
    }

    private void validateParams() {
        if (!service.isPresent()) {
            throw new BadRequestException("Service is not specified");
        }
    }
}
