package ru.yandex.chemodan.app.notifier.actions;

import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.convert.ConverterToDataApiUserId;
import ru.yandex.chemodan.app.lentaloader.DynamicVars;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.push.NotificationPushInfo;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.worker.task.HolderForNotifierPushDeviceTask;
import ru.yandex.chemodan.bazinga.PgOnetimeUtils;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.commune.a3.action.CloneableAction;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.pg.storage.JobSaveResult;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author vpronto
 */
@Path(value = "/send-upgrade-notification", methods = {HttpMethod.POST})
@RequiredArgsConstructor
public class SendUpgradeNotificationPushAction implements CloneableAction {

    private final static Logger logger = LoggerFactory.getLogger(SendUpgradeNotificationPushAction.class);

    @RequestParam(value = "uid", customConverter = ConverterToDataApiUserId.class)
    private DataApiUserId uid;
    @RequestParam(value = "deviceId")
    private Option<String> deviceId;
    @RequestParam(value = "os")
    private Option<String> os;
    @RequestParam(value = "direct")
    private Option<Boolean> direct;

    private final NotificationPushManager pushManager;
    private final NotificationRecordTypeManager typeManager;
    private final PgBazingaStorage bazingaStorage;

    @Override
    public Object execute() {
        validateParams();
        if (direct.isSome(true) || filter()) {
            send(deviceId, uid, os);
        }
        return new OkPojo();
    }

    private boolean filter() {
        HolderForNotifierPushDeviceTask holder = new HolderForNotifierPushDeviceTask(uid, deviceId.get());
        OnetimeJob holderJob = PgOnetimeUtils.makeJob(holder, Instant.now().plus(DynamicVars.notifierPushDeviceDelay.get()));

        JobSaveResult scheduled = bazingaStorage.addOnetimeJobX(holderJob, ActiveUidDuplicateBehavior.DO_NOTHING);
        if (scheduled.isMerged()) {
            logger.info("Already sent to uid={}, device={}", uid, deviceId);
            return false;
        }
        return true;
    }

    public void send(Option<String> deviceId, DataApiUserId uid, Option<String> os) {
        logger.info("About to sent to uid={} with deviceId={}, os={}", uid, deviceId, os);

        if (os.isMatch(s -> GlobalSubscriptionChannel.IOS.value().equalsIgnoreCase(s))) {
            NotificationType unlimIos1 = typeManager.resolveRecordType(
                    DiskNotifications.UPGRADE_REQUIRED_2017_UNLIM_IOS);

            pushManager.pushNotification(uid, new NotificationPushInfo(unlimIos1.consMainTemplate(),
                            NotificationActor.YA_DISK,
                            Instant.now()),
                    unlimIos1.defaultMetadata, deviceId);

        } else {
            NotificationType unlimAndroid = typeManager.resolveRecordType(
                    DiskNotifications.UPGRADE_REQUIRED_2017_UNLIM_ANDROID);

            pushManager.pushNotification(uid, new NotificationPushInfo(unlimAndroid.consMainTemplate(),
                            NotificationActor.YA_DISK,
                            Instant.now()),
                    unlimAndroid.defaultMetadata, deviceId);
        }

    }

    private void validateParams() {
        if (!deviceId.isPresent()) {
            throw new BadRequestException("deviceId is not specified");
        }
        if (!os.isPresent()) {
            throw new BadRequestException("os is not specified");
        }
    }

}
