package ru.yandex.chemodan.app.notifier.admin.dao.mapper;

import java.sql.ResultSet;
import java.sql.SQLException;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.notifier.config.CounterConfiguration;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NotificationIcon;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.NotificationVariant;
import ru.yandex.commune.bazinga.pg.storage.dao.support.JsonSqlValue;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author akirakozov
 */
public class NotificationRecordTypeMarshaller implements RowMarshaller<NotificationType> {
    private static final BenderParserSerializer<CounterConfiguration> counterConfigBender =
            Bender.cons(CounterConfiguration.class);

    private static final BenderParserSerializer<NotificationVariant> variantBender =
            MetadataWrapper.mapper.createParserSerializer(NotificationVariant.class);

    @Override
    public NotificationType mapRow(ResultSet rs, int rowNum) throws SQLException {
        // TODO:
        // * icon should be nullable in db, get default for null
        return new NotificationType(
                rs.getInt("id"),
                rs.getInt("group_id"),
                rs.getString("name"),
                new NotificationIcon(rs.getString("icon")),
                rs.getString("tanker_msg_key"),
                rs.getString("tanker_title_msg_key"),
                rs.getString("tanker_short_msg_key"),
                Option.ofNullable(rs.getString("description")),
                parseAggregatorConfiguration(rs.getString("aggregator_conf")),
                parseDefaultMetaData(rs.getString("default_metadata")),
                Duration.millis(rs.getInt("delay")),
                parseAlternativeVariants(rs.getString("alternative_variants")));
    }

    @Override
    public Tuple2List<String, Object> asColumnsValues(NotificationType type) {
        Tuple2List<String, Object> res = Tuple2List.arrayList();

        type.id.forEach(id -> res.add("id", id));
        type.groupId.forEach(id -> res.add("group_id", id));
        res.add("name", type.name);
        res.add("icon", type.icon.name);
        res.add("tanker_msg_key", type.tankerMessageKey);
        res.add("tanker_title_msg_key", type.tankerTitleMessageKey);
        res.add("tanker_short_msg_key", type.tankerShortMessageKey);
        res.add("description", type.description.getOrElse(""));
        res.add("aggregator_conf", serializeAggregatorConfig(type.counterConfiguration));
        res.add("delay", type.defaultDelay.getMillis());
        res.add("default_metadata", type.defaultMetadata.toJsonString());
        res.add("alternative_variants", serializeAlternativeVariants(type.alternativeVariants));

        return res;
    }

    private String serializeAggregatorConfig(CounterConfiguration config) {
        return new String(counterConfigBender.getSerializer().serializeJson(config));
    }

    private CounterConfiguration parseAggregatorConfiguration(String counterConf) {
        return counterConf == null ?
               CounterConfiguration.NONE :
               counterConfigBender.getParser().parseJson(counterConf);
    }

    private MetadataWrapper parseDefaultMetaData(String defaultMetadata) {
        return StringUtils.isEmpty(defaultMetadata) ?
               MetadataWrapper.createEmpty() :
               MetadataWrapper.fromJsonString(defaultMetadata);
    }

    private JsonSqlValue serializeAlternativeVariants(ListF<NotificationVariant> variants) {
        return variants.isEmpty()
                ? new JsonSqlValue(null)
                : new JsonSqlValue(new String(variantBender.getSerializer().serializeListJson(variants)));
    }

    private ListF<NotificationVariant> parseAlternativeVariants(String variants) {
        return StringUtils.isEmpty(variants)
                ? Cf.list()
                : variantBender.getParser().parseListJson(variants);
    }
}
