package ru.yandex.chemodan.app.notifier.admin.web;

import java.util.Comparator;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.DataNamesUtils;
import ru.yandex.chemodan.app.notifier.admin.NotifierTypesRegistry;
import ru.yandex.chemodan.app.notifier.config.CounterConfiguration;
import ru.yandex.chemodan.app.notifier.config.CounterType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.NotificationGroup;
import ru.yandex.chemodan.app.notifier.notification.NotificationIcon;
import ru.yandex.chemodan.app.notifier.notification.NotificationService;
import ru.yandex.chemodan.app.notifier.notification.NotificationServiceWidgetSettings;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.NotificationVariant;
import ru.yandex.chemodan.app.notifier.settings.NotifierWidgetSettingsManager;
import ru.yandex.chemodan.app.notifier.tanker.TankerRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.EmptyContentPojo;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author akirakozov
 */
@ActionContainer
@RequiredArgsConstructor
public class NotifierTypesAdminPage {

    private final NotifierTypesRegistry typesRegistry;
    private final TankerRegistry tankerRegistry;
    private final NotifierWidgetSettingsManager notifierWidgetSettingsManager;

    @ZAction(defaultAction = true, file = "NotifierTypesAdminPage-index.xsl")
    @Path("/notifier-types")
    public Object index() {
        return new NotifierServicesPojo(typesRegistry.getAllServices());
    }

    @ZAction(file = "NotifierTypesAdminPage-service-contents.xsl")
    @Path("/service-notifier-types")
    public Object getAllServiceInfo(@RequestParam("serviceId") int serviceId) {
        NotificationService service = typesRegistry.getServiceById(serviceId);
        ListF<NotificationGroup> groups = typesRegistry.getGroupsByServiceId(serviceId);
        MapF<Integer, NotificationGroup> groupsMap = groups.toMapMappingToKey(g -> g.id.get());
        ListF<NotificationType> types = typesRegistry
                .getTypesByServiceId(serviceId)
                .map(t -> t.withGroup(groupsMap.getTs(t.groupId.get())))
                .sorted(Comparator.comparing(t -> t.name))
                .sorted(Comparator.comparing(t -> t.getGroup().name));

        return new NotifierTypesPojo(service, groups, types);
    }

    @ZAction(file = "NotifierTypesAdminPage-service-info.xsl")
    @Path("/notifier-types/add-service")
    public Object addUpdateServicePage(
            @RequestParam("serviceId") Option<Integer> serviceId,
            @RequestParam("newServiceName") Option<String> newServiceName)
    {
        Validate.isTrue(serviceId.isPresent() || newServiceName.filter(StringUtils::isNotEmpty).isPresent(),
                "Either service id or service name must be specified");
        Option<NotificationService> serviceO = serviceId.map(typesRegistry::getServiceById);
        MapF<NotifierLanguage, String> names =
                serviceO.map(notifierWidgetSettingsManager::getServiceNames).getOrElse(getDefaultLanguageMap());

        NotificationService service = serviceO.getOrElse(
                () -> new NotificationService(Option.empty(), newServiceName.get(), newServiceName.get(), false));

        return new NotifierServicePojo(service, names, notifierWidgetSettingsManager.getServiceWidgetSettings(service));
    }

    @Path(value = "/notifier-types/add-service", methods = HttpMethod.POST)
    public EmptyContentPojo addService(
            @RequestParam(value = "serviceId", ignoreEmpty = true) Option<Integer> serviceId,
            @RequestParam("name") String name,
            @RequestParam("tankerKeysetId") String tankerKeysetId,
            @RequestParam("isYaTeam") Option<Boolean> isYaTeam,
            @RequestParam("englishName") String englishName,
            @RequestParam("russianName") String russianName,
            @RequestParam("ukrainianName") String ukrainianName,
            @RequestParam("turkishName") String turkishName,
            @RequestParam("belarusianName") String belarusianName,
            @RequestParam("kazakhName") String kazakhName,
            @RequestParam("uzbekName") String uzbekName,
            @RequestParam("blockCollection") String blockCollection,
            @RequestParam("iconSrc") String iconSrc,
            @RequestParam("iconType") String iconType,
            @RequestParam("messageCollection") String messageCollection,
            @RequestParam("settingsCollection") String settingsCollection,
            @RequestParam("unviewedCollection") String unviewedCollection,
            @RequestParam("unviewedRecord") String unviewedRecord)
    {
        Validate.isTrue(StringUtils.isNotEmpty(name), "Service name must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(tankerKeysetId), "Tanker keyset id must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(iconSrc), "Icon src must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(englishName), "Language-specific name must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(russianName), "Language-specific name must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(ukrainianName), "Language-specific name must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(turkishName), "Language-specific name must be specified");

        NotificationService service = new NotificationService(serviceId, name, tankerKeysetId, isYaTeam.getOrElse(false));
        typesRegistry.saveService(service);

        MapF<NotifierLanguage, String> names = Tuple2List.<NotifierLanguage, String>fromPairs(
                NotifierLanguage.ENGLISH, englishName,
                NotifierLanguage.RUSSIAN, russianName,
                NotifierLanguage.UKRAINIAN, ukrainianName,
                NotifierLanguage.TURKISH, turkishName,
                NotifierLanguage.BELARUSIAN, belarusianName,
                NotifierLanguage.KAZAKH, kazakhName,
                NotifierLanguage.UZBEK, uzbekName)
                .toMap();
        notifierWidgetSettingsManager.saveServiceNames(service, names);

        NotificationServiceWidgetSettings widgetSettings =
                new NotificationServiceWidgetSettings(blockCollection, iconSrc, iconType, messageCollection,
                        settingsCollection, unviewedCollection, unviewedRecord);
        notifierWidgetSettingsManager.saveServiceWidgetSettings(service, widgetSettings);

        throw new ZRedirectException("/z/notifier-types");
    }

    @ZAction(file = "NotifierTypesAdminPage-group.xsl")
    @Path("/notifier-types/add-group")
    public Object addUpdateGroupPage(
            @RequestParam("serviceId") int serviceId,
            @RequestParam("groupId") Option<Integer> groupId) {
        NotificationService service = typesRegistry.getServiceById(serviceId);
        Option<NotificationGroup> groupO = groupId.map(typesRegistry::getGroupById).map(group -> group.withService(service));
        MapF<NotifierLanguage, String> names =
                groupO.map(notifierWidgetSettingsManager::getGroupNames).getOrElse(getDefaultLanguageMap());

        return new NotifierGroupPojo(service, groupO, names);
    }

    @Path(value = "/notifier-types/add-group", methods = HttpMethod.POST)
    public Object addGroup(
            @RequestParam(value = "groupId", ignoreEmpty = true) Option<Integer> groupId,
            @RequestParam("name") String name,
            @RequestParam("serviceId") int serviceId,
            @RequestParam("description") String description,
            @RequestParam("settingsGroupName") String settingsGroupName,
            @RequestParam("englishName") String englishName,
            @RequestParam("russianName") String russianName,
            @RequestParam("ukrainianName") String ukrainianName,
            @RequestParam("turkishName") String turkishName,
            @RequestParam("belarusianName") String belarusianName,
            @RequestParam("kazakhName") String kazakhName,
            @RequestParam("uzbekName") String uzbekName)
    {
        Validate.isTrue(StringUtils.isNotEmpty(name), "Group name must be specified");
        Validate.isTrue(DataNamesUtils.isValidId(name), "Invalid group name");
        Validate.isTrue(StringUtils.isNotEmpty(englishName), "Language-specific name must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(russianName), "Language-specific name must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(ukrainianName), "Language-specific name must be specified");
        Validate.isTrue(StringUtils.isNotEmpty(turkishName), "Language-specific name must be specified");

        typesRegistry.saveGroup(new NotificationGroup(groupId, serviceId, name, StringUtils.notBlankO(description),
                StringUtils.notBlankO(settingsGroupName)));

        MapF<NotifierLanguage, String> names = Tuple2List.<NotifierLanguage, String>fromPairs(
                NotifierLanguage.ENGLISH, englishName,
                NotifierLanguage.RUSSIAN, russianName,
                NotifierLanguage.UKRAINIAN, ukrainianName,
                NotifierLanguage.TURKISH, turkishName,
                NotifierLanguage.BELARUSIAN, belarusianName,
                NotifierLanguage.KAZAKH, kazakhName,
                NotifierLanguage.UZBEK, uzbekName)
                .toMap();

        notifierWidgetSettingsManager.saveGroupNames(name, typesRegistry.getServiceById(serviceId), names);

        throw new ZRedirectException("/z/service-notifier-types?serviceId=" + serviceId);
    }

    @ZAction(file = "NotifierTypesAdminPage-type.xsl")
    @Path("/notifier-types/add-type")
    public Object addUpdateTypePage(
            @RequestParam("serviceId") int serviceId,
            @RequestParam("typeId") Option<Integer> typeId)
    {
        NotificationService service = typesRegistry.getServiceById(serviceId);
        ListF<NotificationGroup> groups = typesRegistry.getGroupsByServiceId(serviceId);
        Option<NotificationType> typeO = typeId.map(typesRegistry::getTypeById);

        return new NotifierTypePojo(service, groups, typeO,
                typeO.map(type -> type.defaultDelay.getMillis()),
                typeO.map(type -> type.defaultMetadata.toJsonString()),
                typeO.flatMap(type -> type.alternativeVariants.map(
                        v -> v.getDefaultMeta().map(MetadataWrapper::toJsonString).getOrElse(""))));
    }

    @Path(value = "/notifier-types/add-type", methods = HttpMethod.POST)
    public Object addUpdateType(
            @RequestParam("serviceId") int serviceId,
            @RequestParam(value = "typeId", ignoreEmpty = true) Option<Integer> typeId,
            @RequestParam("name") String name,
            @RequestParam("icon") String icon,
            @RequestParam("groupId") int groupId,
            @RequestParam("tankerMessageKey") String tankerMessageKey,
            @RequestParam("tankerShortMessageKey") String tankerShortMessageKey,
            @RequestParam("tankerTitleMessageKey") String tankerTitleMessageKey,
            @RequestParam("delay") long delay,
            @RequestParam("description") String description,
            @RequestParam("aggregatorType") String aggregatorType,
            @RequestParam("aggregatorParams") String aggregatorParams,
            @RequestParam("defaultMetadata") String defaultMetadata,
            @RequestParam(value = "variants.name", required = false) ListF<String> variantsName,
            @RequestParam(value = "variants.users", required = false) ListF<Double> variantsUsers,
            @RequestParam(value = "variants.offset", required = false) ListF<Double> variantsOffset,
            @RequestParam(value = "variants.icon", required = false) ListF<String> variantsIcon,
            @RequestParam(value = "variants.message", required = false) ListF<String> variantsMessage,
            @RequestParam(value = "variants.title", required = false) ListF<String> variantsTitle,
            @RequestParam(value = "variants.short", required = false) ListF<String> variantsShort,
            @RequestParam(value = "variants.meta", required = false) ListF<String> variantsMeta,
            @RequestParam(value = "variants.delay", required = false) ListF<Long> variantsDelay)
    {
        IteratorF<String> icons = variantsIcon.iterator(),
                messages = variantsMessage.iterator(), titles = variantsTitle.iterator(),
                shorts = variantsShort.iterator(), metas = variantsMeta.iterator();

        IteratorF<Double> offsets = variantsOffset.iterator(), users = variantsUsers.iterator();
        IteratorF<Long> delays = variantsDelay.iterator();

        typesRegistry.saveType(new NotificationType(
                typeId, Option.of(groupId), name, Option.empty(), new NotificationIcon(icon),
                tankerMessageKey, tankerTitleMessageKey, tankerShortMessageKey,
                StringUtils.notBlankO(description),
                new CounterConfiguration(
                        CounterType.R.valueOf(aggregatorType),
                        parseAggregatorParams(aggregatorParams)),
                parseDefaultMetaData(defaultMetadata),
                delay <= 0 ? Duration.ZERO : Duration.millis(delay),
                variantsName.map(n -> new NotificationVariant(
                        StringUtils.notBlankO(n).getOrThrow("Variant name is required"),
                        users.next().floatValue(), offsets.next().floatValue(),
                        StringUtils.notBlankO(icons.next()), StringUtils.notBlankO(messages.next()),
                        StringUtils.notBlankO(titles.next()), StringUtils.notBlankO(shorts.next()),
                        StringUtils.notBlankO(metas.next()).map(MetadataWrapper::fromJsonString),
                        Option.ofNullable(Duration.millis(delays.next()))))
            ));

        throw new ZRedirectException("/z/service-notifier-types?serviceId=" + serviceId);
    }


    @Path(value = "/notifier-types/update-tanker-keys", methods = HttpMethod.POST)
    public EmptyContentPojo updateTankerKeys() {
        tankerRegistry.refreshCachedState();
        throw new ZRedirectException("/z/notifier-types");
    }

    private MapF<NotifierLanguage, String> getDefaultLanguageMap() {
        return Cf.x(NotifierLanguage.values()).toMapMappingToValue(l -> "");
    }

    @Bendable
    @XmlRootElement(name = "content")
    @BenderBindAllFields
    private static final class NotifierServicesPojo {
        @BenderPart(name = "service", wrapperName = "services")
        private final ListF<NotificationService> services;

        private NotifierServicesPojo() {
            this(Cf.list());
        }

        private NotifierServicesPojo(ListF<NotificationService> services) {
            this.services = services;
        }
    }

    @Bendable
    @XmlRootElement(name = "content")
    @BenderBindAllFields
    private static final class NotifierTypesPojo {
        private final NotificationService service;
        @BenderPart(name = "group", wrapperName = "groups")
        private final ListF<NotificationGroup> groups;
        @BenderPart(name = "type", wrapperName = "types")
        private final ListF<NotificationType> types;

        private NotifierTypesPojo() {
            this(null, Cf.list(), Cf.list());
        }

        private NotifierTypesPojo(NotificationService service, ListF<NotificationGroup> groups,
                ListF<NotificationType> types)
        {
            this.service = service;
            this.groups = groups;
            this.types = types;
        }
    }

    @Data
    @Bendable
    @XmlRootElement(name = "content")
    @BenderBindAllFields
    private static final class NotifierTypePojo {
        private final NotificationService service;
        @BenderPart(name = "group", wrapperName = "groups")
        private final ListF<NotificationGroup> groups;
        @BenderPart(name = "type", wrapperName = "aggregatorTypes")
        private final ListF<CounterType> aggregatorTypes =
                Cf.list(CounterType.NONE, CounterType.UNIQUE_BUT_ONE, CounterType.UNIQUE);
        private final Option<NotificationType> type;
        private final Option<Long> delay;
        private final Option<String> defaultMetadata;
        @BenderPart(name = "variant", wrapperName = "alternativeMetadata")
        private final ListF<String> alternativeMetadata;
    }

    @Data
    @Bendable
    @XmlRootElement(name = "content")
    @BenderBindAllFields
    private static final class NotifierServicePojo {
        private final NotificationService service;
        private final MapF<NotifierLanguage, String> names;
        private final NotificationServiceWidgetSettings widgetSettings;
    }

    @Data
    @Bendable
    @XmlRootElement(name = "content")
    @BenderBindAllFields
    private static final class NotifierGroupPojo {
        private final NotificationService service;
        private final Option<NotificationGroup> group;
        private final MapF<NotifierLanguage, String> names;
    }

    private MetadataWrapper parseDefaultMetaData(String defaultMetadata) {
        return StringUtils.isEmpty(defaultMetadata) ?
               MetadataWrapper.createEmpty() :
               MetadataWrapper.fromJsonString(defaultMetadata);
    }

    private ListF<String> parseAggregatorParams(String aggregatorParams) {
        return Cf.list(StringUtils.split(aggregatorParams.replaceAll("\\s+", ""), ","));
    }

}
