package ru.yandex.chemodan.app.notifier.dao;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.ConvertingDataColumn;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordIdColumn;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByDataRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaUpdateOrDeleteNonExistentRecordException;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockUpdate;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.notifier.NotifierDatabasing;
import ru.yandex.chemodan.notifier.NotifierUnreadCountProvider;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
@AllArgsConstructor
public class NotificationBlockDaoImpl implements NotificationBlockDao {
    private static final Logger logger = LoggerFactory.getLogger(NotificationBlockDaoImpl.class);

    private static final String BLOCKS_COLLECTION = "blocks";

    private static final String LANG_COLLECTION_PREFIX = "message_";

    private static final String UNVIEWED_COUNT_FIELD = NotifierDatabasing.UNREAD_COUNT_FIELD;

    private final DataApiManager dataApiManager;
    private final NotificationRecordTypeManager notificationRecordTypeManager;
    private final NotifierUnreadCountProvider notifierUnreadCountProvider;

    private final int maxBlocksCount;

    @Override
    public Option<NotificationBlockRecord> findNotificationBlock(
            DataApiUserId userId, ServiceAndType type, String group)
    {
        Database db = getOrCreateDatabase(userId);

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(CollectionIdCondition.eq(getBlocksCollection(type)))
                .withDataCond(createTypeAndGroupKeyCondition(type, group))
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_DESC)
                .withLimits(SqlLimits.first(1));

        return dataApiManager.getRecords(db.spec(), filter)
                .map(rec -> NotificationBlockRecord.fromDataRecord(rec, type.getService(), notificationRecordTypeManager))
                .singleO();
    }

    @Override
    public ListF<NotificationBlockRecord> findNotificationBlocks(
            DataApiUserId userId, ListF<ServiceAndType> types, String group)
    {
        if (types.isEmpty()) {
            return Cf.list();
        }

        Database db = getOrCreateDatabase(userId);

        // All types should be from on service
        ServiceAndType type = types.first();
        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(CollectionIdCondition.eq(getBlocksCollection(type)))
                .withDataCond(createTypeAndGroupKeyCondition(types, group))
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_DESC);

        return dataApiManager.getRecords(db.spec(), filter)
                .map(rec -> NotificationBlockRecord.fromDataRecord(
                        rec, type.getService(), notificationRecordTypeManager));
    }

    @Override
    public void updateNotificationBlock(
            DataApiUserId userId, String service, String recordId,
            NotificationBlockUpdate update,
            MapF<NotifierLanguage, LocalizedMessage> messages)
    {
        Database db = getOrCreateDatabase(userId);

        ListF<FieldChange> fieldChanges = Cf.arrayList();

        update.newRecordType.forEach(f -> fieldChanges.add(NotificationBlockRecord.Fields.TYPE.changedTo(f.value())));

        update.counterUniques.forEach(cU -> {
            String uniquesAsString = NotificationBlockRecord.uniquesToString(cU);
            fieldChanges.add(NotificationBlockRecord.Fields.COUNTER_UNIQUES.changedTo(uniquesAsString));
        });

        update.count.forEach(f -> fieldChanges.add(NotificationBlockRecord.Fields.COUNT.changedTo(f)));
        update.metadata.forEach(f -> fieldChanges.add(NotificationBlockRecord.Fields.META.changedTo(f)));
        update.mtime.forEach(f -> fieldChanges.add(NotificationBlockRecord.Fields.MTIME.changedTo(f)));
        update.isViewed.forEach(f -> fieldChanges.add(NotificationBlockRecord.Fields.IS_READ.changedTo(f)));

        ListF<RecordChange> changes = Cf.list(RecordChange.update(getBlocksCollection(service), recordId, fieldChanges));

        changes = addLocalizationChanges(changes, recordId, service, messages);

        if (update.isViewed.isPresent()) {
            int unviewedBlocksCount = getUnviewedBlocksCount(db, service);

            Lazy<DataRecord> record = Lazy.withSupplier(() -> findBlockRecord(db, service, recordId));

            boolean originalBlockViewed = unviewedBlocksCount <= 0
                    || NotificationBlockRecord.Fields.IS_READ.get(record.get())
                    || countUnreadBlocksNewerThan(db, service, record.get()) >= unviewedBlocksCount;

            if (originalBlockViewed && !update.isViewed.get()) {
                changes = addIncUnviewedCountChange(service, unviewedBlocksCount, changes);
            } else if (!originalBlockViewed && update.isViewed.get()) {
                changes = addDecUnviewedCountChange(service, unviewedBlocksCount, changes);
            }
        }

        dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    @Override
    public void createNewNotificationBlockAndSiftOldest(
            DataApiUserId userId,
            NotificationBlockRecord record,
            MapF<NotifierLanguage, LocalizedMessage> messages)
    {
        Database db = getOrCreateDatabase(userId);

        int unviewedBlocksCount = getUnviewedBlocksCount(db, record.type.getService().name);

        ListF<RecordChange> changes = Cf.list(RecordChange.insert(
                getBlocksCollection(record.type), record.id, record.toData()));
        changes = addLocalizationChanges(changes, record.id, record.type.getService().name, messages);
        if (!record.isRead) {
            changes = addIncUnviewedCountChange(record.type.getService().name, unviewedBlocksCount, changes);
        }

        dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, new Delta(changes));

        deleteOldBlocks(db, record.type);
    }

    @Override
    public void markNotificationBlockAsRead(DataApiUserId userId,
        String groupKey, NotificationType recordType)
    {
        Database db = getOrCreateDatabase(userId);

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionId(getBlocksCollection(recordType.getFullName()))
                .withDataCond(NotificationBlockRecord.Fields.GROUP_KEY.column().eq(groupKey)
                        .and(NotificationBlockRecord.Fields.TYPE.column().eq(recordType.value())));

        Option<DataRecord> unreadRecordO = dataApiManager.getRecords(db.spec(), filter).firstO();
        if (unreadRecordO.isPresent()) {
            DataRecord unreadRecord = unreadRecordO.get();
            int unviewedBlocksCount = getUnviewedBlocksCount(db, recordType.getService().name);

            boolean blockIsUnviewed = unviewedBlocksCount > 0
                    && !NotificationBlockRecord.Fields.IS_READ.get(unreadRecord)
                    && countUnreadBlocksNewerThan(db, recordType.getService().name, unreadRecord) < unviewedBlocksCount;

            Tuple2<String, DataField> isReadField = NotificationBlockRecord.Fields.IS_READ.toData(true);
            ListF<RecordChange> changes = Cf.list(RecordChange.update(getBlocksCollection(recordType), unreadRecord.getRecordId(),
                            FieldChange.put(isReadField.get1(), isReadField.get2())));

            if (blockIsUnviewed) {
                changes = addDecUnviewedCountChange(recordType.getService().name, unviewedBlocksCount, changes);
            }
            dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, new Delta(changes));
        }
    }

    @Override
    public int getUnviewedBlocksCountForService(DataApiUserId userId, String service) {
        return notifierUnreadCountProvider.getForService(userId, service);
    }

    @Override
    public int getTotalUnviewedBlocksCountForEnabledServices(DataApiUserId userId) {
        return notifierUnreadCountProvider.getForEnabledServices(userId);
    }

    @Override
    public void deleteBlock(DataApiUserId userId, NotificationBlockRecord block) {
        Database db = getOrCreateDatabase(userId);

        ListF<RecordChange> changes = getDeleteChanges(block.type.getService().name, block.id);
        if (!block.isRead) {
            int unviewedBlocksCount = getUnviewedBlocksCount(db, block.type.getService().name);
            changes = addDecUnviewedCountChange(block.type.getService().name, unviewedBlocksCount, changes);
        }
        applyDeleteChanges(db, changes);
    }

    private void deleteOldBlocks(Database db, NotificationType type) {
        int excessBlockCount = findNotificationBlocksCount(db, type) - maxBlocksCount;
        ListF<RecordChange> changes = findOldestBlockIds(db, type.getService().name, excessBlockCount)
                .flatMap(id -> getDeleteChanges(type.getService().name, id));

        if (!changes.isEmpty()) {
            applyDeleteChanges(db, changes);
        }
    }

    private ListF<RecordChange> getDeleteChanges(String service, String blockId) {
        ListF<RecordChange> changes = Cf.list(RecordChange.delete(getBlocksCollection(service), blockId));

        for (NotifierLanguage lang : NotifierLanguage.values()) {
            changes = changes.plus1(RecordChange.delete(getLanguageCollectionId(service, lang), blockId));
        }

        return changes;
    }

    private void applyDeleteChanges(Database db, ListF<RecordChange> changes) {
        try {
            dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, new Delta(changes));
        } catch (OutdatedChangeException | DeltaUpdateOrDeleteNonExistentRecordException e) {
            // it's not a big deal if block was not deleted, we should just clean up everything separately
            logger.debug("Can't apply delete changes for uid={} as single delta, cleaning up", db.uid, e);
            changes.forEach(change -> {
                try {
                    dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD, new Delta(change));
                } catch (OutdatedChangeException | DeltaUpdateOrDeleteNonExistentRecordException ignored) {
                    // ignore already empty records during cleanup
                }
            });
        }
    }

    private ListF<RecordChange> addIncUnviewedCountChange(
            String service, int unviewedBlocksCount, ListF<RecordChange> changes)
    {
        if (unviewedBlocksCount < maxBlocksCount) {
            MapF<String, DataField> data = Cf.map(UNVIEWED_COUNT_FIELD, DataField.integer(unviewedBlocksCount + 1));
            changes = changes.plus1(RecordChange.set(NotifierDatabasing.metaRecordIdFor(service), data));
        }
        return changes;
    }

    private ListF<RecordChange> addDecUnviewedCountChange(
            String service, int unviewedBlocksCount, ListF<RecordChange> changes)
    {
        if (unviewedBlocksCount > 0) {
            MapF<String, DataField> data = Cf.map(UNVIEWED_COUNT_FIELD, DataField.integer(unviewedBlocksCount - 1));
            changes = changes.plus1(RecordChange.set(NotifierDatabasing.metaRecordIdFor(service), data));
        }
        return changes;
    }

    private int findNotificationBlocksCount(Database db, NotificationType type) {
        RecordsFilter filter = RecordsFilter.DEFAULT.withCollectionId(getBlocksCollection(type));
        return dataApiManager.getRecordsCount(db.spec(), filter);
    }

    private DataRecord findBlockRecord(Database db, String service, String recordId) {
        return dataApiManager.getRecord(db.spec(), new SimpleRecordId(getBlocksCollection(service), recordId))
                .getOrThrow("Record not found for service " + service + " and id " + recordId);
    }

    private int countUnreadBlocksNewerThan(Database db, String service, DataRecord boundRecord) {
        ConvertingDataColumn<Boolean> isReadField = NotificationBlockRecord.Fields.IS_READ.column();
        ConvertingDataColumn<Instant> mtimeField = NotificationBlockRecord.Fields.MTIME.column();

        Instant mtime = NotificationBlockRecord.Fields.MTIME.get(boundRecord);
        RecordCondition recordIdGreater = RecordIdColumn.C.gt(boundRecord.getRecordId());

        RecordCondition condition = RecordCondition.all()
                .and(isReadField.eq(false))
                .and(mtimeField.gt(mtime).or(mtimeField.eq(mtime).and(recordIdGreater)));

        return dataApiManager.getRecordsCount(db.spec(), RecordsFilter.DEFAULT
                .withCollectionId(getBlocksCollection(service))
                .withRecordCond(condition));
    }

    private ListF<String> findOldestBlockIds(Database db, String service, int limit) {
        if (limit <= 0) {
            return Cf.list();
        }

        ByDataRecordOrder order = NotificationBlockRecord.Fields.MTIME.column().orderBy();

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(CollectionIdCondition.eq(getBlocksCollection(service)))
                .withRecordOrder(order)
                .withLimits(SqlLimits.first(limit));

        return dataApiManager.getRecords(db.spec(), filter).map(DataRecord::getRecordId);
    }

    private DataCondition createTypeAndGroupKeyCondition(ServiceAndType type, String group) {
        return DataCondition.all()
                .and(NotificationRecord.Fields.TYPE.column().eq(type.getType()))
                .and(NotificationRecord.Fields.GROUP_KEY.column().eq(group));
    }

    private DataCondition createTypeAndGroupKeyCondition(ListF<ServiceAndType> types, String group) {
        DataCondition cond = DataCondition.all()
                .and(NotificationRecord.Fields.GROUP_KEY.column().eq(group));

        DataCondition typesCondition = DataCondition.none().or(types.map(
                type -> NotificationRecord.Fields.TYPE.column().eq(type.getType())));

        return cond.and(typesCondition);
    }

    private Database getOrCreateDatabase(DataApiUserId userId) {
        return dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(userId, NotifierDatabasing.NOTIFICATIONS_DB));
    }

    private int getUnviewedBlocksCount(Database db, String service) {
        return notifierUnreadCountProvider.getForService(db, service);
    }

    private ListF<RecordChange> addLocalizationChanges(ListF<RecordChange> changes,
            String recordId, String service, MapF<NotifierLanguage, LocalizedMessage> messages)
    {
        return changes.plus(messages.mapEntries((lang, message) ->
                        RecordChange.set(getLanguageCollectionId(service, lang), recordId, Cf.map(
                                LocalizedMessage.Fields.MESSAGE.name, DataField.string(message.message),
                                LocalizedMessage.Fields.META.name, DataField.string(message.meta.toJsonString())))
        ));
    }

    private String getBlocksCollection(NotificationType type) {
        return getBlocksCollection(type.getFullName());
    }

    private String getBlocksCollection(ServiceAndType type) {
        return getBlocksCollection(type.getService());
    }

    private String getBlocksCollection(String serviceName) {
        return getPrefixByService(serviceName) + BLOCKS_COLLECTION;
    }

    private String getLanguageCollectionId(String service, NotifierLanguage lang) {
        return getPrefixByService(service) + LANG_COLLECTION_PREFIX + lang.value();
    }

    private String getPrefixByService(String serviceName) {
        return serviceName.equals(DiskServices.DISK) ? "" : serviceName + "_";
    }

    protected ListF<LocalizedMessage> getLocalizedMessages(
            DataApiUserId userId, String service, NotifierLanguage lang, String id)
    {
        Database db = getOrCreateDatabase(userId);

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(CollectionIdCondition.eq(getLanguageCollectionId(service, lang)))
                .withRecordId(id);

        return dataApiManager.getRecords(db.spec(), filter)
                .map(LocalizedMessage::fromDataRecord);
    }

    protected void resetUnviewedBlocksCount(DataApiUserId userId, String service) {
        MapF<String, DataField> data = Cf.map(UNVIEWED_COUNT_FIELD, DataField.integer(0));
        RecordChange change = RecordChange.set(NotifierDatabasing.metaRecordIdFor(service), data);
        dataApiManager.applyDelta(getOrCreateDatabase(userId), RevisionCheckMode.PER_RECORD, new Delta(change));
    }
}
