package ru.yandex.chemodan.app.notifier.dao;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.notifier.NotificationTestUtils;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.config.CounterConfiguration;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockUpdate;
import ru.yandex.chemodan.app.notifier.notification.NotificationGroup;
import ru.yandex.chemodan.app.notifier.notification.NotificationIcon;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationService;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.notifier.NotifierEnabledServicesProvider;
import ru.yandex.chemodan.notifier.NotifierUnreadCountProvider;
import ru.yandex.misc.test.Assert;

/**
 * @author akirakozov
 */
@ActivateDataApiEmbeddedPg
@ActivateNotificationEmbeddedPg
@ContextConfiguration(classes = NotificationTypeManagerTestContextConfiguration.class)
public class NotificationBlockDaoTest extends DataApiTestSupport {
    // custom max block count used to test auto-delete logic
    public static final int BLOCKS_LIMIT = 2;

    public static final MapF<NotifierLanguage, LocalizedMessage> LOCALIZED_MESSAGES =
            Cf.map(NotifierLanguage.RUSSIAN, new LocalizedMessage("Russian message"),
                    NotifierLanguage.ENGLISH, new LocalizedMessage("English message"));
    public static final String DISK_SERVICE = DiskServices.DISK;

    @Autowired
    private DataApiManager dataApiManager;
    @Autowired
    private NotificationRecordTypeManager notificationRecordTypeManager;
    @Mock
    private NotifierEnabledServicesProvider notifierEnabledServicesProvider;

    private DataApiUserId uid;
    private NotifierUnreadCountProvider unreadCountProvider;
    private NotificationBlockDaoImpl notificationBlockDao;

    @Before
    public void setup() {
        uid = createRandomCleanUserInDefaultShard();

        MockitoAnnotations.initMocks(this);

        Mockito.when(notifierEnabledServicesProvider.getEnabledServiceNames()).thenReturn(Cf.list(DISK_SERVICE));
        Mockito.when(notifierEnabledServicesProvider.getEnabledYaTeamServiceNames()).thenReturn(Cf.list(DISK_SERVICE));

        unreadCountProvider = new NotifierUnreadCountProvider(dataApiManager, notifierEnabledServicesProvider);

        notificationBlockDao = new NotificationBlockDaoImpl(
                dataApiManager, notificationRecordTypeManager, unreadCountProvider, BLOCKS_LIMIT);
    }

    @Test
    public void createFindNotificationBlock() {
        NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord("group_key1",
                Option.empty(), notificationRecordTypeManager.resolveRecordType(DiskNotifications.COMMENT_LIKE));

        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, Cf.hashMap());
        Option<NotificationBlockRecord> blockRecord = notificationBlockDao.findNotificationBlock(
                uid, record.type.getFullName(), record.groupKey);

        Assert.some(blockRecord);
        Assert.equals(record, blockRecord.get(), "Notification record block was stored or loaded incorrectly");
    }

    @Test
    public void createFindUpdateBlock() {
        NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord("group_key1",
                Option.empty(), notificationRecordTypeManager.resolveRecordType(DiskNotifications.COMMENT_LIKE));

        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, Cf.hashMap());
        Option<NotificationBlockRecord> blockRecord = notificationBlockDao.findNotificationBlock(
                uid, record.type.getFullName(), record.groupKey);

        Assert.some(blockRecord);
        Assert.equals(record, blockRecord.get(), "Notification record block was stored or loaded incorrectly");

        NotificationActor actor = NotificationTestUtils.createRandomActor();
        NotificationBlockUpdate update = new NotificationBlockUpdate(
                Cf.list(actor.toString()), 10, "{\"meta\": {}}", Instant.now(), false);

        notificationBlockDao.updateNotificationBlock(
                uid,
                DISK_SERVICE,
                blockRecord.get().id,
                update,
                LOCALIZED_MESSAGES);

        blockRecord = notificationBlockDao.findNotificationBlock(
                uid, record.type.getFullName(), record.groupKey);

        Assert.some(blockRecord);
        Assert.some(update.count.get(), blockRecord.get().count);
        Assert.equals(update.mtime.get(), blockRecord.get().mtime);
        Assert.equals(update.counterUniques.get(), blockRecord.get().counterUniques);

        for (NotifierLanguage lang : LOCALIZED_MESSAGES.keySet()) {
            ListF<LocalizedMessage> localizedMessagesRu =
                    notificationBlockDao.getLocalizedMessages(uid, DISK_SERVICE, lang, blockRecord.get().id);
            Assert.sizeIs(1, localizedMessagesRu);
        }
    }

    @Test
    public void createNewNotificationBlockAndSiftOldest() throws Exception {
        Instant now = Instant.now().minus(Duration.standardHours(1));
        for (int i = 0; i < BLOCKS_LIMIT; i++) {
            NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord(
                    "group" + i, now.plus(Duration.standardMinutes(i)));

            notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, LOCALIZED_MESSAGES);
        }

        NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord(
                "group", now.plus(Duration.standardMinutes(10)));

        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, LOCALIZED_MESSAGES);

        Assert.some(notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.COMMENT_LIKE, "group"));
        Assert.some(notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.COMMENT_LIKE, "group1"));
        Assert.none(notificationBlockDao.findNotificationBlock(
                uid, DiskNotifications.COMMENT_LIKE, "group0"));


        for (NotifierLanguage lang : LOCALIZED_MESSAGES.keySet()) {
            ListF<LocalizedMessage> localizedMessagesRu =
                    notificationBlockDao.getLocalizedMessages(uid, DISK_SERVICE, lang, record.id);
            Assert.sizeIs(1, localizedMessagesRu);
        }
    }

    @Test
    public void createNewNotificationBlockAndSiftMultipleOldest() throws Exception {
        String oldGroupPrefix = "oldgroup";
        String groupPrefix = "group";
        int oldBlockCount = 15;

        Instant now = Instant.now().minus(Duration.standardHours(1));
        for (int i = 0; i < oldBlockCount; i++) {
            NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord(
                    oldGroupPrefix + i, now.plus(Duration.standardMinutes(i)));
            notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, LOCALIZED_MESSAGES);
        }

        for (int i = 0; i < BLOCKS_LIMIT; i++) {
            NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord(
                    groupPrefix + i, now.plus(Duration.standardMinutes(oldBlockCount + i)));
            notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, LOCALIZED_MESSAGES);
        }

        for (int i = 0; i < oldBlockCount; i++) {
            Assert.none(notificationBlockDao.findNotificationBlock(
                    uid, DiskNotifications.COMMENT_LIKE, oldGroupPrefix + i));
        }

        for (int i = 0; i < BLOCKS_LIMIT; i++) {
            Assert.some(notificationBlockDao.findNotificationBlock(
                    uid, DiskNotifications.COMMENT_LIKE, groupPrefix + i));
        }
    }

    @Test
    public void createNewBlockAndCheckUnviewedCount() {
        NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord("group_key1");
        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, Cf.hashMap());

        checkSingleServiceUnviewedCounts(1, uid, DISK_SERVICE);

        record = NotificationTestUtils.createTestNotificationBlockRecord("group_key2");
        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, Cf.hashMap());

        checkSingleServiceUnviewedCounts(2, uid, DISK_SERVICE);
    }

    @Test
    public void createNewBlocksForDifferentServicesAndCheckUnviewedCount() {
        NotificationType type = new NotificationType(
                Option.empty(), Option.empty(), "type",
                Option.of(new NotificationGroup("group",
                        new NotificationService(Option.empty(), "service", "keyset", false))),
                NotificationIcon.DEFAULT, "message", "title", "short", Option.empty(),
                CounterConfiguration.NONE, MetadataWrapper.createEmpty(), Duration.ZERO, Cf.list());

        notificationBlockDao.createNewNotificationBlockAndSiftOldest(
                uid, NotificationTestUtils.createTestNotificationBlockRecord("group_key1"), Cf.hashMap());
        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid,
                NotificationTestUtils.createTestNotificationBlockRecord("group_key1", Option.empty(), type),
                Cf.hashMap());

        Assert.equals(1, notificationBlockDao.getUnviewedBlocksCountForService(uid, DISK_SERVICE));
        Assert.equals(1, notificationBlockDao.getUnviewedBlocksCountForService(uid, type.getService().name));

        // other service is not enabled, so total should still be 1
        Assert.equals(1, notificationBlockDao.getTotalUnviewedBlocksCountForEnabledServices(uid));
    }

    @Test
    public void createNewBlockAndCheckUnviewedCountNotGreaterThanLimit() {
        Instant now = Instant.now().minus(Duration.standardHours(1));
        for (int i = 0; i < BLOCKS_LIMIT + 1; i++) {
            NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord(
                    "group" + i, now.plus(Duration.standardMinutes(i)));
            notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, Cf.hashMap());
        }

        checkSingleServiceUnviewedCounts(BLOCKS_LIMIT, uid, DISK_SERVICE);
    }

    @Test
    public void markAsReadAndCheckRemaining() {
        NotificationBlockRecord record1 = NotificationTestUtils.createTestNotificationBlockRecord("group_key1");
        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record1, Cf.hashMap());

        NotificationBlockRecord record2 = NotificationTestUtils.createTestNotificationBlockRecord("group_key2");
        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record2, Cf.hashMap());

        checkSingleServiceUnviewedCounts(2, uid, DISK_SERVICE);

        // Now the order is (newest to oldest):
        // record2 (unread)
        // record1 (unread)

        notificationBlockDao.markNotificationBlockAsRead(uid, record2.groupKey, record2.type);

        checkSingleServiceUnviewedCounts(1, uid, DISK_SERVICE);

        // record2 (read)
        // record1 (unread)

        notificationBlockDao.updateNotificationBlock(uid, DISK_SERVICE, record1.id,
                new NotificationBlockUpdate(record1.counterUniques, 2, record1.metadata.toJsonString(), Instant.now(), false),
                LOCALIZED_MESSAGES);

        checkSingleServiceUnviewedCounts(1, uid, DISK_SERVICE);

        // record1 (unread)
        // record2 (read)

        notificationBlockDao.resetUnviewedBlocksCount(uid, DISK_SERVICE);

        checkSingleServiceUnviewedCounts(0, uid, DISK_SERVICE);

        notificationBlockDao.updateNotificationBlock(uid, DISK_SERVICE, record1.id,
                new NotificationBlockUpdate(record1.counterUniques, 2, record1.metadata.toJsonString(), Instant.now(), false),
                LOCALIZED_MESSAGES);
        // still
        // record1 (unread)
        // record2 (read)
        // but unread count must be increased, for the unread message is not in top 0 unread ones

        checkSingleServiceUnviewedCounts(1, uid, DISK_SERVICE);

        notificationBlockDao.updateNotificationBlock(uid, DISK_SERVICE, record2.id,
                new NotificationBlockUpdate(record2.counterUniques, 2, record2.metadata.toJsonString(), Instant.now(), false),
                LOCALIZED_MESSAGES);

        checkSingleServiceUnviewedCounts(2, uid, DISK_SERVICE);
    }

    @Test
    public void updateBlockAndCheckUnviewedWithReset() {
        updateBlockAndCheckUnviewedCount(true);
    }

    @Test
    public void updateBlockAndCheckUnviewedWithoutReset() {
        updateBlockAndCheckUnviewedCount(false);
    }

    private void updateBlockAndCheckUnviewedCount(boolean reset) {
        NotificationBlockRecord record = NotificationTestUtils.createTestNotificationBlockRecord("group_key1");
        notificationBlockDao.createNewNotificationBlockAndSiftOldest(uid, record, Cf.hashMap());

        if (reset) {
            notificationBlockDao.resetUnviewedBlocksCount(uid, DISK_SERVICE);
        }

        NotificationActor actor = NotificationTestUtils.createRandomActor();
        NotificationBlockUpdate update = new NotificationBlockUpdate(
                Cf.list(actor.toString()), 10, "{\"meta\": {}}", Instant.now(), false);
        notificationBlockDao.updateNotificationBlock(
                uid,
                DISK_SERVICE,
                record.id,
                update,
                Cf.hashMap());

        checkSingleServiceUnviewedCounts(1, uid, DISK_SERVICE);
    }

    private void checkSingleServiceUnviewedCounts(int count, DataApiUserId uid, String service) {
        Assert.equals(count, notificationBlockDao.getUnviewedBlocksCountForService(uid, service));
        Assert.equals(count, notificationBlockDao.getTotalUnviewedBlocksCountForEnabledServices(uid));
    }
}
