package ru.yandex.chemodan.app.notifier.dao;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notifier.NotifierApp;
import ru.yandex.chemodan.app.notifier.log.NotifierEvent;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;

/**
 * @author akirakozov
 */
public class NotificationCacheDaoImpl implements NotificationCacheDao {
    private static final String DB_NAME_PREFIX = "new_notifications";
    private static final String COLLECTION_PREFIX = "notifications";

    private final DataApiManager dataApiManager;

    public NotificationCacheDaoImpl(
            DataApiManager dataApiManager)
    {
        this.dataApiManager = dataApiManager;
    }

    @Override
    public void createNewNotification(DataApiUserId userId, NotificationRecord record) {
        RecordChange change = RecordChange.insert(
                getCollection(record.type, record.groupKey), record.id, record.toData());
        dataApiManager.applyDelta(getDatabase(userId, record.type), RevisionCheckMode.PER_RECORD, new Delta(change));

        NotifierEvent
                .newNotificationCreated(userId, record.groupKey, record.type)
                .log();
    }

    @Override
    public ListF<NotificationRecord> findNewNotifications(
            DataApiUserId userId, NotificationType type, String group)
    {
        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(CollectionIdCondition.eq(getCollection(type, group)));

        return dataApiManager
                .getRecords(getDatabase(userId, type).spec(), filter)
                .map(rec -> NotificationRecord.fromDataRecord(rec, type));
    }

    @Override
    public void removeNewNotifications(
            DataApiUserId userId, NotificationType type, String groupKey, ListF<String> ids)
    {
        ListF<RecordChange> changes = Cf.arrayList();
        ids.forEach(id -> changes.add(RecordChange.delete(getCollection(type, groupKey), id)));
        dataApiManager.applyDelta(getDatabase(userId, type), RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    private String getCollection(NotificationType type, String group) {
        return COLLECTION_PREFIX + "." + type.value() + "." + group.replace(":", "_");
    }

    private Database getDatabase(DataApiUserId userId, NotificationType type) {
        String service = type.getService().name;
        AppDatabaseRef dbRef = new AppDatabaseRef(NotifierApp.APP_NAME, DB_NAME_PREFIX + "_" + service);
        return dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(userId, dbRef));
    }
}
