package ru.yandex.chemodan.app.notifier.dao;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notifier.NotifierApp;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndGroup;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.app.notifier.settings.SettingsRecord;

/**
 * DAO for Notification settings (subscription / unsubscription).
 *
 * DB contains subscription marks, which means, "true" in DB means that
 * user wants to receive notifications for the given resource and type, "false" means that user does not want to
 * receive such notifications. Default value (if there's no record in DB) is true.
 *
 * @author buberman
 */
public class NotificationSettingsDaoImpl implements NotificationSettingsDao {
    private static final String DB_NAME_PREFIX = "notifications_resource_settings";
    private static final String UNSUBSCRIBE_COLLECTION = "resource_settings";

    private final DataApiManager dataApiManager;
    private final NotificationRecordTypeManager notificationRecordTypeManager;

    public NotificationSettingsDaoImpl(
            DataApiManager dataApiManager,
            NotificationRecordTypeManager notificationRecordTypeManager)
    {
        this.dataApiManager = dataApiManager;
        this.notificationRecordTypeManager = notificationRecordTypeManager;
    }

    @Override
    public void disableResourceNotifications(
            DataApiUserId dataApiUserId, String resourceId, SetF<ServiceAndGroup> groups)
    {
        if (groups.isEmpty()) {
            return;
        }

        Database database = getOrCreateDatabase(dataApiUserId);
        String service = groups.iterator().next().getService();

        Option<DataRecord> record =
                dataApiManager.getRecords(database.spec(), recordsFilter(resourceId, service)).singleO();

        SettingsRecord settingsRecord;

        if (record.isPresent()) {
            // Updating existing record
            settingsRecord = SettingsRecord.fromDataRecord(record.get(), service, notificationRecordTypeManager);
        } else {
            // Creating new record
            settingsRecord = new SettingsRecord(makeResourceId(resourceId), service,
                    Cf.toSet(notificationRecordTypeManager.getNotificationGroupNames(service)));
        }
        RecordChange changes = RecordChange.set(getCollectionName(service), settingsRecord.id,
                settingsRecord.minusNotificationGroups(groups).toData(notificationRecordTypeManager));

        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    @Override
    public void enableResourceNotifications(
            DataApiUserId dataApiUserId, String resourceId, SetF<ServiceAndGroup> groups)
    {
        if (groups.isEmpty()) {
            return;
        }

        Database database = getOrCreateDatabase(dataApiUserId);
        String service = groups.iterator().next().getService();

        Option<DataRecord> record =
                dataApiManager.getRecords(database.spec(), recordsFilter(resourceId, service)).singleO();

        // Nothing to do if there's no such record: that means, all subscriptions are "on".
        if (record.isPresent()) {
            ListF<RecordChange> changes = Cf.arrayList();

            SettingsRecord settingsRecord =
                    SettingsRecord.fromDataRecord(record.get(), service, notificationRecordTypeManager);

            settingsRecord = settingsRecord.plusNotificationGroups(groups);

            if (isFull(settingsRecord.notificationGroups, service)) {
                // No unsubscriptions left, remove record
                changes.add(RecordChange.delete(record.get().id()));
            } else {
                // Update existing record
                changes.add(RecordChange.set(getCollectionName(service), settingsRecord.id,
                        settingsRecord.toData(notificationRecordTypeManager)));
            }
            dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
        }
    }

    @Override
    public boolean isSubscribed(DataApiUserId dataApiUserId, String resourceId, ServiceAndGroup group) {
        Database database = getOrCreateDatabase(dataApiUserId);
        String service = group.getService();

        Option<DataRecord> record =
                dataApiManager.getRecords(database.spec(), recordsFilter(resourceId, service)).singleO();
        if (record.isPresent()) {
            return SettingsRecord.fromDataRecord(record.get(), service, notificationRecordTypeManager)
                    .notificationGroups.containsTs(group);
        }
        return true;
    }

    @Override
    public CollectionF<SettingsRecord> getSubscribedResources(DataApiUserId dataApiUserId,
            String service, Option<String> resource)
    {
        Database database = getOrCreateDatabase(dataApiUserId);

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionId(getCollectionName(service));

        if (resource.isPresent()) {
            filter = filter.withRecordId(makeResourceId(resource.get()));
        }

        return dataApiManager.getRecords(database.spec(), filter).map(
                record -> SettingsRecord.fromDataRecord(record, service, notificationRecordTypeManager)
        );
    }

    private Database getOrCreateDatabase(DataApiUserId userId) {
        AppDatabaseRef dbRef =
                new AppDatabaseRef(NotifierApp.APP_NAME, DB_NAME_PREFIX + "_" + DiskServices.DISK);
        return dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(userId, dbRef));
    }

    private static RecordsFilter recordsFilter(String resourceId, String service) {
        String safeResourceId = makeResourceId(resourceId);
        return RecordsFilter.DEFAULT.withCollectionId(getCollectionName(service)).withRecordId(safeResourceId);
    }

    public static String makeResourceId(String resourceId) {
        return resourceId.replace(':', '-');
    }

    private boolean isFull(SetF<ServiceAndGroup> groups, String service) {
        ListF<ServiceAndGroup> allGroups = notificationRecordTypeManager.getNotificationGroupNames(service);
        return allGroups.size() == groups.size() && groups.containsAllTs(allGroups);
    }

    private static String getCollectionName(String service) {
        return getPrefixByService(service) + UNSUBSCRIBE_COLLECTION;
    }

    private static String getPrefixByService(String serviceName) {
        return serviceName.equals(DiskServices.DISK) ? "" : serviceName + "_";
    }

}
