package ru.yandex.chemodan.app.notifier.dao;

import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskGroups;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.app.notifier.settings.SettingsRecord;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author buberman
 */
@ActivateDataApiEmbeddedPg
@ActivateNotificationEmbeddedPg
@ContextConfiguration(classes = NotificationTypeManagerTestContextConfiguration.class)
public class NotificationSettingsDaoTest extends DataApiTestSupport {
    @Autowired
    private DataApiManager dataApiManager;
    @Autowired
    private NotificationRecordTypeManager notificationRecordTypeManager;

    private DataApiUserId uid;
    private NotificationSettingsDao notificationSettingsDao;
    private static final String RESOURCE =
            "4000475747:38bf383b093a344665df1b138f8fb3dba87677ed642840c63ec60889ce819197";
    private static final String RESOURCE2 =
            "4000475747:dcd2b7b278e5e28996df501455c66bb79c8e4f788b5f9b230a3f5ec88bf22809";
    private static final String SERVICE = DiskServices.DISK;

    @Before
    public void setup() {
        uid = createRandomCleanUserInDefaultShard();

        notificationSettingsDao = new NotificationSettingsDaoImpl(dataApiManager, notificationRecordTypeManager);
    }

    /**
     * Test run:
     * 1. Subscribe for some actions for some resource.
     * 2. Check that these (and only these) actions are unsubscribed and that there's a record in database.
     * 3. Unsubscribe from all actions.
     * 4. Check that all actions are unsubscribed.
     * 5. Check that the record itself was removed from the database.
     */
    @Test
    public void testUnsubscribeFromOneResource() {
        notificationSettingsDao.disableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.LIKES));

        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.COMMENTS));
        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.LIKES));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.AUTOUPLOAD));
        assertFalse(notificationSettingsDao.getSubscribedResources(uid, SERVICE, Option.empty()).isEmpty());

        notificationSettingsDao.enableResourceNotifications(uid, RESOURCE, Cf.toSet(
                notificationRecordTypeManager.getNotificationGroupNames(SERVICE)));

        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.COMMENTS));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.LIKES));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.AUTOUPLOAD));

        assertTrue(notificationSettingsDao.getSubscribedResources(uid, SERVICE, Option.empty()).isEmpty());
    }

    /**
     * Test run:
     * <p>
     * 1. Unsubscribe from some channels.
     * 2. Subscribe to some of them.
     * 3. Unsubscribe again (including channels that were already unsubscribed)
     * 4. Subscribe to some (including channels that were already subscribed).
     * 5. Unsubscribe leaving only one.
     * 6. Unsubscribe from that one.
     * 7. Check that the record was removed.
     */
    @Test
    public void testPartialUnsubscribe() {
        notificationSettingsDao.disableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.LIKES, DiskGroups.AUTOUPLOAD));

        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.COMMENTS));
        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.LIKES));
        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.AUTOUPLOAD));

        notificationSettingsDao.enableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.AUTOUPLOAD));

        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.COMMENTS));
        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.LIKES));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.AUTOUPLOAD));

        notificationSettingsDao.disableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.LIKES));

        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.COMMENTS));
        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.LIKES));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.AUTOUPLOAD));

        notificationSettingsDao.enableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.AUTOUPLOAD));

        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.COMMENTS));
        assertFalse(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.LIKES));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.AUTOUPLOAD));

        notificationSettingsDao.enableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.LIKES));

        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.COMMENTS));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.LIKES));
        assertTrue(notificationSettingsDao.isSubscribed(uid, RESOURCE, DiskGroups.AUTOUPLOAD));
        assertTrue(notificationSettingsDao.getSubscribedResources(uid, SERVICE, Option.empty()).isEmpty());
    }

    /**
     * Test run:
     * 1. Unsubscribe from some channels.
     * 2. Check that settings record is correct.
     * 3. Resubscribe to some of them
     * 4. Same check.
     * 5. Unsubscribe from some channels for another resource.
     * 6. Check that a new set of subscriptions contains two records.
     * 7. Subscribe to all channels for one of the resources.
     * 8. Check that a new set of subscriptions contains one record, the other is removed.
     * 9. Subscribe to all channels for another resource.
     * 10. Check that all unsubscription marks are removed.
     */
    @Test
    public void testGetSettingRecords() {
        notificationSettingsDao.disableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.LIKES, DiskGroups.AUTOUPLOAD));

        CollectionF<SettingsRecord> unsubscribedResources = notificationSettingsDao.getSubscribedResources(uid,
                SERVICE, Option.empty());
        assertEquals(1, unsubscribedResources.size());
        SettingsRecord record = unsubscribedResources.single();

        assertFalse(record.notificationGroups.containsTs(DiskGroups.COMMENTS));
        assertFalse(record.notificationGroups.containsTs(DiskGroups.LIKES));
        assertFalse(record.notificationGroups.containsTs(DiskGroups.AUTOUPLOAD));

        notificationSettingsDao.enableResourceNotifications(uid, RESOURCE,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.AUTOUPLOAD));

        unsubscribedResources = notificationSettingsDao.getSubscribedResources(uid, SERVICE, Option.empty());
        assertEquals(1, unsubscribedResources.size());
        record = unsubscribedResources.single();

        assertTrue(record.notificationGroups.containsTs(DiskGroups.COMMENTS));
        assertFalse(record.notificationGroups.containsTs(DiskGroups.LIKES));
        assertTrue(record.notificationGroups.containsTs(DiskGroups.AUTOUPLOAD));

        notificationSettingsDao.disableResourceNotifications(uid, RESOURCE2,
                Cf.set(DiskGroups.COMMENTS, DiskGroups.LIKES, DiskGroups.AUTOUPLOAD));

        unsubscribedResources = notificationSettingsDao.getSubscribedResources(uid, SERVICE, Option.empty());
        assertEquals(2, unsubscribedResources.size());

        notificationSettingsDao.enableResourceNotifications(uid, RESOURCE,
                Cf.toSet(notificationRecordTypeManager.getNotificationGroupNames(SERVICE)));

        unsubscribedResources = notificationSettingsDao.getSubscribedResources(uid, SERVICE, Option.empty());
        assertEquals(1, unsubscribedResources.size());

        notificationSettingsDao.enableResourceNotifications(uid, RESOURCE2,
                Cf.toSet(notificationRecordTypeManager.getNotificationGroupNames(SERVICE)));
        assertTrue(notificationSettingsDao.getSubscribedResources(uid, SERVICE, Option.empty()).isEmpty());
    }
}
