package ru.yandex.chemodan.app.notifier.dao;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notifier.NotifierApp;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndGroup;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.settings.SettingsRecord;

/**
 * DAO for global notification settings configuring subscription / unsubscription.
 *
 * @author buberman
 */
public class NotificationsGlobalSettingsDaoImpl implements NotificationsGlobalSettingsDao {
    private static final String DB_NAME_PREFIX = "notifications_global_settings";
    private static final String COMMON_SETTINGS_COLLECTION = "common_settings";

    private final DataApiManager dataApiManager;
    private final NotificationRecordTypeManager notificationRecordTypeManager;

    public NotificationsGlobalSettingsDaoImpl(
            DataApiManager dataApiManager,
            NotificationRecordTypeManager notificationRecordTypeManager)
    {
        this.dataApiManager = dataApiManager;
        this.notificationRecordTypeManager = notificationRecordTypeManager;
    }

    @Override
    public void disableNotifications(
            DataApiUserId dataApiUserId, GlobalSubscriptionChannel channel, SetF<ServiceAndGroup> groups)
    {
        if (groups.isEmpty()) {
            return;
        }

        Database database = getOrCreateDatabase(dataApiUserId);
        String service = groups.iterator().next().getService();

        Option<DataRecord> record =
                dataApiManager.getRecords(database.spec(), recordsFilter(channel, service)).singleO();

        SettingsRecord settingsRecord;

        if (record.isPresent()) {
            settingsRecord = SettingsRecord.fromDataRecord(record.get(), service, notificationRecordTypeManager);
        } else {
            settingsRecord = new SettingsRecord(channel.value(), service,
                    Cf.toSet(notificationRecordTypeManager.getNotificationGroupNames(service)));
        }

        RecordChange changes = RecordChange.set(getCollectionName(service), settingsRecord.id,
                settingsRecord.minusNotificationGroups(groups).toData(notificationRecordTypeManager));

        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    @Override
    public void enableNotifications(
            DataApiUserId dataApiUserId, GlobalSubscriptionChannel channel, SetF<ServiceAndGroup> groups)
    {
        if (groups.isEmpty()) {
            return;
        }

        Database database = getOrCreateDatabase(dataApiUserId);
        String service = groups.iterator().next().getService();

        Option<DataRecord> record =
                dataApiManager.getRecords(database.spec(), recordsFilter(channel, service)).singleO();

        SettingsRecord settingsRecord;

        if (record.isPresent()) {
            settingsRecord = SettingsRecord.fromDataRecord(record.get(), service, notificationRecordTypeManager);
        } else {
            settingsRecord = new SettingsRecord(channel.value(), service, Cf.set());
        }

        RecordChange changes = RecordChange.set(getCollectionName(service), settingsRecord.id,
                settingsRecord.plusNotificationGroups(groups).toData(notificationRecordTypeManager));

        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    @Override
    public SetF<ServiceAndGroup> getNotificationsSettings(
            DataApiUserId dataApiUserId, GlobalSubscriptionChannel channel, String service)
    {
        Database database = getOrCreateDatabase(dataApiUserId);

        Option<DataRecord> record =
                dataApiManager.getRecords(database.spec(), recordsFilter(channel, service)).singleO();

        if (record.isPresent()) {
            return SettingsRecord
                    .fromDataRecord(record.get(), service, notificationRecordTypeManager).notificationGroups;
        } else {
            return Cf.toSet(notificationRecordTypeManager.getNotificationGroupNames(service));
        }
    }

    protected RecordsFilter recordsFilter(GlobalSubscriptionChannel channel, String service) {
        return RecordsFilter.DEFAULT
                .withCollectionId(getCollectionName(service))
                .withRecordId(channel.value());
    }

    private Database getOrCreateDatabase(DataApiUserId userId) {
        AppDatabaseRef dbRef =
                new AppDatabaseRef(NotifierApp.APP_NAME, DB_NAME_PREFIX + "_" + DiskServices.DISK);
        return dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(userId, dbRef));
    }

    private static String getCollectionName(String service) {
        return getPrefixByService(service) + COMMON_SETTINGS_COLLECTION;
    }

    private static String getPrefixByService(String serviceName) {
        return serviceName.equals(DiskServices.DISK) ? "" : serviceName + "_";
    }
}
