package ru.yandex.chemodan.app.notifier.dao;

import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndGroup;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskGroups;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author buberman
 */
@ActivateDataApiEmbeddedPg
@ActivateNotificationEmbeddedPg
@ContextConfiguration(classes = NotificationTypeManagerTestContextConfiguration.class)
public class NotificationsGlobalSettingsDaoTest extends DataApiTestSupport {
    private static final String SERVICE = DiskServices.DISK;

    @Autowired
    private DataApiManager dataApiManager;
    @Autowired
    private NotificationRecordTypeManager notificationRecordTypeManager;

    private DataApiUserId uid;
    private NotificationsGlobalSettingsDao notificationsGlobalSettingsDao;

    @Before
    public void setup() {
        uid = createRandomCleanUserInDefaultShard();

        notificationsGlobalSettingsDao =
                new NotificationsGlobalSettingsDaoImpl(dataApiManager, notificationRecordTypeManager);
    }

    @Test
    public void testDefaultValues() {
        SetF<ServiceAndGroup> notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);

        assertTrue(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));
    }

    @Test
    public void testDisableNotificationsOneByOne() {
        notificationsGlobalSettingsDao.disableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.LIKES));

        SetF<ServiceAndGroup> notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertFalse(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.disableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.COMMENTS));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertFalse(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertFalse(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.disableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.AUTOUPLOAD));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertFalse(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertFalse(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertFalse(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));
    }

    @Test
    public void testDisableEnableOneByOne() {
        notificationsGlobalSettingsDao.disableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.LIKES));

        SetF<ServiceAndGroup> notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertFalse(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.enableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.LIKES));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertTrue(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.disableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.COMMENTS));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertTrue(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertFalse(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.enableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.COMMENTS));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertTrue(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.disableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.AUTOUPLOAD));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertTrue(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertFalse(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.enableNotifications(uid, GlobalSubscriptionChannel.WEB, Cf.set
                (DiskGroups.AUTOUPLOAD));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.WEB, SERVICE);
        assertTrue(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));
    }

    @Test
    public void testDisableIntersectingThenReenableAll() {
        notificationsGlobalSettingsDao
                .disableNotifications(uid, GlobalSubscriptionChannel.DESKTOP, Cf.set(DiskGroups.LIKES,
                        DiskGroups.COMMENTS));

        SetF<ServiceAndGroup> notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.DESKTOP, SERVICE);
        assertFalse(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertFalse(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao
                .disableNotifications(uid, GlobalSubscriptionChannel.DESKTOP, Cf.set(DiskGroups.COMMENTS,
                        DiskGroups.AUTOUPLOAD));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.DESKTOP, SERVICE);
        assertFalse(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertFalse(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertFalse(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));

        notificationsGlobalSettingsDao.enableNotifications(uid, GlobalSubscriptionChannel.DESKTOP,
                Cf.set(DiskGroups.AUTOUPLOAD, DiskGroups.COMMENTS, DiskGroups.LIKES));

        notificationsSettings = notificationsGlobalSettingsDao.getNotificationsSettings(uid,
                GlobalSubscriptionChannel.DESKTOP, SERVICE);
        assertTrue(notificationsSettings.containsTs(DiskGroups.LIKES));
        assertTrue(notificationsSettings.containsTs(DiskGroups.COMMENTS));
        assertTrue(notificationsSettings.containsTs(DiskGroups.AUTOUPLOAD));
    }
}
