package ru.yandex.chemodan.app.notifier.locale;

import java.util.regex.Matcher;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.notifier.inflector.Gender;
import ru.yandex.chemodan.app.notifier.inflector.InflectorClient;
import ru.yandex.chemodan.app.notifier.inflector.InflectorResponse;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplate;
import ru.yandex.chemodan.app.notifier.push.TextMessageGenerator;
import ru.yandex.chemodan.app.notifier.tanker.TankerManager;
import ru.yandex.chemodan.app.notifier.tanker.TankerMessageKey;
import ru.yandex.inside.tanker.model.FormWithCases;
import ru.yandex.inside.tanker.model.TankerTranslation;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author buberman
 */
public class LocaleManager {
    private final TankerManager tankerManager;
    private final InflectorClient inflectorClient;

    private static final Logger logger = LoggerFactory.getLogger(LocaleManager.class);

    protected static boolean endsWith1(long number) {
        return number % 10 == 1 && number % 100 != 11;
    }

    protected static boolean endsWith2to4(long number) {
        return number % 10 >= 2 && number % 10 <= 4 && (number % 100 < 12 || number % 100 > 14);
    }

    protected static ListF<String> getPlaceholders(FormWithCases form) {
        Matcher m = TextMessageGenerator.ENTITY_PATTERN.matcher(form.text);

        ListF<String> result = Cf.arrayList();

        while (m.find()) {
            result.add(m.group(1));
        }
        return result;
    }

    public LocaleManager(TankerManager tankerManager, InflectorClient inflectorClient) {
        this.tankerManager = tankerManager;
        this.inflectorClient = inflectorClient;
    }

    public String getMessage(NotifierLanguage locale, TankerMessageKey key, long number) {
        Option<TankerTranslation> translationO = tankerManager.getTranslation(locale, key);

        if (translationO.isPresent()) {
            TankerTranslation translation = translationO.get();
            Option<FormWithCases> correctForm = getCorrectForm(translation, Option.of(number));
            if (correctForm.isPresent()) {
                return correctForm.get().text;
            }
        }
        return "Error";
    }

    public SetF<String> getAllMessagesPlaceholders(NotificationTemplate template) {
        SetF<String> placeholders = Cf.hashSet();

        Cf.list(template.getMessageKey(), template.getTitleMessageKey(), template.getShortMessageKey())
                .forEach(key -> tankerManager.getAllTranslations(key).values()
                        .forEach(tr -> tr.form.plus(tr.form1).plus(tr.form2).plus(tr.form3).plus(tr.form4)
                                .forEach(t -> { placeholders.addAll(getPlaceholders(t)); })));

        return placeholders;
    }

    public MapF<NotifierLanguage, FormWithCases> getAllFormsWithCases(TankerMessageKey messageKey, Option<Long> number) {
        MapF<NotifierLanguage, FormWithCases> result = Cf.hashMap();

        MapF<String, TankerTranslation> translations = tankerManager.getAllTranslations(messageKey);
        for (Tuple2<String, TankerTranslation> translation : translations.entries()) {
            Option<FormWithCases> correctForm = getCorrectForm(translation.get2(), number)
                    .orElse(getCorrectForm(translation.get2(), Option.empty()));

            if (correctForm.isPresent()) {
                result.put(NotifierLanguage.R.fromValue(translation.get1()), correctForm.get());
            }
        }
        return result;
    }

    public MapF<NotifierLanguage, LocalizedMessage> getAllLocalizations(
            NotificationTemplate template, long number, MetadataWrapper metadata)
    {
        return getAllLocalizations(template.getMessageKey(), Option.of(number), Option.of(metadata));
    }

    public MapF<NotifierLanguage, LocalizedMessage> getAllLocalizations(TankerMessageKey messageKey, long number) {
        return getAllLocalizations(messageKey, Option.of(number), Option.empty());
    }

    public MapF<NotifierLanguage, LocalizedMessage> getAllLocalizations(TankerMessageKey messageKey) {
        return getAllLocalizations(messageKey, Option.empty(), Option.empty());
    }

    public MapF<NotifierLanguage, LocalizedMessage> getAllLocalizations(
            NotificationTemplate recordType, Option<Long> number, MetadataWrapper metadata)
    {
        return getAllLocalizations(recordType.getMessageKey(), number, Option.of(metadata));
    }

    public MapF<NotifierLanguage, LocalizedMessage> getAllLocalizations(TankerMessageKey messageKey,
            Option<Long> number, Option<MetadataWrapper> metadata)
    {
        MapF<NotifierLanguage, LocalizedMessage> result = Cf.hashMap();

        MapF<String, TankerTranslation> translations =
                tankerManager.getAllTranslations(messageKey);

        MapF<NotifierLanguage, FormWithCases> allFormsWithCases = getAllFormsWithCases(messageKey, number);

        for (NotifierLanguage lang : NotifierLanguage.values()) {
            if (translations.containsKeyTs(lang.value())) {
                TankerTranslation ts = translations.getTs(lang.value());
                Validate.notNull(ts, "Can't find translation for " + messageKey + ", lang: " + lang.value());
                Option<FormWithCases> correctFormO = getCorrectForm(ts, number)
                        .orElse(getCorrectForm(ts, Option.empty()));

                Validate.isTrue(correctFormO.isPresent(), "Can't find form for " + messageKey
                        + ", number: " + number + ", t: " + ts);
                FormWithCases correctForm = correctFormO.get();
                if (metadata.isPresent()) {
                    MetadataWrapper localizationMetadata = new MetadataWrapper(Cf.hashMap());
                    if (allFormsWithCases.containsKeyTs(lang)) {
                        try {
                            fillLocalizationMetadata(
                                    localizationMetadata, allFormsWithCases.getTs(lang), metadata.get(), lang);
                        } catch (Exception e) {
                            // If Inflector fails even after several retries, no inflected data should be provided.
                            // Default texts will then be used.
                            logger.error("Exception while accessing Inflector; no inflection data provided", e);
                        }
                    }
                    result.put(lang, new LocalizedMessage("", correctForm.text, localizationMetadata));
                } else {
                    result.put(lang, new LocalizedMessage(correctForm.text));
                }
            }
        }

        return result;
    }

    private void fillLocalizationMetadata(MetadataWrapper localizationMetadata, FormWithCases formWithCases,
            MetadataWrapper metadata, NotifierLanguage lang)
    {
        ListF<String> placeholders = getPlaceholders(formWithCases);

        metadata.meta.forEach((key, entity) -> {
            Option<String> text = entity.getO(MetadataEntity.translatedTextField(lang));

            if (text.isPresent() && placeholders.containsTs(key)) {
                localizationMetadata.put(key, "text", text.get());
            }
        });

        formWithCases.cases.forEach((field, aCase) -> {
            Option<String> fieldValue = localizationMetadata.getEntityField(field, "text")
                    .orElse(metadata.getEntityField(field, "text"));

            fieldValue.forEach(text -> {
                InflectorResponse response;


                Option<Gender> gender = metadata
                        .getEntityField(field, "gender")
                        .map(g -> Gender.R.fromValue(g.toLowerCase()));

                response = inflectorClient.inflectFio(text, lang, gender);

                response.forms.getO(aCase).forEach(caseValue -> {
                    localizationMetadata.put(field, "text", caseValue);
                });
            });
        });
    }

    private Option<FormWithCases> getCorrectForm(TankerTranslation translation, Option<Long> numberO) {
        if (!numberO.isPresent()) {
            return translation.form;
        } else {
            long number = numberO.get();
            if (number == 0) {
                return translation.form4;
            } else if (endsWith1(number)) {
                return translation.form1;
            } else if (endsWith2to4(number)) {
                return translation.form2;
            } else {
                return translation.form3;
            }
        }
    }
}
