package ru.yandex.chemodan.app.notifier.locale;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.notifier.inflector.InflectorClient;
import ru.yandex.chemodan.app.notifier.inflector.InflectorResponse;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.chemodan.app.notifier.tanker.TankerManager;
import ru.yandex.chemodan.app.notifier.tanker.TankerMessageKey;
import ru.yandex.inside.tanker.model.FormWithCases;
import ru.yandex.inside.tanker.model.TankerTranslation;
import ru.yandex.misc.test.Assert;

import static org.mockito.Mockito.any;
import static org.mockito.Mockito.anyString;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.when;
import static ru.yandex.chemodan.app.notifier.NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS;

/**
 * @author buberman
 */
@RunWith(MockitoJUnitRunner.class)
public class LocaleManagerTest {
    public static final String RUSSIAN_FORM_1 =
            "Пользователь %{actor} и ещё %{count} пользователь прокомментировали Вашу ссылку на файл %{entity}.";
    public static final String RUSSIAN_FORM_2 =
            "Пользователь %{actor} и ещё %{count} пользователя прокомментировали Вашу ссылку на файл %{entity}.";
    public static final String RUSSIAN_FORM_3 =
            "Пользователь %{actor} и ещё %{count} пользователей прокомментировали Вашу ссылку на файл %{entity}.";
    public static final String RUSSIAN_FORM_4 =
            "Пользователь %{actor} ответил на Ваш комментарий %{comment}.";

    public static final String ENGLISH_FORM_1 = "Users %{actor} and %{count} more disliked your link %{entity}.";
    public static final String ENGLISH_FORM_4 = "User %{actor} liked your comment: %{comment}.";

    LocaleManager localeManager;

    @Mock
    TankerManager tankerManager;

    @Mock
    InflectorClient inflectorClient;

    @Before
    public void init() {
        localeManager = new LocaleManager(tankerManager, inflectorClient);

        TankerTranslation translationRu = new TankerTranslation();
        translationRu.form = Option.empty();
        translationRu.form1 = Option.of(new FormWithCases(RUSSIAN_FORM_1, Cf.map("actor", "дат")));
        translationRu.form2 = Option.of(new FormWithCases(RUSSIAN_FORM_2, Cf.map("actor", "дат")));
        translationRu.form3 = Option.of(new FormWithCases(RUSSIAN_FORM_3, Cf.map("actor", "дат")));
        translationRu.form4 = Option.of(new FormWithCases(RUSSIAN_FORM_4, Cf.map("actor", "дат")));

        TankerTranslation translationEn = new TankerTranslation();
        translationEn.form = Option.empty();
        translationEn.form1 = Option.of(new FormWithCases(ENGLISH_FORM_1, Cf.map()));
        translationEn.form2 = Option.of(new FormWithCases(ENGLISH_FORM_1, Cf.map()));
        translationEn.form3 = Option.of(new FormWithCases(ENGLISH_FORM_1, Cf.map()));
        translationEn.form4 = Option.of(new FormWithCases(ENGLISH_FORM_4, Cf.map()));

        MapF<String, TankerTranslation> allTranslations = Cf.map(NotifierLanguage.RUSSIAN.value(), translationRu,
                NotifierLanguage.ENGLISH.value(), translationEn);

        when(tankerManager.getTranslation(eq(NotifierLanguage.ENGLISH), any()))
                .thenReturn(Option.of(translationEn));
        when(tankerManager.getTranslation(eq(NotifierLanguage.RUSSIAN), any()))
                .thenReturn(Option.of(translationRu));

        when(tankerManager.getAllTranslations(any())).thenReturn(allTranslations);

        InflectorResponse inflectorResponse = new InflectorResponse();
        inflectorResponse.forms = Cf.map("им", "Вася Пупкин", "дат", "Васе Пупкину");
        when(inflectorClient.inflectFio(anyString(), any(), any())).thenReturn(inflectorResponse);
    }

    @Test
    public void testRuLocaleManager() {
        TankerMessageKey key = NOTIFICATION_TYPE_FOR_TESTS.consMainTemplate().getMessageKey();

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 0), RUSSIAN_FORM_4);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 1), RUSSIAN_FORM_1);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 21), RUSSIAN_FORM_1);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 2), RUSSIAN_FORM_2);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 4), RUSSIAN_FORM_2);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 192), RUSSIAN_FORM_2);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 9174), RUSSIAN_FORM_2);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 5), RUSSIAN_FORM_3);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 10), RUSSIAN_FORM_3);

        Assert.equals(localeManager.getMessage(NotifierLanguage.RUSSIAN, key, 11), RUSSIAN_FORM_3);
    }

    @Test
    public void testEnLocaleManager() {
        TankerMessageKey key = NOTIFICATION_TYPE_FOR_TESTS.consMainTemplate().getMessageKey();

        Assert.equals(localeManager.getMessage(NotifierLanguage.ENGLISH, key, 0), ENGLISH_FORM_4);

        Assert.equals(localeManager.getMessage(NotifierLanguage.ENGLISH, key, 92), ENGLISH_FORM_1);
    }

    @Test
    public void testAllLocaleManager() {
        MetadataWrapper metadata = new MetadataWrapper(Cf.hashMap());
        metadata.put("actor", "text", "Вася Пупкин");
        metadata.put("actor", "gender", "m");

        MapF<NotifierLanguage, LocalizedMessage> map =
                localeManager.getAllLocalizations(NOTIFICATION_TYPE_FOR_TESTS.consMainTemplate(), 0L, metadata);

        Assert.equals(RUSSIAN_FORM_4, map.getTs(NotifierLanguage.RUSSIAN).message);
        Assert.equals("Васе Пупкину", map.getTs(NotifierLanguage.RUSSIAN).meta.getEntityField("actor", "text").get());
    }

    @Test
    public void messagePlaceholders() {
        Assert.equals(Cf.set("actor", "count", "comment", "entity"),
                localeManager.getAllMessagesPlaceholders(NOTIFICATION_TYPE_FOR_TESTS.consMainTemplate()));
    }

    @Test
    public void translatedMessagesLocalization() {
        TankerMessageKey key = Mockito.mock(TankerMessageKey.class);

        TankerTranslation ru = new TankerTranslation();
        ru.form = Option.of(new FormWithCases("На дворе %{month}", Cf.map()));

        TankerTranslation en = new TankerTranslation();
        en.form = Option.of(new FormWithCases("%{month} is at the bow", Cf.map()));

        when(tankerManager.getAllTranslations(eq(key))).thenReturn(Cf.map("ru", ru, "en", en));


        MetadataWrapper metadata = new MetadataWrapper(Cf.hashMap());

        metadata.put("month", "text@ru", "январь");
        metadata.put("month", "text@en", "January");

        MapF<NotifierLanguage, LocalizedMessage> messages = localeManager.getAllLocalizations(
                key, Option.empty(), Option.of(metadata));

        Assert.some("январь", messages.getTs(NotifierLanguage.RUSSIAN).meta.getEntityField("month", "text"));
        Assert.some("January", messages.getTs(NotifierLanguage.ENGLISH).meta.getEntityField("month", "text"));
    }

    @Test
    public void redundantLocalizationMeta() {
        TankerMessageKey key = Mockito.mock(TankerMessageKey.class);

        TankerTranslation translation = new TankerTranslation();
        translation.form = Option.of(new FormWithCases("Visit something!", Cf.map()));

        when(tankerManager.getAllTranslations(eq(key))).thenReturn(Cf.map("ru", translation));

        MetadataWrapper metadata = new MetadataWrapper(Cf.hashMap());
        metadata.put("country", "text@ru", "Украина");

        LocalizedMessage message = localeManager.getAllLocalizations(key, Option.empty(), Option.of(metadata))
                .getOrThrow(NotifierLanguage.RUSSIAN);

        Assert.none(message.meta.getEntityFieldsO("country"));
    }

    @Test
    public void pluralizationFallback() {
        TankerMessageKey key = Mockito.mock(TankerMessageKey.class);

        TankerTranslation translation = new TankerTranslation();
        translation.form = Option.of(new FormWithCases("Single variant form", Cf.map()));
        translation.form1 = Option.empty();

        when(tankerManager.getAllTranslations(eq(key))).thenReturn(Cf.map("ru", translation));

        Assert.equals(translation.form.get().text,
                localeManager.getAllFormsWithCases(key, Option.of(1L)).getOrThrow(NotifierLanguage.RUSSIAN).text);

        Assert.equals(translation.form.get().text,
                localeManager.getAllLocalizations(key, Option.of(1L), Option.empty()).getOrThrow(NotifierLanguage.RUSSIAN).message);
    }
}
