package ru.yandex.chemodan.app.notifier.log;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.log.LogDataProvider;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplate;
import ru.yandex.chemodan.app.notifier.notification.NotificationTemplateSelectionReason;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.eventlog.events.EventType;
import ru.yandex.chemodan.xiva.XivaResponseHeaders;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author buberman
 */
public class NotifierEvent extends DefaultObject implements LogDataProvider {
    public final DataApiUserId uid;
    public final NotificationLifecycleStage stage;

    public final Option<EventType> eventType;
    public final Option<String> recordType;
    public final Option<String> group;
    public final Option<String> service;
    public final Option<String> templateName;
    public final Option<NotificationTemplateSelectionReason> templateSelectionReason;
    public final Option<String> resourceId;
    public final Option<String> groupKey;
    public final ListF<String> counterUniques;
    public final Option<NotificationBlockUpdateMode> blockUpdateMode;
    public final Option<String> metadata;
    public final Option<String> xivaNotificationId;
    public final Option<String> xivaTransitId;
    public final ListF<String> xivaTags;
    public final Option<String> channel;
    public final Option<String> lentaBlockId;

    public final Option<String> pushTitle;
    public final Option<String> pushText;

    private NotifierEvent(Builder builder) {
        this.uid = builder.uid;
        this.stage = builder.stage;

        this.eventType = builder.eventType;
        this.recordType = builder.recordType;
        this.group = builder.group;
        this.service = builder.service;
        this.templateName = builder.templateName;
        this.templateSelectionReason = builder.templateSelectionReason;
        this.resourceId = builder.resourceId;
        this.groupKey = builder.groupKey;
        this.counterUniques = builder.counterUniques;
        this.blockUpdateMode = builder.blockUpdateMode;
        this.metadata = builder.metadata;
        this.xivaNotificationId = builder.xivaNotificationId;
        this.xivaTransitId = builder.xivaTransitId;
        this.channel = builder.channel;
        this.lentaBlockId = builder.lentaBlockId;
        this.xivaTags = builder.xivaTags;
        this.pushTitle = builder.pushTitle;
        this.pushText = builder.pushText;
    }

    @Override
    public Tuple2List<String, String> logData() {
        Tuple2List<String, String> result = Tuple2List.arrayList();

        result.add("uid", uid.serialize());

        result.add("lifecycle_stage", stage.value());
        eventType.forEach(c -> result.add("event_type", c.value()));
        recordType.forEach(c -> result.add("record_type", c));
        group.forEach(c -> result.add("group", c));
        service.forEach(c -> result.add("service", c));
        templateName.forEach(c -> result.add("template", c));
        resourceId.forEach(c -> result.add("resource_id", c));
        groupKey.forEach(c -> result.add("group_key", c));
        blockUpdateMode.forEach(c -> result.add("block_update_mode", c.value()));
        channel.forEach(c -> result.add("channel", c));
        metadata.forEach(c -> result.add("metadata", c));

        if (counterUniques.isNotEmpty()) {
            result.add("counterUniques", StringUtils.join(counterUniques, ","));
        }

        if (xivaTags.isNotEmpty()) {
            result.add("xiva_tags", StringUtils.join(xivaTags, ","));
        }

        xivaNotificationId.forEach(c -> result.add("xiva_notification_id", c));
        xivaTransitId.forEach(c -> result.add("xiva_transit_id", c));

        templateSelectionReason.forEach(reason -> {
            result.add("template_reason_source", reason.getReasonSource().name().toLowerCase());
            reason.metadata.entries().forEach(e -> result.add("template_reason_" + e._1, e._2));
        });

        pushTitle.forEach(c -> result.add("push_title", c));
        pushText.forEach(c -> result.add("push_text", c));

        return result;
    }

    public void log() {
        NotifierLogger.log(this);
    }

    public static NotifierEvent progress(DataApiUserId uid, NotificationLifecycleStage stage) {
        return new NotifierEvent(builder(uid, stage));
    }

    public static NotifierEvent logEventReceived(DataApiUserId uid, String resourceId, EventType eventType) {
        return new NotifierEvent(builder(uid, NotificationLifecycleStage.EVENT_RECEIVED)
                .withResourceId(resourceId)
                .withEventType(eventType));
    }

    public static NotifierEvent logLentaBlockEventReceived(DataApiUserId uid, String lentaBlockId, EventType eventType) {
        return new NotifierEvent(builder(uid, NotificationLifecycleStage.EVENT_RECEIVED)
                .withLentaBlockId(lentaBlockId)
                .withEventType(eventType));
    }

    public static NotifierEvent newNotificationCreated(DataApiUserId uid, String groupKey,
            NotificationType recordType)
    {
        return new NotifierEvent(builder(uid, NotificationLifecycleStage.NEW_NOTIFICATION_CREATED)
                .withGroupKey(groupKey).withRecordType(recordType));
    }

    public static NotifierEvent blockUpdateAlreadyScheduled(DataApiUserId uid, String groupKey,
            NotificationType recordType)
    {
        return new NotifierEvent(builder(uid, NotificationLifecycleStage.BLOCK_UPDATE_ALREADY_SCHEDULED)
                .withGroupKey(groupKey).withRecordType(recordType));
    }

    public static NotifierEvent blockUpdateScheduled(DataApiUserId uid, String groupKey,
             NotificationType recordType)
    {
        return new NotifierEvent(builder(uid, NotificationLifecycleStage.BLOCK_UPDATE_SCHEDULED)
                .withGroupKey(groupKey).withRecordType(recordType));
    }

    public static NotifierEvent performScheduledBlockUpdate(DataApiUserId uid, String groupKey,
            String recordType)
    {
        return new NotifierEvent(builder(uid, NotificationLifecycleStage.BLOCK_UPDATE_EXECUTED)
                .withGroupKey(groupKey).withRecordType(recordType));
    }

    public static NotifierEvent createOrUpdateNotificationBlock(
            DataApiUserId uid, NotificationBlockUpdateMode blockUpdateMode,
            String groupKey, NotificationType recordType, ListF<String> counterUniques, String templateName)
    {
        Builder builder = builder(uid, NotificationLifecycleStage.NOTIFICATION_BLOCK_CREATED)
                .withGroupKey(groupKey).withRecordType(recordType)
                .withBlockUpdateMode(blockUpdateMode)
                .withCounterUniques(counterUniques)
                .withTemplateName(templateName);
        return new NotifierEvent(builder);
    }

    public static NotifierEvent sendPushMessage(
            DataApiUserId uid, NotificationTemplate template,
            String channel, XivaResponseHeaders responseHeaders, MetadataWrapper metadata,
            ListF<String> xivaTags,
            String pushTitle, String pushText)
    {
        return new NotifierEvent(builder(uid, NotificationLifecycleStage.PUSH_SENT)
                .withGroupKey(template.getGroup())
                .withRecordType(template.getType())
                .withTemplateName(template.getName())
                .withTemplateSelectionReason(template.getTemplateSelectionReason())
                .withChannel(channel)
                .withXivaHeaders(responseHeaders)
                .withXivaTags(xivaTags)
                .withMetadata(metadata)
                .withPushTitle(pushTitle)
                .withPushText(pushText));
    }

    private static Builder builder(DataApiUserId uid, NotificationLifecycleStage stage) {
        return new Builder(uid, stage);
    }

    private static class Builder {
        public final DataApiUserId uid;
        public final NotificationLifecycleStage stage;

        public Option<EventType> eventType = Option.empty();
        public Option<String> recordType = Option.empty();
        public Option<String> group = Option.empty();
        public Option<String> service = Option.empty();
        public Option<String> templateName = Option.empty();
        public Option<NotificationTemplateSelectionReason> templateSelectionReason = Option.empty();
        public Option<String> resourceId = Option.empty();
        public Option<String> groupKey = Option.empty();
        public ListF<String> counterUniques = Cf.list();
        public Option<NotificationBlockUpdateMode> blockUpdateMode = Option.empty();
        public Option<String> metadata = Option.empty();
        public Option<String> xivaNotificationId = Option.empty();
        public Option<String> xivaTransitId = Option.empty();
        public Option<String> channel = Option.empty();
        public Option<String> lentaBlockId = Option.empty();
        public ListF<String> xivaTags = Cf.list();
        public Option<String> pushTitle = Option.empty();
        public Option<String> pushText = Option.empty();

        public Builder(DataApiUserId uid, NotificationLifecycleStage stage) {
            this.uid = uid;
            this.stage = stage;
        }

        public Builder withRecordType(NotificationType recordType) {
            this.recordType = Option.of(recordType.value());
            this.group = Option.of(recordType.getGroup().name);
            this.service = Option.of(recordType.getService().name);
            return this;
        }

        public Builder withRecordType(String recordType) {
            this.recordType = Option.of(recordType);
            return this;
        }

        public Builder withTemplateName(String name) {
            this.templateName = Option.of(name);
            return this;
        }

        public Builder withTemplateSelectionReason(Option<NotificationTemplateSelectionReason> reason) {
            this.templateSelectionReason = reason;
            return this;
        }

        public Builder withResourceId(String resourceId) {
            this.resourceId = Option.of(resourceId);
            return this;
        }

        public Builder withCounterUniques(ListF<String> counterUniques) {
            this.counterUniques = counterUniques;
            return this;
        }

        public Builder withEventType(EventType eventType) {
            this.eventType = Option.of(eventType);
            return this;
        }

        public Builder withGroupKey(String groupKey) {
            this.groupKey = Option.of(groupKey);
            return this;
        }

        public Builder withBlockUpdateMode(NotificationBlockUpdateMode blockUpdateMode) {
            this.blockUpdateMode = Option.of(blockUpdateMode);
            return this;
        }

        public Builder withMetadata(MetadataWrapper metadata) {
            this.metadata = Option.of(metadata.toJsonString());
            return this;
        }

        public Builder withXivaHeaders(XivaResponseHeaders responseHeaders) {
            this.xivaNotificationId = responseHeaders.notificationId;
            this.xivaTransitId = responseHeaders.transitId;
            return this;
        }

        public Builder withChannel(String channel) {
            this.channel = Option.of(channel);
            return this;
        }

        public Builder withLentaBlockId(String lentaBlockId) {
            this.lentaBlockId = Option.of(lentaBlockId);
            return this;
        }

        public Builder withXivaTags(ListF<String> xivaTags) {
            this.xivaTags = xivaTags;
            return this;
        }

        public Builder withPushText(String pushText) {
            this.pushText = Option.of(pushText);
            return this;
        }

        public Builder withPushTitle(String pushTitle) {
            this.pushTitle = Option.of(pushTitle);
            return this;
        }
    }
}
