package ru.yandex.chemodan.app.notifier.log.listener.events;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.utils.BleedingEdgeUtils;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.notifier.DiskNotificationManager;
import ru.yandex.chemodan.app.notifier.NotificationManager;
import ru.yandex.chemodan.app.notifier.log.NotifierEvent;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NewNotificationData;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.toggling.NotificationToggleRegistry;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.comment.AbstractCommentEvent;
import ru.yandex.chemodan.eventlog.events.comment.CommentEvent;
import ru.yandex.chemodan.eventlog.events.comment.CommentRef;
import ru.yandex.chemodan.eventlog.events.comment.EntityRef;
import ru.yandex.chemodan.eventlog.events.comment.LikeDislikeEvent;
import ru.yandex.chemodan.eventlog.events.eventlog.CallbackEventLogListener;
import ru.yandex.chemodan.eventlog.events.lenta.LentaLogDummyEvent;
import ru.yandex.chemodan.eventlog.events.lenta.share.SharedFolderInviteEvent;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author buberman
 */
public class NotifierEventsLogListener extends CallbackEventLogListener {
    private static final Logger logger = LoggerFactory.getLogger(NotifierEventsLogListener.class);

    private final NotificationManager notificationManager;
    private final BleedingEdge bleedingEdge;
    private final MpfsClient mpfsClient;
    private final DiskNotificationManager diskNotificationManager;
    private final NotificationRecordTypeManager notificationRecordTypeManager;
    private final NotificationToggleRegistry notificationToggleRegistry;
    private final NotificationsGlobalSettingsManager notificationsGlobalSettingsManager;

    private final DynamicProperty<Boolean> selfNotificationsEnabled;
    private final DynamicProperty<Boolean> enablePrivateCommentsProcessing;
    private final DynamicProperty<Boolean> enableLentaBlocksProcessing;

    public NotifierEventsLogListener(
            NotificationManager notificationManager,
            MpfsClient mpfsClient,
            BleedingEdge bleedingEdge,
            DiskNotificationManager diskNotificationManager,
            NotificationRecordTypeManager notificationRecordTypeManager,
            NotificationToggleRegistry notificationToggleRegistry,
            NotificationsGlobalSettingsManager notificationsGlobalSettingsManager)
    {
        this(notificationManager, mpfsClient, bleedingEdge, diskNotificationManager,
                notificationRecordTypeManager, notificationToggleRegistry, notificationsGlobalSettingsManager,
                false, false);
    }

    NotifierEventsLogListener(
            NotificationManager notificationManager,
            MpfsClient mpfsClient,
            BleedingEdge bleedingEdge,
            DiskNotificationManager diskNotificationManager,
            NotificationRecordTypeManager notificationRecordTypeManager,
            NotificationToggleRegistry notificationToggleRegistry,
            NotificationsGlobalSettingsManager notificationsGlobalSettingsManager,
            boolean enablePrivateResourceProcessing,
            boolean enableLentaLogsProcessing)
    {
        this.notificationManager = notificationManager;
        this.bleedingEdge = bleedingEdge;
        this.mpfsClient = mpfsClient;
        this.diskNotificationManager = diskNotificationManager;
        this.notificationRecordTypeManager = notificationRecordTypeManager;
        this.notificationToggleRegistry = notificationToggleRegistry;
        this.notificationsGlobalSettingsManager = notificationsGlobalSettingsManager;

        selfNotificationsEnabled = DynamicProperty.cons("notifier-self-notifications-enabled", false);
        enablePrivateCommentsProcessing = DynamicProperty.cons(
                "notifier-enable-private-comments", enablePrivateResourceProcessing);
        enableLentaBlocksProcessing = DynamicProperty.cons(
                "notifier-enable-lenta-logs-processing", enableLentaLogsProcessing);

        this.registerProcessor(SharedFolderInviteEvent.class, this::processSharedFolderInviteEventSafely);
    }

    @Override
    protected boolean shouldSkipCompletely(AbstractEvent event) {
        return event instanceof LentaLogDummyEvent;
    }

    private void processSharedFolderInviteEventSafely(DataApiUserId uid, SharedFolderInviteEvent event) {
        processEventSafely(uid, event, this::processSharedFolderInviteEvent);
    }

    private <T extends AbstractEvent> void processEventSafely(DataApiUserId uid, T event,
        Function2V<DataApiUserId, T> processor)
    {
        try {
            processor.apply(uid, event);
        } catch (Exception e) {
            logger.warn("Couldn't process event {}, {}", event, e);
        }
    }

    private void processSharedFolderInviteEvent(DataApiUserId uid, SharedFolderInviteEvent event) {
        if (!enableLentaBlocksProcessing.get()) {
            logger.debug("Lenta blocks processing disabled");
            return;
        }

        if (!isOnBleedingEdge(uid)) {
            return;
        }

        NotifierEvent.logLentaBlockEventReceived(uid, event.lentaBlockId, event.getEventType()).log();

        diskNotificationManager.addLentaBlock(uid, event.lentaBlockId);
    }

    public static String createSubscriptionKey(AbstractCommentEvent event) {
        return createKeyFromEvent(event);
    }

    public static String createGroupKey(AbstractCommentEvent event) {
        return createKeyFromEvent(event);
    }

    private boolean needProcessEvent(AbstractCommentEvent event) {
        return event.isPublicResource() || (event.isPrivateResource() && enablePrivateCommentsProcessing.get());
    }

    private static String createKeyFromEvent(AbstractCommentEvent event) {
        String prefix =
                event.isPublicResource() ? "public-" :
                event.isPrivateResource() ? "private-" : "";

        return prefix + event.entity.id;
    }

    private MetadataEntity entityOfActor(MpfsUid actorUid) {
        return new MetadataEntity(
                MetadataEntityType.USER,
                Cf.map("uid", Long.toString(actorUid.getUid().getUid())));
    }

    private MetadataEntity entityOfComment(CommentRef comment) {
        MapF<String, String> commentInfo = Cf.hashMap();
        commentInfo.put("id", comment.commentId.get());
        if (comment.commentText.isPresent()) {
            commentInfo.put("text", comment.commentText.get());
        }
        return new MetadataEntity(MetadataEntityType.COMMENT, commentInfo);
    }

    private MetadataEntity entityOfEntity(DataApiUserId uid, AbstractCommentEvent event) {
        MapF<String, String> map = Cf.<String, String>hashMap()
                .plus1("id", event.entity.id)
                .plus1("uid", uid.serialize())
                .plus1("entity_type",
                        event.isPublicResource()
                                ? "public_resource"
                                : event.isPrivateResource() ? "private_resource" : "");
        return new MetadataEntity(MetadataEntityType.RESOURCE, map);
    }

    private MetadataEntity entityOfAction(DataApiUserId uid, AbstractCommentEvent event) {
        MetadataEntity actionData = new MetadataEntity(MetadataEntityType.ACTION);

        MpfsResourceId resId = MpfsResourceId.parse(event.entity.id);
        String owner = resId.owner.getUid().toString();

        if (event.isPublicResource()) {
            if (owner.equals(uid.toString())) {
                // Own resource
                LentaRecordType lentaType = LentaRecordType.PUBLIC_RESOURCE_OWNED;
                actionData.put("block-type", lentaType.value());
            } else {
                // Other user's public resource
                LentaRecordType lentaType = LentaRecordType.PUBLIC_RESOURCE;
                actionData.put("block-type", lentaType.value());
            }
            actionData.put("file_id", resId.fileId);
        } else {
            actionData.put("entity_id", resId.serialize());
            actionData.put("block-type", LentaRecordType.SHARED_RESOURCE.value());
        }

        actionData.put("uid", uid.toString());
        actionData.put("mtime", Long.toString(Instant.now().getMillis()));

        return actionData;
    }

    private boolean isOnBleedingEdge(DataApiUserId uid) {
        boolean enabled = BleedingEdgeUtils.isOnBleedingEdge(bleedingEdge).test(uid);
        if (!enabled) {
            logger.debug("Notifications for user: " + uid + " are disabled now");
        }
        return enabled;
    }

    private boolean shouldRejectSelfNotification(DataApiUserId uid, AbstractCommentEvent event) {
        return event.isPublicResource() && !selfNotificationsEnabled.get()
                && uid.toPassportUid().equalsTs(event.performerUid.getUid());
    }

    private Option<Boolean> isFileResource(EntityRef entity) {
        MpfsResourceId resourceId = MpfsResourceId.parse(entity.id);
        return mpfsClient
                .getFileInfoOByFileId(MpfsUser.of(resourceId.owner.getUid()), resourceId.fileId)
                .map(info -> info.type.isSome("file"));
    }

    private void createNotifications(SetF<DataApiUserId> recipientUids, AbstractCommentEvent event,
        NotificationActor actor, NotificationType notificationType,
        String groupKey, MetadataWrapper metaWrapper)
    {
        Option<String> subscriptionKey = Option.of(createSubscriptionKey(event));
        for (DataApiUserId recipientUid : recipientUids) {
            try {
                if (!notificationToggleRegistry.isNotificationEnabled(notificationType, recipientUid.toString())
                        || !notificationsGlobalSettingsManager
                        .isUserSubscribed(recipientUid, GlobalSubscriptionChannel.ALL,
                                notificationType.getSettingsGroupName()))
                {
                    continue;
                }

                MetadataWrapper wrapper = metaWrapper.clone();
                wrapper.meta.put(MetadataEntityNames.ENTITY, entityOfEntity(recipientUid, event));
                wrapper.meta.put(MetadataEntityNames.ACTION, entityOfAction(recipientUid, event));

                notificationManager.createNewNotification(new NewNotificationData(
                        recipientUid,
                        actor,
                        subscriptionKey,
                        groupKey,
                        notificationType,
                        wrapper));
            } catch (Exception e) {
                logger.warn("Couldn't create notification for uid " + recipientUid, e);
            }
        }
    }

    private SetF<DataApiUserId> getRecipientsForCommentEvent(CommentEvent event) {
        if (event.isPublicResource()) {
            PassportUid recipientUid = event.parent.isEmpty() ?
                    event.getUid().getUid() : event.parent.parentAuthorUid.get().getUid();
            return Cf.set(new DataApiPassportUserId(recipientUid));
        } else {
            return getMpfsResourceOwners(MpfsResourceId.parse(event.entity.id), event.performerUid);
        }
    }

    private SetF<DataApiUserId> getRecipientsForLikeDislikeEvent(LikeDislikeEvent event) {
        if (event.isPublicResource()) {
            PassportUid recipientUid = event.comment.isEmpty() ?
                    event.getUid().getUid() : event.comment.commentAuthorUid.get().getUid();
            return Cf.set(new DataApiPassportUserId(recipientUid));
        } else {
            return getMpfsResourceOwners(MpfsResourceId.parse(event.entity.id), event.performerUid);
        }
    }

    private SetF<DataApiUserId> getMpfsResourceOwners(MpfsResourceId resourceId, MpfsUid performerUid) {
        return resourceId.owner.getUidO()
                .flatMapO(owner -> mpfsClient.getShareUidsInGroupO(MpfsUser.of(owner), resourceId.fileId))
                .map(users -> users.getUids().<DataApiUserId>map(DataApiPassportUserId::new))
                .getOrElse(toDataApiUid(resourceId.owner))
                .filter(this::isOnBleedingEdge)
                .unique()
                .minus1(new DataApiPassportUserId(performerUid.getUid()));
    }

}
