package ru.yandex.chemodan.app.notifier.log.listener.events;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.DiskNotificationManager;
import ru.yandex.chemodan.app.notifier.NotificationManager;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.notification.NewNotificationData;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.toggling.NotificationToggleRegistry;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.commentaries.EntityTypes;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.ResourceType;
import ru.yandex.chemodan.eventlog.events.comment.AbstractCommentEvent;
import ru.yandex.chemodan.eventlog.events.comment.CommentEvent;
import ru.yandex.chemodan.eventlog.events.comment.CommentEventType;
import ru.yandex.chemodan.eventlog.events.comment.CommentRef;
import ru.yandex.chemodan.eventlog.events.comment.EntityRef;
import ru.yandex.chemodan.eventlog.events.comment.LikeDislikeEvent;
import ru.yandex.chemodan.eventlog.events.comment.LikeDislikeEventType;
import ru.yandex.chemodan.eventlog.events.comment.ParentCommentRef;
import ru.yandex.chemodan.eventlog.events.lenta.LentaLogDummyEvent;
import ru.yandex.chemodan.eventlog.events.lenta.share.SharedFolderInviteEvent;
import ru.yandex.chemodan.eventlog.events.lenta.share.SharedFolderInviteEventType;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsGroupUids;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

/**
 * @author akirakozov
 */
@ActivateNotificationEmbeddedPg
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = NotificationTypeManagerTestContextConfiguration.class)
@TestExecutionListeners(value = DependencyInjectionTestExecutionListener.class)
public class NotifierEventsLogListenerTest {

    @Autowired
    private NotificationRecordTypeManager notificationRecordTypeManager;

    @Mock
    private NotificationManager notificationManager;
    @Mock
    private BleedingEdge bleedingEdge;
    @Mock
    private MpfsClient mpfsClient;
    @Mock
    private DiskNotificationManager diskNotificationManager;
    @Mock
    private NotificationToggleRegistry notificationToggleRegistry;
    @Mock
    private NotificationsGlobalSettingsManager notificationsGlobalSettingsManager;

    NotifierEventsLogListener notifierListener;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);

        notifierListener = new NotifierEventsLogListener(
                notificationManager, mpfsClient, bleedingEdge, diskNotificationManager,
                notificationRecordTypeManager, notificationToggleRegistry, notificationsGlobalSettingsManager, true, true);

        when(bleedingEdge.isOnBleedingEdge(any(PassportUid.class))).thenReturn(true);

        PassportUid uid = new PassportUid(101L);
        when(mpfsClient.getFileInfoOByFileId(MpfsUser.of(uid), "some-file-id")).
                thenReturn(Option.of(new MpfsFileInfo("/disk/file.txt", "file", MpfsFileMetaDto.builder().build())));
        when(mpfsClient.getFileInfoOByFileId(MpfsUser.of(uid), "some-folder-id")).
                thenReturn(Option.of(new MpfsFileInfo("/disk/folder", "dir", MpfsFileMetaDto.builder().build())));

        MpfsGroupUids uids = new MpfsGroupUids(Cf.arrayList(new MpfsGroupUids.User(uid, MpfsGroupUids.Status.OWNER)));
        when(mpfsClient.getShareUidsInGroupO(MpfsUser.of(uid), "some-file-id")).thenReturn(Option.of(uids));
        when(mpfsClient.getShareUidsInGroupO(MpfsUser.of(uid), "some-folder-id")).thenReturn(Option.of(uids));

        when(notificationToggleRegistry.isNotificationEnabled(any(), any())).thenReturn(true);
        when(notificationsGlobalSettingsManager.isUserSubscribed(any(), any(), any())).thenReturn(true);
    }

    @Test
    public void processRemovedFromDiskPrivateFile() throws Exception {
        when(mpfsClient.getFileInfoOByFileId(MpfsUser.of(101L), "some-file-id"))
                .thenReturn(Option.empty());
        checkNotificationCreation(createCommentEvent(EntityTypes.PRIVATE_RESOURCE, false, ResourceType.FILE), false);
    }

    @Test
    public void processCommentEventWithException() {
        when(bleedingEdge.isOnBleedingEdge(any(PassportUid.class))).thenThrow(new RuntimeException("Bad situation"));
        checkNotificationCreation(createCommentEvent(EntityTypes.PUBLIC_RESOURCE, false, ResourceType.FILE), false);
    }

    @Test
    public void processLikeDislikeEventWithException() {
        when(bleedingEdge.isOnBleedingEdge(any(PassportUid.class))).thenThrow(new RuntimeException("Bad situation"));
        checkNotificationCreation(
                createLikeDislikeEvent(EntityTypes.PUBLIC_RESOURCE, LikeDislikeEventType.LIKE_ADD, false, ResourceType.FILE),
                false);
    }

    @Test
    public void createSubscriptionKey() throws Exception {
        Assert.equals("public-101:some-file-id", NotifierEventsLogListener.createSubscriptionKey(
                createCommentEvent(EntityTypes.PUBLIC_RESOURCE, true, ResourceType.FILE)));
        Assert.equals("private-101:some-file-id", NotifierEventsLogListener.createSubscriptionKey(
                createCommentEvent(EntityTypes.PRIVATE_RESOURCE, true, ResourceType.FILE)));
        Assert.equals("101:some-file-id", NotifierEventsLogListener.createSubscriptionKey(
                createCommentEvent("unknown", true, ResourceType.FILE)));
    }

    @Test
    public void testSharedFolderInviteReceivedNotification() {
        checkSharedFolderInviteCreationArgs(SharedFolderInviteEventType.RECEIVED);
    }

    @Test
    public void testSharedFolderInviteAcceptedNotification() {
        checkSharedFolderInviteCreationArgs(SharedFolderInviteEventType.ACCEPTED);
    }

    @Test
    public void testShouldSkipCompletely() {
        Assert.isTrue(notifierListener.shouldSkipCompletely(LentaLogDummyEvent.INSTANCE));
        Assert.isFalse(notifierListener
                .shouldSkipCompletely(createCommentEvent(EntityTypes.PUBLIC_RESOURCE, false, ResourceType.FILE)));
    }

    private void checkTypeOfCreatedNotification(AbstractCommentEvent commentEvent, String type) {
        ArgumentCaptor<NewNotificationData> args = ArgumentCaptor.forClass(NewNotificationData.class);
        notifierListener.processEvent(commentEvent);

        verify(notificationManager, atLeastOnce()).createNewNotification(args.capture());
        Assert.equals(type, Cf.toList(args.getAllValues()).last().type.name);
    }

    private void checkMetadataOfCreatedNotification(AbstractCommentEvent commentEvent, String blockType) {
        ArgumentCaptor<NewNotificationData> args = ArgumentCaptor.forClass(NewNotificationData.class);
        notifierListener.processEvent(commentEvent);

        verify(notificationManager, atLeastOnce()).createNewNotification(args.capture());
        NewNotificationData data = Cf.toList(args.getAllValues()).last();
        MetadataEntity actionData = data.metadata.getEntityFieldsO(MetadataEntityNames.ACTION).get();
        Assert.equals(blockType, actionData.get("block-type"));
    }

    private void checkNotificationCreation(AbstractCommentEvent commentEvent, boolean shouldBeCreated) {
        notifierListener.processEvent(commentEvent);
        verify(notificationManager, shouldBeCreated ? times(1) : never()).createNewNotification(any());
    }

    private void checkSharedFolderInviteCreationArgs(SharedFolderInviteEventType type) {
        String lentaBlockId = "000000150477432602804860000001504624621829";
        SharedFolderInviteEvent event = createSharedFolderInviteEvent(type, lentaBlockId);

        DataApiUserId expectedUid = event.getUid().getUidO().map(DataApiPassportUserId::new).get();

        ArgumentCaptor<DataApiUserId> uidCaptor = ArgumentCaptor.forClass(DataApiUserId.class);
        ArgumentCaptor<String> lentaBlockIdCaptor = ArgumentCaptor.forClass(String.class);

        notifierListener.processEvent(event);

        verify(diskNotificationManager, atLeastOnce()).addLentaBlock(uidCaptor.capture(), lentaBlockIdCaptor.capture());

        Assert.equals(expectedUid, Cf.toList(uidCaptor.getAllValues()).last());
        Assert.equals(lentaBlockId, Cf.toList(lentaBlockIdCaptor.getAllValues()).last());
    }

    private CommentEvent createCommentEvent(String entityType, boolean topic, ResourceType resourceType) {
        MpfsUid author = new MpfsUid(321L);
        ParentCommentRef parentRef = topic ?
                new ParentCommentRef(Option.empty(), Option.empty()) :
                new ParentCommentRef("23", new MpfsUid(423L));

        return new CommentEvent(
                    CommentEventType.COMMENT_ADD,
                    new EventMetadata(new MpfsUid(123L), Instant.now(), Option.empty()),
                    author,
                    new EntityRef(entityType, getEntityId(resourceType), resourceType),
                    new CommentRef("1", author, "Some comment text"),
                    parentRef, Cf.list());
    }

    private LikeDislikeEvent createLikeDislikeEvent(String entityType, LikeDislikeEventType type,
            boolean forComment, ResourceType resourceType)
    {
        CommentRef commentRef = forComment ?
                new CommentRef("1", new MpfsUid(321L), "Some comment text") :
                new CommentRef(Option.empty(), Option.empty(), Option.empty());

        return new LikeDislikeEvent(
                type,
                new EventMetadata(new MpfsUid(123L), Instant.now(), Option.empty()),
                new MpfsUid(432L),
                new EntityRef(entityType, getEntityId(resourceType), resourceType),
                commentRef,
                Cf.list());
    }

    private SharedFolderInviteEvent createSharedFolderInviteEvent(SharedFolderInviteEventType type, String lentaBlockId) {
        return new SharedFolderInviteEvent(
                type,
                new EventMetadata(new MpfsUid(123L), Instant.now(), Option.empty()),
                lentaBlockId);
    }

    private String getEntityId(ResourceType resourceType) {
        return resourceType == ResourceType.FILE ?
                "101:some-file-id" : "101:some-folder-id";
    }

}
