package ru.yandex.chemodan.app.notifier.masters.counter;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.notifier.NotificationTestUtils;
import ru.yandex.chemodan.app.notifier.config.CounterConfiguration;
import ru.yandex.chemodan.app.notifier.config.CounterType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationBlockUpdate;
import ru.yandex.chemodan.app.notifier.notification.NotificationGroup;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskIcons;
import ru.yandex.misc.test.Assert;

/**
 * @author buberman
 */
public class CountMastersTest {

    public static final String GROUP_1 = "group1";
    private final CountMasterManager countMasterManager = new CountMasterManager(Cf.list(
            new IncrementCountMaster(),
            new PlusCountMaster(),
            new UniqueCountMaster(),
            new UniqueButOneCountMaster()
    ));

    @Test
    public void testIncrementCountMaster() {
        NotificationType recordType = createType(new CounterConfiguration(CounterType.INCREMENT, Cf.list()));

        ListF<NotificationRecord> records = makeNotificationRecords(3, recordType);

        NotificationBlockUpdate update = new NotificationBlockUpdate();
        update = countMasterManager.startCount(update, records);
        Assert.equals(3l, update.count.get());

        NotificationBlockRecord block = createNotificationBlockRecord(Cf.list(), recordType);
        update = countMasterManager.adjustCount(update, block, records);
        Assert.equals(5l, update.count.get());
    }

    @Test
    public void testPlusCountMaster() {
        NotificationType recordType = createType(new CounterConfiguration(CounterType.PLUS, Cf.list("foo.bar")));

        ListF<NotificationRecord> records = makeNotificationRecords(3, recordType);

        for (int i = 0; i < records.size(); i++) {
            records.get(i).metadata.meta.put("foo", new MetadataEntity(MetadataEntityType.OTHER,
                    Cf.map("bar", Long.toString(4 * i + 1))));
        }

        NotificationBlockUpdate update = new NotificationBlockUpdate();
        update = countMasterManager.startCount(update, records);

        Assert.equals(15l, update.count.get());

        NotificationBlockRecord block = createNotificationBlockRecord(Cf.list(), recordType);
        update = countMasterManager.adjustCount(update, block, records);
        Assert.equals(17l, update.count.get());
    }

    @Test
    public void testUniquesCountManager() {
        NotificationType recordType = createType(new CounterConfiguration(CounterType.UNIQUE, Cf.list("actor.uid")));

        ListF<NotificationRecord> records = makeNotificationRecords(5, recordType);

        NotificationBlockUpdate update = new NotificationBlockUpdate();
        update = countMasterManager.startCount(update, records);

        Assert.equals(5l, update.count.get());

        // Current uniques: three same as in new records list, two others new
        ListF<String> firstThreeActors = records
                .take(3)
                .plus(makeNotificationRecords(3, recordType))
                .map(nr -> nr.metadata.getEntityField("actor", "uid").get());

        NotificationBlockRecord block = createNotificationBlockRecord(firstThreeActors, recordType);
        update = countMasterManager.adjustCount(update, block, records);

        // Total expected: 6 existing plus 2 new from new notifications list.
        Assert.equals(8l, update.count.get());
    }

    @Test
    public void testUniquesButOneCountManager() {
        NotificationType recordType = createType(new CounterConfiguration(CounterType.UNIQUE_BUT_ONE, Cf.list("actor.uid")));

        ListF<NotificationRecord> records = makeNotificationRecords(5, recordType);

        NotificationBlockUpdate update = new NotificationBlockUpdate();
        update = countMasterManager.startCount(update, records);

        Assert.equals(4l, update.count.get());

        ListF<String> firstThreeActors = records
                .take(3)
                .plus(makeNotificationRecords(3, recordType))
                .map(nr -> nr.metadata.getEntityField("actor", "uid").get());

        NotificationBlockRecord block = createNotificationBlockRecord(firstThreeActors, recordType);
        update = countMasterManager.adjustCount(update, block, records);

        Assert.equals(7l, update.count.get());
    }

    private static NotificationType createType(CounterConfiguration counters) {
        return new NotificationType(
                Option.empty(), Option.empty(), "plus_me",
                Option.of(new NotificationGroup(Option.empty(), 1, "comments", Option.empty(), Option.empty())),
                DiskIcons.COMMENT, "message", "title", "short", Option.empty(), counters,
                MetadataWrapper.createEmpty(), Duration.ZERO, Cf.list());
    }

    private ListF<NotificationRecord> makeNotificationRecords(int count, NotificationType recordType) {
        return Cf.range(0, count).map(i ->
                NotificationTestUtils.createTestNewNotificationRecord(recordType, GROUP_1, Option.empty()));
    }

    private NotificationBlockRecord createNotificationBlockRecord(ListF<String> actors, NotificationType recordType) {
        return new NotificationBlockRecord("", actors, GROUP_1, recordType, Option.of(2L),
                new MetadataWrapper(Cf.hashMap()), "", Option.empty(), "", Instant.now(),
                Instant.now(), false, Option.empty());
    }
}
