package ru.yandex.chemodan.app.notifier.notification;

import lombok.EqualsAndHashCode;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.bolts.collection.Either;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataColumn;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderTextValue;
import ru.yandex.misc.lang.Validate;

/**
 * @author akirakozov
 */
@EqualsAndHashCode
public class NotificationActor {
    public static final String YA_DISK_ACTOR = "ya_disk";
    public static final NotificationActor YA_DISK = consFromStringName(YA_DISK_ACTOR);

    private final Either<DataApiUserId, String> actor;

    private NotificationActor(DataApiUserId uid) {
        actor = Either.left(uid);
    }

    private NotificationActor(String serviceName) {
        actor = Either.right(serviceName);
    }

    public static NotificationActor consFromUid(DataApiUserId uid) {
        return new NotificationActor(uid);
    }

    public static NotificationActor consFromUid(PassportUid uid) {
        Validate.isTrue(uid.isPublic());
        return new NotificationActor(new DataApiPassportUserId(uid));
    }

    public static NotificationActor consFromUid(long uid) {
        return consFromUid(new DataApiPassportUserId(uid));
    }

    public static NotificationActor consFromStringName(String serviceName) {
        Validate.notNull(serviceName);
        return new NotificationActor(serviceName);
    }

    public boolean isUser() {
        return actor.isLeft();
    }

    public DataApiUserId getUid() {
        return actor.getLeft();
    }

    @BenderTextValue
    public static NotificationActor consFromSerialized(String value) {
        if (StringUtils.isNumeric(value) || StringUtils.startsWith(value, "yt:")) {
            return NotificationActor.consFromUid(DataApiUserId.parse(value));
        } else {
            return NotificationActor.consFromStringName(value);
        }
    }

    @BenderTextValue
    public String serialize() {
        return actor.isLeft() ?
            actor.getLeft().toString() : actor.getRight();
    }

    public static RecordField<NotificationActor> createRecordField(String name) {
        return new RecordField<>(name,
                d -> NotificationActor.consFromSerialized(d.stringValue()),
                v -> DataField.string(v.serialize()),
                DataColumn::string, NotificationActor::serialize);
    }

    @Override
    public String toString() {
        return serialize();
    }
}
