package ru.yandex.chemodan.app.notifier.notification;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.chemodan.app.dataapi.support.RecordFieldUtils;
import ru.yandex.chemodan.app.notifier.NotificationManager;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author akirakozov
 */
public class NotificationBlockRecord extends DefaultObject {
    public final String id;
    public final ListF<String> counterUniques;
    public final String groupKey;
    public final NotificationType type;
    public final Option<Long> count;
    public final MetadataWrapper metadata;
    public final String actor;
    public final Option<String> preview;
    public final String action;
    public final Instant ctime;
    public final Instant mtime;
    public final boolean isRead;
    public final Option<String> subscriptionKey;

    public NotificationBlockRecord(
            String id, ListF<String> counterUniques,
            String groupKey, NotificationType type,
            Option<Long> count, MetadataWrapper metadata,
            String actor, Option<String> preview, String action,
            Instant ctime, Instant mtime,
            boolean isRead,
            Option<String> subscriptionKey)
    {
        this.id = id;
        this.counterUniques = counterUniques;
        this.groupKey = groupKey;
        this.type = type;
        this.count = count;
        this.metadata = metadata;
        this.actor = actor;
        this.preview = preview;
        this.action = action;
        this.ctime = ctime;
        this.mtime = mtime;
        this.isRead = isRead;
        this.subscriptionKey = subscriptionKey;
    }

    public interface Fields {
        /** @deprecated Use counterUniques instead */
        RecordField<String> ACTORS = RecordField.string("actors");

        RecordField<String> COUNTER_UNIQUES = RecordField.string("counter_uniques");
        RecordField<String> GROUP_KEY = RecordField.string("group_key");
        RecordField<String> TYPE = RecordField.string("type");
        RecordField<Long> COUNT = RecordField.longNumber("count");
        RecordField<String> META = RecordField.string("meta");
        RecordField<String> ACTOR = RecordField.string("actor");
        RecordField<String> PREVIEW = RecordField.string("preview");
        RecordField<String> ACTION = RecordField.string("action");
        RecordField<Instant> CTIME = RecordField.instant("ctime");
        RecordField<Instant> MTIME = RecordField.instant("mtime");
        RecordField<Boolean> IS_READ = RecordField.bool("is_read");
        RecordField<String> SUBSCRIPTION_KEY = RecordField.string("subscription_key");

        SetF<String> NAMES = RecordFieldUtils.fieldNames(Fields.class);
    }

    public static NotificationBlockRecord fromDataRecord(DataRecord rec, String service,
        NotificationRecordTypeManager typeManager)
    {
        MetadataWrapper metadata = MetadataWrapper.fromJsonString(Fields.META.get(rec));

        Option<String> uniquesO = Fields.COUNTER_UNIQUES.getO(rec);
        ListF<String> uniques;
        if (uniquesO.isPresent()) {
            uniques = Cf.list(StringUtils.split(uniquesO.get(), ","));
        } else {
            uniques = Cf.list(StringUtils.split(Fields.ACTORS.getO(rec).getOrElse(""), ","));
        }

        return new NotificationBlockRecord(
                rec.getRecordId(),
                uniques, Fields.GROUP_KEY.get(rec), typeManager.resolveRecordType(Fields.TYPE.get(rec), service),
                Fields.COUNT.getO(rec), metadata, Fields.ACTOR.get(rec), Fields.PREVIEW.getO(rec),
                Fields.ACTION.getO(rec).getOrElse(NotificationManager.ACTION_KEY),
                Fields.CTIME.get(rec), Fields.MTIME.get(rec), Fields.IS_READ.get(rec),
                Fields.SUBSCRIPTION_KEY.getO(rec));
    }

    public MapF<String, DataField> toData() {
        String metadataAsString = metadata.toJsonString();
        String actorsAsString = uniquesToString(counterUniques);

        return Cf.toMap(
                Cf.list(
                        Fields.COUNTER_UNIQUES.toData(actorsAsString), Fields.GROUP_KEY.toData(groupKey),
                        Fields.TYPE.toData(type.value()), Fields.META.toData(metadataAsString),
                        Fields.ACTOR.toData(actor), Fields.ACTION.toData(action), Fields.CTIME.toData(ctime),
                        Fields.MTIME.toData(mtime), Fields.IS_READ.toData(isRead)
                )
                        .plus(subscriptionKey.map(Fields.SUBSCRIPTION_KEY::toData))
                        .plus(count.map(Fields.COUNT::toData))
                        .plus(preview.map(Fields.PREVIEW::toData))
        );
    }

    public static String uniquesToString(ListF<String> counterUniques) {
        return StringUtils.join(counterUniques, ",");
    }

    public NotificationBlockRecord withCounterUniques(ListF<String> counterUniques) {
        return new NotificationBlockRecord(
                id, counterUniques, groupKey, type, Option.of(counterUniques.size() - 1L), metadata,
                actor, preview, action, ctime, mtime, isRead, subscriptionKey);
    }
}
