package ru.yandex.chemodan.app.notifier.notification;

import lombok.Data;
import lombok.EqualsAndHashCode;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.chemodan.app.dataapi.support.RecordFieldUtils;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author akirakozov
 */
@Data
@EqualsAndHashCode(exclude = "actor")
public class NotificationRecord implements Comparable<NotificationRecord> {
    public final String id;

    /**
     *  Actor, who initiate notification (user or Yandex.Disk)
     */
    public final NotificationActor actor;

    /**
     * Key for aggregation of notifications
     */
    public final String groupKey;
    public final Option<String> subscriptionKey;
    public final NotificationType type;
    public final Instant ctime;

    public final MetadataWrapper metadata;


    public NotificationRecord(
            String id,
            NotificationActor actor,
            Option<String> subscriptionKey,
            String groupKey,
            NotificationType type,
            MetadataWrapper metadata,
            Instant ctime)
    {
        this.id = id;
        this.actor = actor;
        this.subscriptionKey = subscriptionKey;
        this.groupKey = groupKey;
        this.type = type;
        this.metadata = metadata;
        this.ctime = ctime;
    }

    @Override
    public int compareTo(@NotNull NotificationRecord o) {
        return this.ctime.compareTo(o.ctime);
    }

    public interface Fields {
        RecordField<NotificationActor> ACTOR = NotificationActor.createRecordField("actor");
        RecordField<String> GROUP_KEY = RecordField.string("group_key");
        RecordField<String> TYPE = RecordField.string("type");
        RecordField<Instant> CTIME = RecordField.instant("ctime");
        RecordField<String> META = RecordField.string("meta");
        RecordField<String> SUBSCRIPTION_KEY = RecordField.string("subscription_key");
        // XXX: remove this field after migration
        RecordField<String> RESOURCE_ID = RecordField.string("resourceid");

        SetF<String> NAMES = RecordFieldUtils.fieldNames(Fields.class);
    }

    public static NotificationRecord fromDataRecord(
            DataRecord rec, NotificationType type)
    {
        String serializedMetadata = Fields.META.get(rec);

        MetadataWrapper metadata = null;
        if (!StringUtils.isEmpty(serializedMetadata)) {
            metadata = MetadataWrapper.fromJsonString(serializedMetadata);
        }

        Option<String> subscriptionKey = Fields.SUBSCRIPTION_KEY.getO(rec);
        if (!subscriptionKey.isPresent()) {
            // XXX: remove after migration
            subscriptionKey = Fields.RESOURCE_ID.getO(rec);
        }

        Assert.equals(Fields.TYPE.get(rec), type.name, "Unsuitable type name");
        return new NotificationRecord(
                rec.getRecordId(),
                Fields.ACTOR.get(rec), subscriptionKey, Fields.GROUP_KEY.get(rec),
                type, metadata, Fields.CTIME.get(rec));
    }

    public MapF<String, DataField> toData() {
        String serializedMetadata = metadata.toJsonString();

        return Cf.toMap(
                Cf.list(
                        Fields.ACTOR.toData(actor), Fields.GROUP_KEY.toData(groupKey), Fields.TYPE.toData(type.value()),
                        Fields.META.toData(serializedMetadata), Fields.CTIME.toData(ctime)
                ).plus(subscriptionKey.map(Fields.SUBSCRIPTION_KEY::toData))
        );
    }

    public NotificationRecord withLastActor(NotificationActor actor) {
        MetadataWrapper metadata = this.metadata;
        if (metadata == null) {
            metadata = new MetadataWrapper(Cf.hashMap());
            metadata.meta.put(MetadataEntityNames.ACTOR, new MetadataEntity(MetadataEntityType.USER, Cf.map(
                    "uid", actor.serialize()
            )));
        } else if (!metadata.meta.containsKeyTs(MetadataEntityNames.ACTOR)) {
            metadata.meta.put(MetadataEntityNames.ACTOR, new MetadataEntity(MetadataEntityType.USER, Cf.map(
                    "uid", actor.serialize()
            )));
        } else {
            metadata.put(MetadataEntityNames.ACTOR, "uid", actor.serialize());
        }
        return new NotificationRecord(id, actor, subscriptionKey, groupKey, type, metadata, ctime);
    }

    public NotificationRecord withMetadata(MetadataWrapper metadata) {
        return new NotificationRecord(id, actor, subscriptionKey, groupKey, type, metadata, ctime);
    }

}
