package ru.yandex.chemodan.app.notifier.notification;

import javax.annotation.PostConstruct;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.notifier.admin.NotifierTypesData;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author buberman
 */
public class NotificationRecordTypeManager extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(NotificationRecordTypeManager.class);

    private volatile MapF<String, MapF<String, NotificationType>> diskNotificationTypesMap;
    private volatile MapF<String, MapF<String, NotificationGroup>> diskNotificationGroupsMap;

    private final NotifierTypesLoader typesLoader;
    private final Option<NotifierTypesFsCache> typesFsCache;

    public NotificationRecordTypeManager(
            NotifierTypesLoader notifierTypesLoader, Option<NotifierTypesFsCache> typesFsCache)
    {
        this.typesLoader = notifierTypesLoader;
        this.typesFsCache = typesFsCache;

        this.diskNotificationTypesMap = Cf.map();
        this.diskNotificationGroupsMap = Cf.map();
    }

    private MapF<String, MapF<String, NotificationType>> typesListToMap(ListF<NotificationType> allTypes) {
        return allTypes
                .groupBy(t -> t.getService().name)
                .mapValues(types -> types.toMap(type -> Tuple2.tuple(type.value(), type)));
    }

    private MapF<String, MapF<String, NotificationGroup>> groupsListToMap(
            ListF<NotificationGroup> allGroups)
    {
        return allGroups
                .groupBy(g -> g.getService().name)
                .mapValues(groups -> groups.toMap(group -> Tuple2.tuple(group.name, group)));
    }

    @PostConstruct
    public void init() {
        refreshCashedTypesData();
    }

    public void refreshCashedTypesData() {
        logger.debug("Start refreshing types information");

        NotifierTypesData typesData = loadNotifierTypesData();
        diskNotificationTypesMap = typesListToMap(typesData.types);
        diskNotificationGroupsMap = groupsListToMap(typesData.groups);

        logger.debug("Types information was refreshed");
    }

    private NotifierTypesData loadNotifierTypesData() {
        try {
            NotifierTypesData typesData = typesLoader.loadFilledRecordTypes();
            typesFsCache.ifPresent(c -> c.storeFilledRecordTypes(typesData));
            return typesData;
        } catch (Exception e) {
            logger.warn("Couldn't load types from db, try load from cache", e);
            try {
                return typesFsCache.map(NotifierTypesFsCache::loadFilledRecordTypes)
                        .getOrThrow(() -> ExceptionUtils.translate(e));
            } catch (Exception t) {
                t.addSuppressed(e);
                throw t;
            }
        }
    }

    public NotificationType resolveRecordType(ServiceAndType name) {
        return resolveRecordType(name.getType(), name.getService());
    }

    public Option<NotificationType> resolveRecordTypeO(ServiceAndType serviceAndType) {
        return resolveRecordTypeO(serviceAndType.getType(), serviceAndType.getService());
    }

    public NotificationType resolveRecordType(String recordTypeName, String service) {
        return resolveRecordTypeO(recordTypeName, service).getOrThrow(
                () -> new UnknownNotificationTypeException("Couldn't find notification type: " + recordTypeName));
    }

    public Option<NotificationType> resolveRecordTypeO(String recordTypeName, String service) {
        return diskNotificationTypesMap.getOrElse(service, Cf.map()).getO(recordTypeName);
    }

    public ListF<NotificationType> getRecordTypes() {
        return diskNotificationTypesMap.values().flatMap(MapF::values);
    }

    public ListF<ServiceAndType> getRecordTypeNames(ServiceAndGroup name) {
        return diskNotificationTypesMap.getOrElse(name.getService(), Cf.map()).values()
                .filterMap(t -> Option.when(t.getGroup().getFullName().equals(name), t.getFullName()));
    }

    public NotificationGroup resolveNotificationGroup(ServiceAndGroup name) {
        return resolveNotificationGroup(name.getGroup(), name.getService());
    }

    public ServiceAndGroup resolveNotificationGroupName(String groupName, String service) {
        return resolveNotificationGroup(groupName, service).getFullName();
    }

    public NotificationGroup resolveNotificationGroup(String groupName, String service) {
        return resolveNotificationGroupO(groupName, service)
                .getOrThrow("Couldn't find notification group: " + groupName);
    }

    public Option<NotificationGroup> resolveNotificationGroupO(String groupName, String service) {
        return diskNotificationGroupsMap.getOrElse(service, Cf.map()).getO(groupName);
    }

    public ListF<NotificationGroup> getNotificationGroups(String service) {
        return diskNotificationGroupsMap.getOrElse(service, Cf.map()).values().toList();
    }

    public ListF<ServiceAndGroup> getNotificationGroupNames(String service) {
        return diskNotificationGroupsMap.getOrElse(service, Cf.map()).values().map(NotificationGroup::getFullName);
    }

    @Override
    protected void execute() {
        refreshCashedTypesData();
    }

    protected Duration defaultDelay() {
        return Duration.standardMinutes(5);
    }
}
