package ru.yandex.chemodan.app.notifier.notification;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.config.CounterConfiguration;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.tanker.TankerMessageKey;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author akirakozov
 */
@BenderBindAllFields
public class NotificationType extends DefaultObject {

    public final Option<Integer> id;
    public final Option<Integer> groupId;

    public final String name;
    public final NotificationIcon icon;
    public final String tankerMessageKey;
    public final String tankerTitleMessageKey;
    public final String tankerShortMessageKey;
    public final Option<String> description;
    public final CounterConfiguration counterConfiguration;
    public final MetadataWrapper defaultMetadata;
    public final Duration defaultDelay;
    public final ListF<NotificationVariant> alternativeVariants;

    private final Option<NotificationGroup> group;

    public NotificationType(
            int id,
            int groupId,
            String name,
            NotificationIcon icon,
            String tankerMessageKey,
            String tankerTitleMessageKey,
            String tankerShortMessageKey,
            Option<String> description,
            CounterConfiguration counterConfiguration,
            MetadataWrapper defaultMetadata,
            Duration delay,
            ListF<NotificationVariant> alternativeVariants)
    {
        this(Option.of(id), Option.of(groupId), name, Option.empty(),
                icon, tankerMessageKey, tankerTitleMessageKey, tankerShortMessageKey,
                description, counterConfiguration, defaultMetadata, delay, alternativeVariants);
    }

    public NotificationType(
            Option<Integer> id,
            Option<Integer> groupId,
            String name,
            Option<NotificationGroup> group,
            NotificationIcon icon,
            String tankerMessageKey,
            String tankerTitleMessageKey,
            String tankerShortMessageKey,
            Option<String> description,
            CounterConfiguration counterConfiguration,
            MetadataWrapper defaultMetadata,
            Duration delay,
            ListF<NotificationVariant> alternativeVariants)
    {
        this.id = id;
        this.groupId = groupId;
        this.name = name;
        this.group = group;
        this.icon = icon;
        this.tankerMessageKey = tankerMessageKey;
        this.tankerTitleMessageKey = tankerTitleMessageKey;
        this.tankerShortMessageKey = tankerShortMessageKey;
        this.description = description;
        this.counterConfiguration = counterConfiguration;
        this.defaultMetadata = defaultMetadata;
        this.defaultDelay = delay;
        this.alternativeVariants = alternativeVariants;
    }

    public String value() {
        return name;
    }

    public NotificationType withCounterConfiguration(CounterConfiguration counterConf) {
        return new NotificationType(id, groupId, name, group, icon, tankerMessageKey, tankerTitleMessageKey,
                tankerShortMessageKey, description, counterConf, defaultMetadata, defaultDelay, alternativeVariants);
    }

    public NotificationType withGroup(NotificationGroup group) {
        return new NotificationType(id, groupId, name, Option.of(group), icon, tankerMessageKey, tankerTitleMessageKey,
                tankerShortMessageKey, description, counterConfiguration, defaultMetadata, defaultDelay, alternativeVariants);
    }

    public NotificationType withAlternativeVariants(ListF<NotificationVariant> alternativeVariants) {
        return new NotificationType(id, groupId, name, group, icon, tankerMessageKey, tankerTitleMessageKey,
                tankerShortMessageKey, description, counterConfiguration, defaultMetadata, defaultDelay, alternativeVariants);
    }

    public NotificationGroup getGroup() {
        return group.getOrThrow("Group is not defined");
    }

    public ServiceAndGroup getGroupName() {
        return getGroup().getFullName();
    }

    public ServiceAndGroup getSettingsGroupName() {
        return getGroup().getSettingsFullName();
    }

    public NotificationService getService() {
        return group.get().getService();
    }

    public ServiceAndType getFullName() {
        return new ServiceAndType(getService().name, name);
    }

    public ListF<NotificationVariant> getAllVariants() {
        return alternativeVariants.plus1(NotificationVariant.consMain());
    }

    public ListF<NotificationTemplate> getAllTemplates() {
        return getAllVariants().map(this::consTemplate);
    }

    public NotificationTemplate consTemplate(DataApiUserId uid) {
        return consTemplate(alternativeVariants.find(v -> v.matches(uid)).getOrElse(NotificationVariant::consMain));
    }

    public NotificationTemplate consMainTemplate() {
        return consTemplate(NotificationVariant.consMain());
    }

    public NotificationTemplate consTemplate(String variantName) {
        return consTemplate(alternativeVariants.find(v -> v.getName().equals(variantName)).getOrElse(NotificationVariant::consMain));
    }

    public NotificationTemplate consTemplate(NotificationVariant variant) {
        return new NotificationTemplate(
                variant.getName(), name, getGroup().name, getService().name,
                variant.getIcon().getOrElse(icon.name),
                getTankerMessageKey(variant.getMessageKey().getOrElse(tankerMessageKey)),
                getTankerMessageKey(variant.getTitleMessageKey().getOrElse(tankerTitleMessageKey)),
                getTankerMessageKey(variant.getShortMessageKey().getOrElse(tankerShortMessageKey)),
                variant.getDefaultMeta().getOrElse(defaultMetadata),
                variant.getDelay().getOrElse(defaultDelay),
                Option.empty()).split().shuffle().first();
    }

    private TankerMessageKey getTankerMessageKey(String messageKey) {
        return new TankerMessageKey(getService().tankerKeysetId, messageKey);
    }
}
