package ru.yandex.chemodan.app.notifier.notification.toggling;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author bursy
 */
public class NotificationToggleRegistry extends ZkRegistry<String, NotificationToggleSettings> {
    private static final Logger logger = LoggerFactory.getLogger(NotificationToggleRegistry.class);

    private static final String ID_SEPARATOR = ":";

    public NotificationToggleRegistry(ZkPath path) {
        super(path, new BenderMapper().createParserSerializer(NotificationToggleSettings.class),
                NotificationToggleRegistry::id, s -> s);
    }

    /**
     * Disables notifications for given service & type. If no settings were saved previously, creates them.
     * Note: this preserves previously added enabled users so it's possible to change the type to ENABLED_FOR_SOME
     * without re-adding them.
     * @param serviceName service name for notification type, example: disk
     * @param notificationTypeName notification type, example: shared_folder_invite_received
     */
    public void disableNotifications(String serviceName, String notificationTypeName) {
        changeType(serviceName, notificationTypeName, NotificationToggleType.DISABLED_FOR_ALL);
    }

    /**
     * Enables notifications for given service & type. If no settings were saved previously, creates them.
     * Note: this preserves previously added enabled users so it's possible to change the type to ENABLED_FOR_SOME
     * without re-adding them.
     * @param serviceName service name for notification type, example: disk
     * @param notificationTypeName notification type, example: shared_folder_invite_received
     */
    public void enableNotifications(String serviceName, String notificationTypeName) {
        changeType(serviceName, notificationTypeName, NotificationToggleType.ENABLED_FOR_ALL);
    }

    /**
     * Enables notifications for given service & type to currently configured users.
     * If no settings were saved previously, creates them.
     * Note: this requires users to be added by {@link #addUser(String, String, String)}
     * @param serviceName service name for notification type, example: disk
     * @param notificationTypeName notification type, example: shared_folder_invite_received
     */
    public void enableNotificationsForSome(String serviceName, String notificationTypeName) {
        changeType(serviceName, notificationTypeName, NotificationToggleType.ENABLED_FOR_SOME);
    }

    /**
     * Adds a user to enabled user list. If no settings were saved previously, creates them.
     * Note: this doesn't change the behavior if the type was not changed to ENABLED_FOR_SOME. It can be done using
     * {@link #enableNotificationsForSome(String, String)}
     * @param serviceName service name for notification type, example: disk
     * @param notificationTypeName notification type, example: shared_folder_invite_received
     */
    public void addUser(String serviceName, String notificationTypeName, String uid) {
        NotificationToggleSettings settings = getOrCreateSettings(serviceName, notificationTypeName);
        settings.enabledUids = settings.enabledUids.plus1(uid);
        put(settings);
    }

    /**
     * Removes a user from enabled user list.
     * Note: this doesn't change the behavior if the type was not changed to ENABLED_FOR_SOME. It can be done using
     * {@link #enableNotificationsForSome(String, String)}
     * @param serviceName service name for notification type, example: disk
     * @param notificationTypeName notification type, example: shared_folder_invite_received
     */
    public void removeUser(String serviceName, String notificationTypeName, String uid) {
        NotificationToggleSettings settings = getOrCreateSettings(serviceName, notificationTypeName);
        settings.enabledUids = settings.enabledUids.minus1(uid);
        put(settings);
    }

    /**
     * Removes all users from enabled user list.
     * Note: this doesn't change the behavior if the type was not changed to ENABLED_FOR_SOME. It can be done using
     * {@link #enableNotificationsForSome(String, String)}
     * @param serviceName service name for notification type, example: disk
     * @param notificationTypeName notification type, example: shared_folder_invite_received
     */
    public void removeAllUsers(String serviceName, String notificationTypeName) {
        NotificationToggleSettings settings = getOrCreateSettings(serviceName, notificationTypeName);
        settings.enabledUids = Cf.set();
        put(settings);
    }

    /**
     * Checks if the notification is enabled for user. If no settings were saved previously, creates them.
     * @param type type of notification
     * @param uid user id
     */
    public boolean isNotificationEnabled(NotificationType type, String uid) {
        NotificationToggleSettings settings = getOrCreateAndPutSettings(type);

        switch (settings.type) {
            case DISABLED_FOR_ALL:
                return false;
            case ENABLED_FOR_ALL:
                return true;
            case ENABLED_FOR_SOME:
                return settings.enabledUids.containsTs(uid);
            default:
                logger.warn("Unknown toggle type {} for notification type {} and service {}", settings.type, type.name,
                        type.getService().name);
                return false;
        }
    }

    /**
     * Returns current settings for given notification
     * @param serviceName service name for notification type, example: disk
     * @param notificationTypeName notification type, example: shared_folder_invite_received
     */
    public Option<NotificationToggleSettings> getSettings(String serviceName, String notificationTypeName) {
        return getO(id(serviceName, notificationTypeName));
    }

    /**
     * Returns current settings for all notifications in given service
     * @param serviceName service name for notification type, example: disk
     */
    public CollectionF<NotificationToggleSettings> getSettingsForService(String serviceName) {
        return getAll().filter(settings -> settings.serviceName.equals(serviceName));
    }

    private void changeType(String serviceName, String notificationTypeName, NotificationToggleType type) {
        NotificationToggleSettings settings = getOrCreateSettings(serviceName, notificationTypeName);
        settings.type = type;
        put(settings);
    }

    private NotificationToggleSettings getOrCreateSettings(String serviceName, String notificationTypeName) {
        return getSettings(serviceName, notificationTypeName)
                .getOrElse(createDefaultSettings(serviceName, notificationTypeName));
    }

    private NotificationToggleSettings getOrCreateAndPutSettings(NotificationType type) {
        String serviceName = type.getService().name;
        String notificationTypeName = type.name;

        Option<NotificationToggleSettings> settingsO = getO(id(serviceName, notificationTypeName));
        if (settingsO.isPresent()) {
            return settingsO.get();
        }

        NotificationToggleSettings settings = createDefaultSettings(serviceName, notificationTypeName);
        put(settings);
        return settings;
    }

    private NotificationToggleSettings createDefaultSettings(String serviceName, String notificationTypeName) {
        return new NotificationToggleSettings(serviceName, notificationTypeName,
                NotificationToggleType.ENABLED_FOR_ALL, Cf.set());
    }

    private static String id(NotificationToggleSettings settings) {
        return id(settings.serviceName, settings.notificationTypeName);
    }

    private static String id(String serviceName, String notificationTypeName) {
        return serviceName + ID_SEPARATOR + notificationTypeName;
    }
}
