package ru.yandex.chemodan.app.notifier.promo;

import java.util.concurrent.atomic.AtomicLong;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.ForZkScript;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.push.NotificationPushInfo;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.worker.task.SendPromoPushTask;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ParallelStreamSupplier;

/**
 * @author akirakozov
 * @author vpronto
 */
@RequiredArgsConstructor
public class PromoActionManager {
    private static final Logger logger = LoggerFactory.getLogger(PromoActionManager.class);

    private static final BenderJsonParser<MetadataWrapper> metadataJsonParser = Bender.jsonParser(MetadataWrapper.class);

    private final NotificationPushManager pushManager;
    private final NotificationRecordTypeManager typeManager;
    private final BazingaTaskManager bazingaTaskManager;

    //for z-admin
    public void sendPromo(String userId) {
        sendPromo(DataApiUserId.parse(userId), Option.empty(), Option.empty());
    }

    public void sendPromoWithLink(String userId, String link) {
        sendPromo(DataApiUserId.parse(userId), Option.empty(), Option.of(new MetadataWrapper(Cf.map(
                "mobile_link", new MetadataEntity(Cf.map(
                        "link", link,
                        "type", "link"
                ))
        ))));
    }

    /**
     * @param userId
     * @param platform - ios/android. if empty - push will be send to both platforms
     * @param metadata
     */
    public void sendPromo(DataApiUserId userId, Option<String> platform, Option<MetadataWrapper> metadata) {
        logger.info("About to send to {}", userId);
        //push to new IOS
        if (!platform.isPresent() || platform.isSome("both") || "ios".equals(platform.get().toLowerCase())) {
            NotificationType ios2 = typeManager.resolveRecordType(DiskNotifications.PROMO_IOS_2);
            pushManager.pushNotificationWithDelay(userId, new NotificationPushInfo(ios2.consMainTemplate(),
                            NotificationActor.YA_DISK,
                            Instant.now()),
                    metadata.getOrElse(ios2.defaultMetadata));
        }

        //push to all android
        if (!platform.isPresent() || platform.isSome("both") || "android".equals(platform.get().toLowerCase())) {
            NotificationType android = typeManager.resolveRecordType(DiskNotifications.PROMO_ANDROID);
            pushManager.pushNotificationWithDelay(userId, new NotificationPushInfo(android.consMainTemplate(),
                            NotificationActor.YA_DISK,
                            Instant.now()),
                    metadata.getOrElse(android.defaultMetadata));
        }
        logger.info("Pushed to uid={}, platform={}", userId, platform.getOrElse("both"));
    }

    @ForZkScript
    public void schedulePromoActionTasksFromFile(String fileName, Instant startTime, Instant endTime) {
        long intervalInMinutes = new Duration(startTime, endTime).getStandardMinutes();
        schedulePromoActionTasksFromFile(new File2(fileName), startTime, intervalInMinutes);
    }

    public void schedulePromoActionTasksFromFile(File2 file, Instant startTime, long interval) {

        Thread thread = new Thread(() -> {
            ListF<SendPromoPushTask.Parameters> tasks = file.readLines().flatMap(PromoActionManager::safeParse);
            int size = tasks.size();
            logger.info("Expected count of tasks: {}", size);
            AtomicLong pushNum = new AtomicLong();
            Instant processStart = Instant.now();
            ParallelStreamSupplier.builder()
                    .threadNameFactory("promo").maxThreadCount(200).minThreadCount(200)
                    .build()
                    .supply(tasks.stream().map(parameters -> () -> {
                        long num = pushNum.incrementAndGet();
                        Instant time = startTime.plus(Duration.standardMinutes((long) (((double) num * interval) / size)));
                        schedulePromoActionTask(parameters, time);
                        return num;
                    }));
            logger.info("Processed: {} for: {}", pushNum.get(), new Duration(processStart, Instant.now()));
        });
        thread.setDaemon(true);
        thread.start();
    }

    public void schedulePromoActionTask(SendPromoPushTask.Parameters parameters, Instant time) {
        try {
            logger.info("About to schedule uuid={}, parameters={}, time={}", parameters.uid, parameters, time);
            bazingaTaskManager.schedule(new SendPromoPushTask(parameters), time);
        } catch (Exception e) {
            logger.info("Couldn't schedule uuid={}, parameters={}, time={}", parameters.uid, parameters, time);
        }
    }

    private static Option<SendPromoPushTask.Parameters> safeParse(String uuid) {
        try {
            String[] parts = uuid.split("\t");
            DataApiUserId uid = DataApiUserId.parse(parts[0]);
            Option<String> platform = Option.empty();
            Option<MetadataWrapper> metadata = Option.empty();

            if (parts.length > 1) {
                platform = Option.of(parts[1]);
            }
            if (parts.length > 2) {
                if (parts[2].startsWith("{")) {
                    metadata = Option.of(metadataJsonParser.parseJson(parts[2]));
                } else {
                    metadata = Option.of(new MetadataWrapper(Cf.map(
                            "mobile_link", new MetadataEntity(Cf.map(
                                    "link", parts[2],
                                    "type", "link"
                            )))));
                }
            }

            return Option.of(new SendPromoPushTask.Parameters(
                    uid,
                    platform,
                    metadata
            ));
        } catch (Exception e) {
            logger.warn("can't parse uuid: {}", uuid, e.getMessage());
            return Option.empty();
        }
    }
}
