package ru.yandex.chemodan.app.notifier.push;


import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.xiva.DataApiXivaUtils;
import ru.yandex.chemodan.app.notifier.locale.LocaleManager;
import ru.yandex.chemodan.app.notifier.log.NotifierEvent;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndGroup;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskGroups;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskServices;
import ru.yandex.chemodan.app.notifier.push.body.IosAlertPushBody;
import ru.yandex.chemodan.app.notifier.push.body.LanguageMap;
import ru.yandex.chemodan.app.notifier.push.body.NotificationDesktopPushBody;
import ru.yandex.chemodan.app.notifier.push.body.NotificationMobilePushBody;
import ru.yandex.chemodan.app.notifier.push.body.XivaBodyWithTitleAndText;
import ru.yandex.chemodan.app.notifier.push.body.XivaPushSpecificParamsCreator;
import ru.yandex.chemodan.app.notifier.push.filter.NotificationPushFilter;
import ru.yandex.chemodan.app.notifier.push.filter.NotificationPushFilterManager;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannelGroup;
import ru.yandex.chemodan.app.notifier.tanker.TankerMessageKey;
import ru.yandex.chemodan.app.notifier.utils.UserLanguageHelper;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.DiskMetadataProcessorManager;
import ru.yandex.chemodan.app.notifier.worker.task.SendPushTask;
import ru.yandex.chemodan.util.AppVersionRange;
import ru.yandex.chemodan.xiva.XivaEvent;
import ru.yandex.chemodan.xiva.XivaPushBody;
import ru.yandex.chemodan.xiva.XivaPushService;
import ru.yandex.chemodan.xiva.XivaSendResponse;
import ru.yandex.chemodan.xiva.XivaSingleTokenClient;
import ru.yandex.chemodan.xiva.XivaSubscriptionFilter;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.BenderSettings;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.internal.pojo.PojoMarshallingJsonCallback;
import ru.yandex.misc.bender.internal.pojo.PojoMarshallingXmlCallback;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.MarshallerContext;
import ru.yandex.misc.bender.serialize.ToFieldMarshallerSupport;
import ru.yandex.misc.bender.serialize.ToFieldWithCallbackMarshaller;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.xml.stream.XmlWriter;

/**
 * @author akirakozov
 */
@AllArgsConstructor
public class NotificationPushManager {
    private static final Logger logger = LoggerFactory.getLogger(NotificationPushManager.class);

    private DynamicProperty<Integer> sendPushTaskTtlMinutes = DynamicProperty.cons("send-push-task-ttl[minutes]", 60);

    private static final DateTimeFormatter DATE_TIME_PATTERN =
            DateTimeFormat.forPattern("yyyy-MM-dd'T'hh:mm:ssZZ").withZoneUTC();

    private static final SetF<String> LENTA_URL_EXCLUDED_FIELDS =
            Cf.set("block-type", "type", "action", "uid", "login");

    private static final ListF<String> PUSH_TITLES =
            Cf.list(", смотрите!", ", помните?", ", ваши фото", ", вас ждут фото", ", вам подборка");

    private static final ListF<String> NON_CUSTOM_NOTIFICATION_TYPES = //excluding thematic, "n years ago" and geo types
            Cf.set(DiskNotifications.PHOTO_SELECTION_COOL_LENTA,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_ONE_DAY,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEKEND,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEKEND_CROSS_YEAR,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEKEND_CROSS_MONTH,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK_CROSS_MONTH,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK_CROSS_YEAR,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_MONTH,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_SEASON,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WINTER,
                    DiskNotifications.PHOTO_SELECTION_COOL_LENTA_YEAR)
                    .map(ServiceAndType::getType);

    private static final BenderMapper mapper = new BenderMapper(new BenderConfiguration(
            new BenderSettings(MembersToBind.ALL_FIELDS),
            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                    .add(LentaUrlParams.class,
                            new LentaUrlParamsMarshaller()
                    )
                    .build()
    ));

    private final TextMessageGenerator textMessageGenerator;
    private final UserLanguageHelper userLanguageHelper;
    private final XivaSingleTokenClient client;
    private final LocaleManager localeManager;
    private final PushSettingsManager pushSettingsManager;
    private final BazingaTaskManager bazingaTaskManager;
    private final NotificationPushFilterManager pushFilterManager;
    private final String lentaBaseUrl;
    private final Option<Integer> maxTitleLength;

    private final DynamicProperty<Boolean> nameInPushEnabled = new DynamicProperty<>("cool-lenta-name-in-push-enabled", false);

    private final DynamicProperty<ListF<String>> groupsWithoutMobilePreview = DynamicProperty.cons(
            "notifier-groups-without-mobile-preview", Cf.list(DiskGroups.PROMO_WITH_LINK.getGroup()));

    private final DynamicProperty<Long> brightPushInitialDelay =
            DynamicProperty.cons("notifier-bright-push-initial-delay", 0L);

    private final DynamicProperty<Double> brightPushBaseDelayRatio =
            DynamicProperty.cons("notifier-bright-push-base-delay-ratio", 0.3d);

    public NotificationPushManager(
            XivaSingleTokenClient client,
            UserLanguageHelper userLanguageHelper,
            LocaleManager localeManager,
            PushSettingsManager pushSettingsManager,
            BazingaTaskManager bazingaTaskManager,
            NotificationPushFilterManager pushFilterManager,
            String lentaBaseUrl,
            int maxTitleLength)
    {
        this.client = client;
        this.userLanguageHelper = userLanguageHelper;
        this.localeManager = localeManager;
        this.textMessageGenerator = new TextMessageGenerator();
        this.bazingaTaskManager = bazingaTaskManager;
        this.lentaBaseUrl = lentaBaseUrl;
        this.maxTitleLength = (maxTitleLength > 0) ? Option.of(maxTitleLength) : Option.empty();
        this.pushSettingsManager = pushSettingsManager;
        this.pushFilterManager = pushFilterManager;

    }

    public PushNotificationJob consJob(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata) {
        return new PushNotificationJob(uid, info, metadata);
    }

    public void pushNotificationWithDelay(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata) {
        consJob(uid, info, metadata)
                .pushWithDelay();
    }

    public void pushNotificationWithDelay(
            DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata, Instant now)
    {
        consJob(uid, info, metadata)
                .pushWithDelay(now);
    }

    public void pushNotification(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata,
            ListF<String> tags)
    {
        new PushNotificationJob(uid, info, metadata, tags)
                .push();
    }

    public void pushNotification(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata,
        ListF<String> tags, SetF<GlobalSubscriptionChannel> channels)
    {
        new PushNotificationJob(uid, info, metadata, tags)
                .push(channels);
    }

    public void sendIosAlertPush(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata,
        String channelGroup)
    {
        new PushNotificationJob(uid, info, metadata, Cf.list(GlobalSubscriptionChannel.IOS.value()))
                .sendIosAlertPush(channelGroup);
    }

    String createUrl(MetadataWrapper metadata) {
        Option<String> jsonParams = getActionParamsAsJsonString(metadata);
        if (jsonParams.isPresent()) {
            MetadataEntity entity = metadata.getEntityFieldsO(MetadataEntityNames.ACTION).get();
            return UrlUtils.addParameter(lentaBaseUrl, "uid", entity.get("uid"), "login", entity.get("login"),
                    "feedBlockData", jsonParams.get());
        } else {
            return lentaBaseUrl;
        }
    }

    Option<String> getPreviewFilePath(MetadataWrapper metadata) {
        return metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_PATH);
    }

    Option<String> getPreviewResourceId(MetadataWrapper metadata) {
        return metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_RESOURCE_ID);
    }

    private Option<String> getActionParamsAsJsonString(MetadataWrapper metadata) {
        Option<MetadataEntity> entityO = metadata.getEntityFieldsO(MetadataEntityNames.ACTION);

        if (!entityO.isPresent()) {
            logger.warn("Couldn't find ACTION in metadata: " + metadata);
            return Option.empty();
        }

        MetadataEntity entity = entityO.get();
        if (entity.isEmpty() || StringUtils.isEmpty(entity.get("block-type"))) {
            logger.warn("Couldn't find block-type in metadata: " + entity);
            return Option.empty();
        }

        String blockType = entity.get("block-type");
        MapF<String, String> params = entityO.get().fields
                .filterKeys(k -> !LENTA_URL_EXCLUDED_FIELDS.containsTs(k))
                .plus1("type", blockType);

        if ("photo_remind_hidden_block".equals(params.getOrElse("type", ""))) {
            params.put("type", "photo_remind_block");
        }

        return Option.of(new String(mapper.serializeJson(new LentaUrlParams(params))));
    }

    private Option<String> extractPreview(MetadataWrapper metadata, String group) {
        return !groupsWithoutMobilePreview.get().containsTs(group) ?
            metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_URL) : Option.empty();
    }

    private Option<String> fetchLink(String mobileLink, NotifierLanguage lang, MetadataWrapper metadata) {
        Option<String> link = metadata.getEntityField(mobileLink, lang.value() + "_link");
        return link.isPresent() ? link : metadata.getEntityField(mobileLink, "link");
    }

    public class PushNotificationJob {
        DataApiUserId uid;

        NotificationPushInfo info;

        MetadataWrapper metadata;

        ListF<String> tags;

        boolean forceNoDelay;

        PushNotificationJob(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata, ListF<String> tags) {
            this.uid = uid;
            this.info = info;
            this.metadata = metadata;
            this.tags = tags;

            this.forceNoDelay = false;
        }

        PushNotificationJob(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata) {
            this(uid, info, metadata, Cf.list());
        }

        void pushWithDelay() {
            pushWithDelay(Instant.now());
        }

        void pushWithDelay(Instant now) {
            if (now.isBefore(Instant.now().minus(Duration.standardHours(1)))) {
                forceNoDelay = true;
                push();
                return;
            }

            if (getDelay().isLongerThan(Duration.ZERO)) {
                schedule(now);
            } else {
                forceNoDelay = true;
                push();
            }
        }

        void schedule(Instant now) {
            schedule(now, Option.empty());
        }

        void schedule(Instant now, Option<SetF<GlobalSubscriptionChannel>> channels) {
            Instant scheduleTime = now.plus(getDelay());
            Instant deadline = getDeadline(scheduleTime);
            bazingaTaskManager.schedule(new SendPushTask(uid, info, metadata, channels, deadline), scheduleTime);
        }

        private Duration getDelay() {
            return info.getTemplate().getDelay();
        }

        public void push() {
            push(Option.empty());
        }

        public void push(Option<SetF<GlobalSubscriptionChannel>> channels) {
            push(channels.getOrElse(pushSettingsManager.getEnabledChannels(uid, info)));
        }

        public void push(SetF<GlobalSubscriptionChannel> channels) {
            prepareEvents(channels.toList())
                    .forEach(this::sendPushMessageIfNotFiltered);

            if (isBrightIosSendNeeded(channels)) {
                long delay = getBrightPushDelay();
                if (delay > 0) {
                    schedule(Instant.now().plus(delay), Option.of(Cf.set(GlobalSubscriptionChannel.IOS_BRIGHT)));
                } else {
                    push(Cf.set(GlobalSubscriptionChannel.IOS_BRIGHT));
                }
            }
        }

        private long getBrightPushDelay() {
            if (forceNoDelay) {
                return 0;
            }
            return Math.round(brightPushInitialDelay.get() + getDelay().getMillis() * brightPushBaseDelayRatio.get());
        }

        private boolean isBrightIosSendNeeded(SetF<GlobalSubscriptionChannel> channels) {
            return channels.containsTs(GlobalSubscriptionChannel.IOS)
                    && !channels.containsTs(GlobalSubscriptionChannel.IOS_BRIGHT);
        }

        Tuple2List<GlobalSubscriptionChannelGroup, XivaEvent> prepareEvents(ListF<GlobalSubscriptionChannel> channels) {
            if (!info.template.getService().equals(DiskServices.DISK)) {
                // don't send any pushes for non-disk services, they will handle that themselves
                return Tuple2List.arrayList();
            }

            return channels.map(GlobalSubscriptionChannel::group)
                    .stableUnique()
                    .zipWith(group -> consEvent(group, channels.filter(channel -> channel.belongsTo(group))));
        }

        XivaEvent consEvent(GlobalSubscriptionChannelGroup channelGroup, ListF<GlobalSubscriptionChannel> channels) {
            return new XivaEvent.Builder()
                    .withRecipient(DataApiXivaUtils.toXivaRecipient(uid))
                    .withTags(tags.isNotEmpty() ? tags : channels.map(GlobalSubscriptionChannel::value))
                    .withEvent(channelGroup.eventType)
                    .withBody(consBody(channelGroup))
                    .build();
        }

        private XivaPushBody consBody(GlobalSubscriptionChannelGroup channelGroup) {
            switch (channelGroup) {
                case WEB_DESKTOP_V1:
                    return createWebDesktopBody();

                case MOBILE_V1:
                    return createMobileBody();

                case MOBILE_V2:
                    return createMobileBodyV2(false);

                case MOBILE_V2_BRIGHT:
                    return createMobileBodyV2(true);

                default:
                    throw new IllegalArgumentException("Unexpected notifications channel group: " + channelGroup);
            }
        }

        private XivaPushBody createWebDesktopBody() {
            Tuple2<LanguageMap, LanguageMap> titlesAndTexts = getPushMessageTexts();
            LanguageMap titles = titlesAndTexts._1;
            LanguageMap texts = titlesAndTexts._2;
            return new NotificationDesktopPushBody(
                    info.template, info.actor.toString(),
                    metadata.getEntityField(MetadataEntityNames.DESKTOP_URL, DiskMetadataProcessorManager.DESKTOP_URL_LINK_VALUE)
                            .getOrElse(() -> createUrl(metadata)),
                    info.ctime.toString(DATE_TIME_PATTERN), titles, texts,

                    extractPreview(metadata, info.template.getGroup()),
                    getPreviewFilePath(metadata),
                    getPreviewResourceId(metadata));
        }

        private Tuple2<LanguageMap, LanguageMap> getPushMessageTexts() {
            return new LocalizedTitleAndTextGenerator()
                    .getAll();
        }

        private XivaPushBody createMobileBody() {
            return createMobileBody(GlobalSubscriptionChannelGroup.MOBILE_V1, Option.empty(), Cf.map());
        }

        private XivaPushBody createMobileBodyV2(boolean bright) {
            return createMobileBody(
                    bright ? GlobalSubscriptionChannelGroup.MOBILE_V2_BRIGHT : GlobalSubscriptionChannelGroup.MOBILE_V2,
                    getActionParamsAsJsonString(metadata),
                    XivaPushSpecificParamsCreator.createVersionCompatibilityParams(info.template, NotificationEvents.MOBILE_V2)
            );
        }

        private XivaPushBody createMobileBody(
                GlobalSubscriptionChannelGroup channelGroup,
                Option<String> actionData,
                MapF<XivaPushService, AppVersionRange> specificParams)
        {
            Tuple2<String, String> mobileTexts2 = getMobilePushMessageTexts();
            String title = mobileTexts2._1;
            String text = mobileTexts2._2;
            return new NotificationMobilePushBody(
                    channelGroup, uid.toPassportUid(), info.template,
                    extractLinkForMobile(), title, text,
                    extractPreview(metadata, info.template.getGroup()).getOrElse(""),
                    actionData,
                    specificParams
            );
        }

        private Tuple2<String, String> getMobilePushMessageTexts() {
            NotifierLanguage lang = userLanguageHelper.getUserLanguageOrDefault(uid);
            return new LocalizedTitleAndTextGenerator().getForMobile(lang, uid);
        }

        void sendPushMessageIfNotFiltered(GlobalSubscriptionChannelGroup channelGroup, XivaEvent event) {
            NotificationPushFilter filterResult = pushFilterManager.getFilter(uid, channelGroup, info, event);
            if (!filterResult.isEnabled()) {
                return;
            }

            if (filterResult.getIncludeSubscriptionIds().isPresent()) {
                event = event.withSubscriptions(
                        XivaSubscriptionFilter.builder()
                                .subscriptionIds(filterResult.getIncludeSubscriptionIds())
                                .build()
                );
            }

            sendPushMessage(channelGroup, event);
        }

        //TODO: refactoring
        void sendIosAlertPush(String notificationEvents) {
            IosAlertPushBody body = new IosAlertPushBody(info, getMobilePushMessageTexts(), notificationEvents,
                    extractLinkForMobile());
            XivaEvent xivaEvent = new XivaEvent(DataApiXivaUtils.toXivaRecipient(uid), notificationEvents, tags)
                    .withBody(body);
            sendPushMessage(GlobalSubscriptionChannelGroup.IOS_NON_SILENT, xivaEvent);
        }

        Option<String> extractLinkForMobile() {
            ServiceAndGroup group = info.template.getGroupName();
            NotifierLanguage lang = userLanguageHelper.getUserLanguageOrDefault(uid);
            if (group.equals(DiskGroups.COMMENTS) || group.equals(DiskGroups.LIKES)) {
                return metadata.getEntityField("comment", "link")
                        .orElse(() -> metadata.getEntityField("entity", "short_url"));
            } else if (group.equals(DiskGroups.AUTOUPLOAD)) {
                return Option.of(createUrl(metadata));
            } else if (group.equals(DiskGroups.PROMO_WITH_LINK)) {
                return fetchLink(MetadataEntityNames.MOBILE_LINK, lang, metadata)
                        .orElse(() -> fetchLink("mobile-link", lang, metadata));
            } else {
                return Option.empty();
            }
        }

        void sendPushMessage(GlobalSubscriptionChannelGroup channelGroup, XivaEvent xivaEvent) {
            try {
                XivaSendResponse response = client.send(xivaEvent);

                Tuple2<String, String> titleAndText = xivaEvent.getBody()
                        .filter(body -> body instanceof XivaBodyWithTitleAndText)
                        .cast(XivaBodyWithTitleAndText.class)
                        .map(XivaBodyWithTitleAndText::getTitleAndText)
                        .getOrElse(Tuple2.tuple("failed", "failed"));

                NotifierEvent.sendPushMessage(uid, info.template, channelGroup.logEventType, response.headers,
                        metadata, xivaEvent.getTags(), titleAndText._1, titleAndText._2)
                        .log();
            } catch (Exception e) {
                logger.warn("Couldn't send push for uid: " + uid, e);
            }
        }

        private class LocalizedTitleAndTextGenerator {
            LocalizedMessageGenerator titleGenerator = new LocalizedMessageGenerator(
                    info.template.getTitleMessageKey(),
                    Option.empty(),
                    maxTitleLength
            );

            LocalizedMessageGenerator textGenerator = new LocalizedMessageGenerator(
                    info.template.getShortMessageKey(),
                    info.count,
                    Option.empty()
            );

            private Tuple2<String, String> getForMobile(NotifierLanguage lang, DataApiUserId uid) {
                final String title = generateTitleWithName(lang, uid).getOrElse(titleGenerator.getOrEmpty(lang));
                final String text = textGenerator.getOrEmpty(lang);
                return Tuple2.tuple(title, text);
            }

            private Tuple2<String, String> get(NotifierLanguage lang) {
                return Tuple2.tuple(titleGenerator.getOrEmpty(lang), textGenerator.getOrEmpty(lang));
            }

            private Tuple2<LanguageMap, LanguageMap> getAll() {
                return Tuple2.tuple(titleGenerator.getAll(), textGenerator.getAll());
            }

            private Option<String> generateTitleWithName(NotifierLanguage lang, DataApiUserId uid) {
                if (!nameInPushEnabled.get() ||
                        lang != NotifierLanguage.RUSSIAN ||
                        !NON_CUSTOM_NOTIFICATION_TYPES.containsTs(info.template.getType())) {
                    return Option.empty();
                }
                String firstName = userLanguageHelper.getFirstNameO(uid).getOrElse("");
                /*  https://wiki.yandex-team.ru/users/slukyanenko/Dobavit-imja-polzovatelja-v-push/
                     оставлять старый title, если:
                     имя пользователя содержит цифры
                     имя пользователя содержит пробелы
                     имя пользователя содержит знаки препинания
                     имя пользователя состоит из 1 символа
                     тайтл пуша с именем пользователя получается больше 20 символов
                */
                if (firstName.length() < 2 || firstName.matches(".*[\\p{Space}\\p{Punct}\\p{Digit}].*")) {
                    return Option.empty();
                }
                final String title = firstName + Random2.R.randomElement(PUSH_TITLES);
                if (title.length() > 20) {
                    return Option.empty();
                }
                return Option.of(title);
            }

        }

        private class LocalizedMessageGenerator {
            final MapF<NotifierLanguage, LocalizedMessage> languageMessages;

            final Option<Integer> maxLength;

            LocalizedMessageGenerator(TankerMessageKey key, Option<Long> number, Option<Integer> maxLength) {
                this.languageMessages = localeManager.getAllLocalizations(key, number, Option.of(metadata));
                this.maxLength = maxLength;
            }

            String getOrEmpty(NotifierLanguage lang) {
                return languageMessages.getO(lang)
                        .map(this::generate)
                        .getOrElse("");
            }

            LanguageMap getAll() {
                return new LanguageMap(languageMessages.mapValues(this::generate));
            }

            String generate(LocalizedMessage m) {
                return textMessageGenerator.generateSimpleTextMessage(m, metadata, maxLength);
            }
        }
    }

    @BenderBindAllFields
    private static class LentaUrlParams {
        private final MapF<String, String> params;

        public LentaUrlParams(MapF<String, String> params) {
            this.params = params;
        }
    }

    private static class LentaUrlParamsMarshaller
            extends ToFieldMarshallerSupport implements ToFieldWithCallbackMarshaller
    {
        @Override
        public void writeJsonToField(BenderJsonWriter writer, Object fieldValue, MarshallerContext context) {
            writer.writeObjectStart();
            ((LentaUrlParams)fieldValue).params.forEach((key, value) -> {
                writer.writeFieldName(key);
                writer.writeString(value);
            });
            writer.writeObjectEnd();
        }

        @Override
        public void writeJsonToFieldWithCallback(BenderJsonWriter writer, Object fieldValue, PojoMarshallingJsonCallback callback, MarshallerContext context) {
            writeJsonToField(writer, fieldValue, context);
        }

        @Override
        public void writeXmlToFieldWithCallback(XmlWriter writer, Object fieldValue, PojoMarshallingXmlCallback callback, MarshallerContext context) {
            writeXmlToField(writer, fieldValue, context);
        }
    }

    private Instant getDeadline(Instant scheduleTime) {
        return scheduleTime.plus(Duration.standardMinutes(sendPushTaskTtlMinutes.get()));
    }
}
