package ru.yandex.chemodan.app.notifier.push;

import java.net.URI;
import java.util.List;

import com.xebialabs.restito.semantics.Call;
import org.glassfish.grizzly.http.Method;
import org.glassfish.grizzly.http.util.HttpStatus;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.xiva.XivaContextConfiguration;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.admin.dao.test.NotifierJdbcDaoTestsContextConfiguration;
import ru.yandex.chemodan.app.notifier.locale.LocaleManager;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationType;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.push.filter.NotificationPushFilter;
import ru.yandex.chemodan.app.notifier.push.filter.NotificationPushFilterManager;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.settings.NotifierCollectionUtils;
import ru.yandex.chemodan.app.notifier.utils.BlackboxMultiplexed;
import ru.yandex.chemodan.app.notifier.utils.UserLanguageHelper;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorTestUtils;
import ru.yandex.chemodan.test.TestHelper;
import ru.yandex.chemodan.util.JsonAssert;
import ru.yandex.chemodan.util.test.StubServerUtils;
import ru.yandex.chemodan.util.test.TestUser;
import ru.yandex.chemodan.xiva.XivaSingleTokenClient;
import ru.yandex.chemodan.zk.configuration.ImportZkEmbeddedConfiguration;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.test.Assert;

import static com.xebialabs.restito.builder.stub.StubHttp.whenHttp;
import static com.xebialabs.restito.semantics.Action.header;
import static com.xebialabs.restito.semantics.Action.status;
import static com.xebialabs.restito.semantics.Condition.endsWithUri;
import static com.xebialabs.restito.semantics.Condition.method;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.when;

/**
 * @author akirakozov
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
        XivaContextConfiguration.class,
        NotifierPushManagerTestContext.class,
        NotifierJdbcDaoTestsContextConfiguration.class,
        NotificationTypeManagerTestContextConfiguration.class
})
@ActivateNotificationEmbeddedPg
@ImportZkEmbeddedConfiguration
public class NotificationPushManagerTest {
    private static final String COMMENT_URL = "http://some.comment.ru/testlink";
    private static final String LENTA_URL =
            "https://disk.dst.yandex.ru/client/feed/item?uid=123456&login=username&feedBlockData=" +
                    "%7B%22modify_uid%22%3A%224001050109%22%2C%22mtime%22%3A%221231434%22%2C%22type%22%3A" +
                    "%22public_resource_owned%22%2C%22file_id%22%3A%22file_id_111%22%7D";
    private static final DataApiPassportUserId TEST_UID = new DataApiPassportUserId(123);

    @Autowired
    private XivaSingleTokenClient client;
    @Autowired
    private LocaleManager localeManager;
    @Autowired
    private NotificationRecordTypeManager typeManager;

    @Mock
    PushSettingsManager pushSettingsManager;
    @Mock
    private BlackboxMultiplexed blackbox;
    @Mock
    private BazingaTaskManager bazingaTaskManager;
    @Mock
    NotificationPushFilterManager pushFilterManager;

    // SUT
    private NotificationPushManager notificationPushManager;

    @Value("${xiva.host}")
    private String xivaUrl;

    @Before
    public void init() {
        TestHelper.initialize();
        MockitoAnnotations.initMocks(this);

        when(pushSettingsManager.getEnabledChannels(eq(TEST_UID), any()))
                .thenReturn(GlobalSubscriptionChannel.PUBLIC);

        when(pushFilterManager.getFilter(any(), any(), any(), any()))
                .thenReturn(
                        NotificationPushFilter.includeSubscriptionIds(Cf.list("123"))
                );

        blackbox = MetadataProcessorTestUtils.createBlackboxMock();
        UserLanguageHelper userLanguageHelper = new UserLanguageHelper(blackbox);

        notificationPushManager = new NotificationPushManager(
                client, userLanguageHelper, localeManager, pushSettingsManager, bazingaTaskManager, pushFilterManager,
                 "https://disk.dst.yandex.ru/client/feed/item", 23);
    }

    @Test
    public void sendPushNotifications() {
        NotificationType type = typeManager.resolveRecordType(DiskNotifications.COMMENT_LIKE);
        NotificationPushInfo info = new NotificationPushInfo(
                type.consMainTemplate(), NotificationActor.consFromUid(920795614889183L),
                Instant.parse("2018-06-29T07:32:52+00:00"),
                Option.of("1234567890"));
        new SendPushTestHelper()
                .info(info)
                .exerciseAndAssertPushEquals(
                        "sendPushNotificationsDesktop.json",
                        "sendPushNotificationsMobile.json",
                        "sendPushNotificationsMobileV2.json",
                        "sendPushNotificationsMobileV2Bright.json"
                );
    }

    @Test
    public void sendPhotoReminder() {
        NotificationType type = typeManager.resolveRecordType(DiskNotifications.PHOTO_REMINDER);
        NotificationPushInfo info = new NotificationPushInfo(
                type.consMainTemplate(), NotificationActor.consFromUid(474938532139016L),
                Instant.parse("2018-06-29T11:56:44+00:00"),
                Option.of("1234567890"),
                Option.of(3L));
        new SendPushTestHelper()
                .info(info)
                .subscriptionChannels(
                        GlobalSubscriptionChannel.DESKTOP, GlobalSubscriptionChannel.WEB,
                        GlobalSubscriptionChannel.ANDROID, GlobalSubscriptionChannel.IOS,
                        GlobalSubscriptionChannel.IOS_BRIGHT
                )
                .exerciseAndAssertPushEquals(
                        "sendPhotoReminderDesktop.json",
                        "sendPhotoReminderMobile.json",
                        "sendPhotoReminderMobileBright.json"
                );
    }

    @Test
    public void createUrl() {
        MetadataWrapper metadata = createMetadata();
        String url = notificationPushManager.createUrl(metadata);
        Assert.equals(LENTA_URL, url);
    }

    @Test
    public void sendAutouploadPushNotification() {
        NotificationType type = typeManager.resolveRecordType(DiskNotifications.AUTOUPLOAD);
        NotificationPushInfo info = new NotificationPushInfo(
                type.consMainTemplate(), NotificationActor.consFromUid(575018490952315L),
                Instant.parse("2018-06-30T12:07:31+00:00"),
                Option.of("1234567890"),
                Option.of(8L));
        new SendPushTestHelper()
                .info(info)
                .exerciseAndAssertPushEquals(
                        "sendAutouploadPushNotificationDesktop.json",
                        "sendAutouploadPushNotificationMobile.json",
                        "sendAutouploadPushNotificationMobileV2.json",
                        "sendAutouploadPushNotificationMobileV2Bright.json"
                );
    }

    @Test
    public void sendSharedFolderPushNotification() {
        NotificationType type = typeManager.resolveRecordType(DiskNotifications.SHARED_FOLDER_FILE_UPDATE);
        NotificationPushInfo info = new NotificationPushInfo(
                type.consMainTemplate(), NotificationActor.consFromUid(804424570742975L),
                Instant.parse("2018-06-30T06:39:13+00:00"),
                Option.of("1234567890"),
                Option.of(8L));
        new SendPushTestHelper()
                .info(info)
                .subscriptionChannels(GlobalSubscriptionChannel.WEB)
                .exerciseAndAssertPushEquals("sendSharedFolderPushNotification.json");
    }

    @Test
    @SuppressWarnings("deprecation")
    public void sendUnlimPromoPushNotification() {
        new SendPushTestHelper()
                .promoAndroidInfo()
                .subscriptionChannels(GlobalSubscriptionChannel.MOBILE)
                .exerciseAndAssertPushEquals("sendUnlimPromoPushNotification.json");
    }

    @Test
    public void sendPromoPushNotification1() {
        sendUpgradeRequiredUnlimInternal("sendPromoPushNotification1.json",
                true, "mobile_link");
    }

    @Test
    public void sendPromoPushNotification2() {
        sendUpgradeRequiredUnlimInternal("sendPromoPushNotification2.json",
                true, "mobile-link");
    }

    @Test
    public void sendPromoPushNotification3() {
        sendUpgradeRequiredUnlimInternal("sendPromoPushNotification3.json", false,
                "mobile_link");
    }

    @SuppressWarnings("deprecation")
    private void sendUpgradeRequiredUnlimInternal(String filename, boolean addLanguages, String mobileLinkEntityName) {
        new SendPushTestHelper()
                .promoAndroidInfo()
                .metadata(createUpgradeRequiredUnlimMetadata(mobileLinkEntityName, addLanguages))
                .subscriptionChannels(GlobalSubscriptionChannel.MOBILE)
                .exerciseAndAssertPushEquals(filename);
    }

    private class SendPushTestHelper {
        NotificationPushInfo info;

        MetadataWrapper metadata = createMetadata();

        SetF<GlobalSubscriptionChannel> subscriptionChannels = Cf.set();

        SendPushTestHelper promoAndroidInfo() {
            NotificationType type = typeManager.resolveRecordType(DiskNotifications.PROMO_ANDROID);
            return info(
                    new NotificationPushInfo(type.consMainTemplate(), NotificationActor.YA_DISK, Instant.now())
            );
        }

        SendPushTestHelper info(NotificationPushInfo info) {
            this.info = info;
            return this;
        }

        SendPushTestHelper metadata(MetadataWrapper metadata) {
            this.metadata = metadata;
            return this;
        }

        SendPushTestHelper subscriptionChannels(GlobalSubscriptionChannel... subscriptionChannels) {
            return subscriptionChannels(NotifierCollectionUtils.linkedSet(subscriptionChannels));
        }

        SendPushTestHelper subscriptionChannels(SetF<GlobalSubscriptionChannel> subscriptionChannels) {
            this.subscriptionChannels = subscriptionChannels;
            return this;
        }

        void exerciseAndAssertPushEquals(String... filenames) {
            if (subscriptionChannels.isNotEmpty()) {
                when(pushSettingsManager.getEnabledChannels(eq(TEST_UID), any()))
                        .thenReturn(subscriptionChannels);
            }
            StubServerUtils.withStubServer(URI.create(xivaUrl).getPort(), stubServer -> {
                whenHttp(stubServer).match(endsWithUri("/v2/send"), method(Method.POST)).then(status(HttpStatus.OK_200),
                        header("NotificationID", "58"), header("TransitID", "CGPIjd0aUuQ1"));

                notificationPushManager.pushNotificationWithDelay(TEST_UID, info, metadata);

                List<Call> calls = stubServer.getCalls();
                Assert.equals(filenames.length, calls.size());

                for (int i = 0; i < calls.size(); i++) {
                    try {
                        JsonAssert.equalsLenient(getClass(), filenames[i], calls.get(i).getPostBody());
                    } catch (AssertionError err) {
                        throw new AssertionError(filenames[i] + " " + err.getMessage(), err.getCause());
                    }
                }
            });
        }

    }

    private MetadataWrapper createMetadata() {
        MapF<String, String> actorInfo = Cf.hashMap();
        actorInfo.put("uid", TestUser.uid2.toString());
        actorInfo.put("text", "Тестер");
        MetadataEntity actorEntity = new MetadataEntity(MetadataEntityType.USER, actorInfo);

        MapF<String, String> commentInfo = Cf.hashMap();
        commentInfo.put("id", "456456456");
        commentInfo.put("text", "Вася дурак!");
        commentInfo.put("link", COMMENT_URL);
        MetadataEntity commentEntity = new MetadataEntity(MetadataEntityType.COMMENT, commentInfo);

        MapF<String, String> entityInfo = Cf.hashMap();
        entityInfo.put("id", "123456789:QWERTYUIOP12");
        entityInfo.put("name", "Some resource");
        entityInfo.put("resourcetype", "FILE");
        entityInfo.put("preview", "Some resource preview");
        MetadataEntity entityEntity = new MetadataEntity(MetadataEntityType.RESOURCE, entityInfo);

        MapF<String, String> actionInfo = Cf.hashMap();
        actionInfo.put("action", "GO_TO_LENTA");
        actionInfo.put("block-type", LentaRecordType.PUBLIC_RESOURCE_OWNED.value());
        actionInfo.put("modify_uid", "4001050109");
        actionInfo.put("mtime", "1231434");
        actionInfo.put("file_id", "file_id_111");
        actionInfo.put("uid", "123456");
        actionInfo.put("login", "username");
        MetadataEntity actionEntity = new MetadataEntity(MetadataEntityType.ACTION, actionInfo);

        MapF<String, String> countInfo = Cf.hashMap();
        countInfo.put("text", "10");
        MetadataEntity countEntity = new MetadataEntity(MetadataEntityType.TEXT, countInfo);

        MapF<String, String> dateInfo = Cf.hashMap();
        dateInfo.put("ms", Long.toString(System.currentTimeMillis()));
        dateInfo.put("text", "06.12.2016");
        MetadataEntity dateEntity = new MetadataEntity(MetadataEntityType.DATE, dateInfo);

        return new MetadataWrapper(
                Cf.map(MetadataEntityNames.ACTOR, actorEntity)
                    .plus1(MetadataEntityNames.COMMENT, commentEntity)
                    .plus1(MetadataEntityNames.ENTITY, entityEntity)
                    .plus1(MetadataEntityNames.ACTION, actionEntity)
                    .plus1(MetadataEntityNames.DATE, dateEntity)
                    .plus1(MetadataEntityNames.COUNT, countEntity));
    }

    public static MetadataWrapper createUpgradeRequiredUnlimMetadata(String name, boolean addLanguages) {
        MetadataWrapper meta = createMobileLinksMetadata(name, createUpgradeRequiredUnlimLinkMap(addLanguages));
        addPreview(meta);
        return meta;
    }

    private static MetadataWrapper createMobileLinksMetadata(String name, MapF<String, String> linksMap) {
        MetadataWrapper metadata = MetadataWrapper.createEmpty();
        MetadataEntity linksEntity = new MetadataEntity(MetadataEntityType.LINK, linksMap);
        metadata.meta.put(name, linksEntity);
        return metadata;
    }

    private static MapF<String, String> createUpgradeRequiredUnlimLinkMap(boolean addLanguages) {
        MapF<String, String> map = Cf.hashMap();
        map.put("link", "https://disk.yandex.ru/unlim_upgrade_2017");
        if (addLanguages) {
            for (NotifierLanguage lang : NotifierLanguage.R.valuesList()) {
                map.put(lang.value() + "_link", "https://disk.yandex.ru/unlim_upgrade_2017/" + lang.value());
            }
        }
        return map;
    }

    private static void addPreview(MetadataWrapper meta) {
        MetadataEntity entity = new MetadataEntity(MetadataEntityType.RESOURCE,
                Cf.map(MetadataEntityNames.PREVIEW_URL, "http://preview.com/preview.jpg"));
        meta.meta.put(MetadataEntityNames.ENTITY, entity);
    }
}
