package ru.yandex.chemodan.app.notifier.push;

import java.util.Arrays;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndGroup;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.settings.NotificationsSettingsManager;
import ru.yandex.chemodan.ratelimiter.RateLimit;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class PushSettingsManager {
    private static final Logger logger = LoggerFactory.getLogger(PushSettingsManager.class);

    private static final ListF<GlobalSubscriptionChannel> ALL_IOS_CHANNELS = Cf.list(
            GlobalSubscriptionChannel.IOS,
            GlobalSubscriptionChannel.IOS_BRIGHT,
            GlobalSubscriptionChannel.IOS_NON_SILENT
    );

    private static final ListF<GlobalSubscriptionChannel> ALL_ANDROID_CHANNELS = Cf.list(
            GlobalSubscriptionChannel.ANDROID
    );

    private static final MapF<String, ListF<GlobalSubscriptionChannel>> CHANNELS_FOR_PLATFORMS = Cf.map(
            GlobalSubscriptionChannel.DESKTOP.value(), Cf.list(GlobalSubscriptionChannel.DESKTOP),
            GlobalSubscriptionChannel.ANDROID.value(), ALL_ANDROID_CHANNELS,
            GlobalSubscriptionChannel.IOS.value(), ALL_IOS_CHANNELS
    );

    private final NotificationsSettingsManager settingsManager;
    private final NotificationsGlobalSettingsManager globalSettingsManager;
    private final RateLimiterClient rateLimiterClient;

    protected final DynamicProperty<Boolean> disablePushMessages;

    // Expected prohibition format: tag#notificationType. Wildcards allowed on both ends (but not regexps).
    // These notifications will not be sent as pushes.
    protected final DynamicProperty<ListF<String>> disallowedNotificationTypes;

    private static final SetF<String> supportedMobileV1Pushes =
            Cf.set("autoupload", "resource_like", "comment_like", "resource_dislike", "comment_dislike",
                    "reply_comment_add", "topic_comment_add", "unlim_autoupload",
                    "promo_ios", "upgrade_required_2017_unlim_ios");

    private static final SetF<ServiceAndType> RATE_LIMITED = Cf.set(
            DiskNotifications.SHARED_FOLDER_FILE_CREATE,
            DiskNotifications.SHARED_FOLDER_FILE_UPDATE);

    public PushSettingsManager(
            NotificationsSettingsManager settingsManager,
            NotificationsGlobalSettingsManager globalSettingsManager,
            RateLimiterClient rateLimiterClient)
    {
        this.settingsManager = settingsManager;
        this.globalSettingsManager = globalSettingsManager;
        this.rateLimiterClient = rateLimiterClient;

        disablePushMessages = DynamicProperty.cons("notifier-disable-push-messages", false);
        disallowedNotificationTypes = DynamicProperty.cons("notifier-disallowed-notification-types",
                Cf.list("ios:photo_reminder",

                        "ios|android|desktop:shared_folder_file_create|shared_folder_file_update",

                        "ios|android:private_topic_comment_added_to_file", "ios|android:private_reply_added_to_file",
                        "ios|android:private_file_like","ios|android:private_comment_like_file",
                        "ios|android:private_file_dislike","ios|android:private_comment_dislike_file",

                        "ios|android:private_topic_comment_added_to_folder", "ios|android:private_reply_added_to_folder",
                        "ios|android:private_folder_like","ios|android:private_comment_like_folder",
                        "ios|android:private_folder_dislike","ios|android:private_comment_dislike_folder",

                        "android|desktop|web:upgrade_required_2017_unlim_ios",
                        "ios|desktop|web:upgrade_required_2017_unlim_android",

                        "ios|desktop|web:promo_android",
                        "mobile|desktop:unlim_autoupload",

                        "*:shared_folder_invite_received|shared_folder_invite_accepted")
                );
    }

    public SetF<GlobalSubscriptionChannel> getEnabledChannels(DataApiUserId uid, NotificationPushInfo info) {
        if (disablePushMessages.get()) {
            logger.debug("Push manager is disabled");
            return Cf.set();
        }

        if (isResourceUnsubscribed(uid, info.subscriptionKey, info.template.getGroupName())) {
            logger.debug("Didn't send push-notification for uid {}, subcription key {} and type {}, " +
                    "because of subscription settings", uid, info.subscriptionKey, info.template);
            return Cf.set();
        }

        SetF<GlobalSubscriptionChannel> tags = GlobalSubscriptionChannel.PUBLIC
                .filter(ch -> isEventAllowedByDynamicProperties(ch, info))
                .filter(ch -> globalSettingsManager.isUserSubscribed(uid, ch, info.template.getGroupName()));

        if (tags.isEmpty()) {
            logger.debug("Didn't send push-notification for uid {} and type {}, " +
                    "because of global and dynamically disallowed settings", uid, info.template);
            return tags;
        }

        if (checkRateLimit(uid, info)) {
            return Cf.set();
        }

        if (info.platformFilter.isNotEmpty()) {
            return Cf.toSet(info.platformFilter.map(platform -> CHANNELS_FOR_PLATFORMS.getO(platform)
                    .map(channels -> tags.filter(channels::containsTs))
                    .getOrElse(Cf::set)).flatten());
        }

        return tags;
    }

    private boolean checkRateLimit(DataApiUserId uid, NotificationPushInfo info) {
        if (RATE_LIMITED.containsTs(info.template.getTypeName())) {
            try {
                RateLimit limit = rateLimiterClient.queryLimit(rateLimiterKey(uid, info));
                if (!limit.proceed) {
                    logger.debug("Push-notification for uid {} and type {}, blocked by rate-limiter", uid, info.template);
                    return true;
                }
            } catch (Exception e) {
                logger.warn("Error during rate limiter call: " + uid, e);
            }
        }
        return false;
    }

    private boolean isResourceUnsubscribed(DataApiUserId uid, Option<String> subscriptionKey, ServiceAndGroup group) {
        return subscriptionKey.isPresent() && !settingsManager.isSubscribed(uid, subscriptionKey.get(), group);
    }

    private String rateLimiterKey(DataApiUserId userId, NotificationPushInfo info) {
        return "disk_" + info.template.getGroup() + "_" + userId.asString();
    }

    private boolean isEventAllowedByDynamicProperties(GlobalSubscriptionChannel channel, NotificationPushInfo info) {
        if (GlobalSubscriptionChannel.MOBILE == channel && !supportedMobileV1Pushes.containsTs(info.template.getType())) {
            // MOBILE channel is deprecated, use only for old pushes
            return false;
        }

        for (String disallowedType : disallowedNotificationTypes.get()) {
            String[] tagAndRecordType = disallowedType.split(":");
            if (tagAndRecordType.length != 2) {
                logger.warn("Invalid disallowed type entered: " + disallowedType);
                continue;
            }
            SetF<String> channels = Cf.toSet(Arrays.asList(tagAndRecordType[0].split("\\|")));
            SetF<String> recordTypes = Cf.toSet(Arrays.asList(tagAndRecordType[1].split("\\|")));
            if (("*".equals(tagAndRecordType[0]) || channels.containsTs(channel.value()))
                    && ("*".equals(tagAndRecordType[1]) || recordTypes.containsTs(info.template.getType())))
            {
                return false;
            }
        }
        return true;
    }

}
