package ru.yandex.chemodan.app.notifier.push;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.admin.dao.test.ActivateNotificationEmbeddedPg;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecordTypeManager;
import ru.yandex.chemodan.app.notifier.notification.NotificationTypeManagerTestContextConfiguration;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndGroup;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskGroups;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.settings.NotificationsGlobalSettingsManager;
import ru.yandex.chemodan.app.notifier.settings.NotificationsSettingsManager;
import ru.yandex.chemodan.ratelimiter.RateLimiterClient;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.when;

/**
 * @author akirakozov
 */
@ActivateNotificationEmbeddedPg
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = NotificationTypeManagerTestContextConfiguration.class)
@TestExecutionListeners(value = DependencyInjectionTestExecutionListener.class)
public class PushSettingsManagerTest {
    @Autowired
    private NotificationRecordTypeManager typeManager;

    private PushSettingsManager pushSettingsManager;
    private DataApiUserId uid = new DataApiPassportUserId(123);

    @Mock
    private NotificationsSettingsManager settingsManager;
    @Mock
    private NotificationsGlobalSettingsManager globalSettingsManager;
    @Mock
    private RateLimiterClient rateLimiterClient;

    @Before
    public void setup() {
        MockitoAnnotations.initMocks(this);

        pushSettingsManager = new PushSettingsManager(settingsManager, globalSettingsManager, rateLimiterClient);
    }

    @Test
    public void getEnabledChannelsForPublicResource() {
        String publicKey = "public-some_resource_id_herer";

        ServiceAndGroup group = DiskGroups.LIKES;
        when(settingsManager.isSubscribed(uid, publicKey, group)).thenReturn(true);
        when(globalSettingsManager.isUserSubscribed(eq(uid), any(), eq(group))).thenReturn(true);

        SetF<GlobalSubscriptionChannel> channels = pushSettingsManager.getEnabledChannels(
                uid, createNotificationPushInfo(publicKey));
        Assert.equals(GlobalSubscriptionChannel.PUBLIC, channels);
    }

    @Test
    public void getEnabledChannelsForEventWithoutSubscriptionKey() {
        ServiceAndGroup group = DiskGroups.LIKES;
        when(globalSettingsManager.isUserSubscribed(eq(uid), any(), eq(group))).thenReturn(true);

        SetF<GlobalSubscriptionChannel> channels = pushSettingsManager.getEnabledChannels(
                uid, createNotificationPushInfo(Option.empty(), DiskNotifications.COMMENT_LIKE));
        Assert.equals(GlobalSubscriptionChannel.PUBLIC, channels);
    }

    @Test
    public void getEnabledChannelsForPhotoReminder() {
        ServiceAndGroup group = DiskGroups.PHOTO_REMINDER;
        when(globalSettingsManager.isUserSubscribed(eq(uid), any(), eq(group))).thenReturn(true);

        SetF<GlobalSubscriptionChannel> channels = pushSettingsManager.getEnabledChannels(
                uid, createNotificationPushInfo(Option.empty(), DiskNotifications.PHOTO_REMINDER));
        Assert.equals(
                Cf.set(GlobalSubscriptionChannel.WEB, GlobalSubscriptionChannel.DESKTOP,
                        GlobalSubscriptionChannel.ANDROID),
                channels);
    }

    @Test
    public void getEnabledChannelsForSharedFolderFileCreate() {
        ServiceAndGroup group = DiskGroups.COMMON_FOLDERS;

        when(globalSettingsManager.isUserSubscribed(eq(uid), any(), eq(group))).thenReturn(true);

        SetF<GlobalSubscriptionChannel> channels = pushSettingsManager.getEnabledChannels(
                uid, createNotificationPushInfo(Option.empty(), DiskNotifications.SHARED_FOLDER_FILE_CREATE));
        Assert.equals(Cf.set(GlobalSubscriptionChannel.WEB), channels);

        channels = pushSettingsManager.getEnabledChannels(
                uid, createNotificationPushInfo(Option.empty(), DiskNotifications.SHARED_FOLDER_FILE_UPDATE));
        Assert.equals(Cf.set(GlobalSubscriptionChannel.WEB), channels);

    }

    @Test
    public void getEnabledChannelsForPhotoSelection_onlyAndroid() {
        when(globalSettingsManager.isUserSubscribed(eq(uid), any(), any())).thenReturn(true);

        NotificationPushInfo info = new NotificationPushInfo(
                typeManager.resolveRecordType(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK).consMainTemplate(),
                NotificationActor.consFromUid(123L),
                Instant.now(),
                Option.empty()).withPlatformFilter(Cf.list("android"));

        SetF<GlobalSubscriptionChannel> channels = pushSettingsManager.getEnabledChannels(uid, info);
        Assert.equals(Cf.set(GlobalSubscriptionChannel.ANDROID), channels);
    }

    @Test
    public void getEnabledChannelsForPhotoSelection_onlyiOS() {
        when(globalSettingsManager.isUserSubscribed(eq(uid), any(), any())).thenReturn(true);

        NotificationPushInfo info = new NotificationPushInfo(
                typeManager.resolveRecordType(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK).consMainTemplate(),
                NotificationActor.consFromUid(123L),
                Instant.now(),
                Option.empty()).withPlatformFilter(Cf.list("ios"));

        SetF<GlobalSubscriptionChannel> channels = pushSettingsManager.getEnabledChannels(uid, info);
        Assert.equals(Cf.set(GlobalSubscriptionChannel.IOS), channels);
    }

    @Test
    public void getEnabledChannelsForPhotoSelection_bothMobile() {
        when(globalSettingsManager.isUserSubscribed(eq(uid), any(), any())).thenReturn(true);

        NotificationPushInfo info = new NotificationPushInfo(
                typeManager.resolveRecordType(DiskNotifications.PHOTO_SELECTION_COOL_LENTA_WEEK).consMainTemplate(),
                NotificationActor.consFromUid(123L),
                Instant.now(),
                Option.empty()).withPlatformFilter(Cf.list("ios", "android"));

        SetF<GlobalSubscriptionChannel> channels = pushSettingsManager.getEnabledChannels(uid, info);
        Assert.equals(Cf.set(GlobalSubscriptionChannel.ANDROID, GlobalSubscriptionChannel.IOS), channels);
    }

    private NotificationPushInfo createNotificationPushInfo(String subscriptionKey) {
        return createNotificationPushInfo(Option.of(subscriptionKey), DiskNotifications.COMMENT_LIKE);
    }

    private NotificationPushInfo createNotificationPushInfo(
            Option<String> subscriptionKey, ServiceAndType type)
    {
        return new NotificationPushInfo(
                typeManager.resolveRecordType(type).consMainTemplate(),
                NotificationActor.consFromUid(123L),
                Instant.now(),
                subscriptionKey);
    }
}
