package ru.yandex.chemodan.app.notifier.push;

import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.LocalizedMessage;
import ru.yandex.misc.test.Assert;

/**
 * @author akirakozov
 */
public class TextMessageGeneratorTest {

    @Test
    public void generateSimpleTextMessage() {
        String test = "%{user} like to %{first} football %{second}";

        TextMessageGenerator generator = new TextMessageGenerator();
        MetadataWrapper meta = new MetadataWrapper(createMeta());
        String result = generator.generateSimpleTextMessage(new LocalizedMessage(test), meta, Option.empty());
        Assert.equals("Vasya like to play football everyday", result);

        result = generator.generateSimpleTextMessage(new LocalizedMessage(test), meta, Option.of(20));
        Assert.equals("Vasya like to pla...", result);

        MetadataWrapper localizedMeta = createLocalizedMeta();
        result = generator.generateSimpleTextMessage(new LocalizedMessage("", test, localizedMeta), meta, Option.empty());
        Assert.equals("Вася like to play football everyday", result);
    }

    @Test
    public void generateSimpleTextMessageWithSpecialSymbols() {
        String test = "%{user} invites you to check out %{folder}";

        TextMessageGenerator generator = new TextMessageGenerator();
        MetadataWrapper meta = new MetadataWrapper(createMetaWithSpecialSymbols());
        String result = generator.generateSimpleTextMessage(new LocalizedMessage(test), meta, Option.empty());
        Assert.equals("Vasya invites you to check out 100$ bill scans", result);
    }

    @Test
    public void capitalizeStart() {
        TextMessageGenerator generator = new TextMessageGenerator();
        MetadataWrapper meta = new MetadataWrapper(createMeta());

        String test = "%{season} 2019 года";
        String result = generator.generateSimpleTextMessage(new LocalizedMessage(test), meta, Option.empty());
        Assert.equals("Осень 2019 года", result);

        test = "2019 года %{season}";
        result = generator.generateSimpleTextMessage(new LocalizedMessage(test), meta, Option.empty());
        Assert.equals("2019 года осень", result);

        test = "2019 года. %{season}";
        result = generator.generateSimpleTextMessage(new LocalizedMessage(test), meta, Option.empty());
        Assert.equals("2019 года. Осень", result);

        test = "2019 года.%{season}";
        result = generator.generateSimpleTextMessage(new LocalizedMessage(test), meta, Option.empty());
        Assert.equals("2019 года.Осень", result);
    }

    private MapF<String, MetadataEntity> createMeta() {
        return Cf.map("user", new MetadataEntity(MetadataEntityType.USER, Cf.map("text", "Vasya")))
                .plus1("first", new MetadataEntity(MetadataEntityType.TEXT, Cf.map("text", "play")))
                .plus1("second", new MetadataEntity(MetadataEntityType.DATE, Cf.map("text", "everyday")))
                .plus1("season", new MetadataEntity(MetadataEntityType.DATE, Cf.map("text", "осень")))
                ;

    }

    private MapF<String, MetadataEntity> createMetaWithSpecialSymbols() {
        return Cf.map("user", new MetadataEntity(MetadataEntityType.USER, Cf.map("text", "Vasya")))
                .plus1("folder", new MetadataEntity(MetadataEntityType.TEXT, Cf.map("text", "100$ bill scans")));

    }

    private MetadataWrapper createLocalizedMeta() {
        return new MetadataWrapper(Cf.map("user", new MetadataEntity(Cf.map("text", "Вася"))));
    }
}
