package ru.yandex.chemodan.app.notifier.push.filter;

import lombok.Value;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DevicePushFilterConfigTest {
    private static final DatabaseHandle DB_HANDLE =
            new DatabaseHandle(NotificationPushFilterManager.DEVICE_PUSH_FILTER_DB_REF, "handle123");

    private static final String COLLECTION_ID_PREFIX = NotificationPushFilterManager.COLLECTION_ID_PREFIX;

    private static final DataApiPassportUserId UID = new DataApiPassportUserId(1L);

    @Test
    public void knownGroupActionUseSpecifiedValue() {
        Assert.isTrue(
                consPushConfig()
                        .isEnabled("group2#1", "action2#1")
        );
        Assert.isFalse(
                consPushConfig()
                        .isEnabled("group2#2", "action2#2")
        );
    }

    @Test
    public void knownByWildcardGroupActionUseSpecifiedValue() {
        Assert.isTrue(
                consPushConfig()
                        .isEnabled("any-group", "action1#3")
        );
        Assert.isFalse(
                consPushConfig()
                        .isEnabled("group1#4", "any-action")
        );
    }

    @Test
    public void unknownGroupActionFellBackToOther() {
        Assert.isFalse(
                consPushConfig(false)
                        .isEnabled("unknown-group", "unknown-action")
        );
        Assert.isTrue(
                consPushConfig(true)
                        .isEnabled("unknown-group", "unknown-action")
        );
    }

    private static DevicePushFilterConfig consPushConfig() {
        return consPushConfig(Option.empty());
    }

    private static DevicePushFilterConfig consPushConfig(boolean otherEnabled) {
        return consPushConfig(Option.of(otherEnabled));
    }

    private static DevicePushFilterConfig consPushConfig(Option<Boolean> otherEnabled) {
        ListF<DataRecord> records = Cf.list(
                consDataApiRecord("rec#1", true,
                        new GroupAction("group1#1", "action1#1"),
                        new GroupAction("group2#1", "action2#1")
                ),
                consDataApiRecord("rec#2", false,
                        new GroupAction("group1#2", "action1#2"),
                        new GroupAction("group2#2", "action2#2"),
                        new GroupAction("*", "action3#2"),
                        new GroupAction("group4#2", "*#2")
                ),
                consDataApiRecord("rec#3", true,
                        new GroupAction("*", "action1#3")
                ),
                consDataApiRecord("rec#4", false,
                        new GroupAction("group1#4", "*"))
        );
        return DevicePushFilterConfig.fromDataApiRecords(
                records.plus(otherEnabled.map(en -> consDataApiRecord("other", en)))
        );
    }

    private static DataRecord consDataApiRecord(String recordId, boolean enabled, GroupAction... items) {
        return new DataRecord(
                UID,
                new DataRecordId(DB_HANDLE, COLLECTION_ID_PREFIX + "_" + "uuid123", recordId),
                12L,
                Cf.map(
                        DevicePushFilterConfig.ENABLED_FIELD, DataField.bool(enabled),
                        DevicePushFilterConfig.FILTER_FIELD,
                        DataField.string(GroupAction.serializeListToJson(Cf.list(items)))
                )
        );
    }

    @Value
    @BenderBindAllFields
    private static class GroupAction {
        static final BenderSerializer<GroupAction> serializer =
                BenderSerializer.cons(ClassX.wrap(GroupAction.class), BenderConfiguration.defaultConfiguration());

        String group;

        String action;

        static String serializeListToJson(ListF<GroupAction> items) {
            return new String(serializer.serializeListJson(items));
        }
    }
}
