package ru.yandex.chemodan.app.notifier.push.filter;

import com.fasterxml.jackson.databind.JsonNode;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.notifier.push.NotificationPushInfo;
import ru.yandex.chemodan.app.notifier.push.body.NotificationMobilePushBody;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannelGroup;
import ru.yandex.chemodan.util.AppVersion;
import ru.yandex.chemodan.util.AppVersionRange;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.xiva.DiskXivaServices;
import ru.yandex.chemodan.xiva.XivaEvent;
import ru.yandex.chemodan.xiva.XivaPushService;
import ru.yandex.chemodan.xiva.XivaSingleTokenClient;
import ru.yandex.chemodan.xiva.XivaSubscription;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Pattern2;

import static ru.yandex.chemodan.app.notifier.push.filter.DevicePushFilterConfig.ALWAYS_ENABLED_CONFIG;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class NotificationPushFilterManager {
    private static final Logger logger = LoggerFactory.getLogger(NotificationPushFilterManager.class);
    static final String COLLECTION_ID_PREFIX = "push_config_";

    private static final Pattern2 COLLECTION_ID_PATTERN = Pattern2.compile("^" + COLLECTION_ID_PREFIX + "(.+)$");

    static final AppDatabaseRef DEVICE_PUSH_FILTER_DB_REF =
            new AppDatabaseRef("yadisk", "mobile_common");

    private final XivaSingleTokenClient client;

    private final DataApiManager dataApiManager;

    public NotificationPushFilterManager(XivaSingleTokenClient client, DataApiManager dataApiManager) {
        this.client = client;
        this.dataApiManager = dataApiManager;
    }

    public NotificationPushFilter getFilter(DataApiUserId userId, GlobalSubscriptionChannelGroup channelGroup,
            NotificationPushInfo info, XivaEvent event)
    {
        ListF<XivaSubscription> subs = client.list(userId.toPassportUid(), DiskXivaServices.DISK_NOTIFIER);
        MapF<String, DevicePushFilterConfig> uuidToPushFilter = getUuidsPushFilters(userId, subs);
        return getFilter(subs, uuidToPushFilter, channelGroup,
                info.template.getGroup(),
                info.template.getType(),
                event
        );
    }

    private MapF<String, DevicePushFilterConfig> getUuidsPushFilters(DataApiUserId uid, ListF<XivaSubscription> subs) {
        ListF<String> uuids = subs.filterMap(XivaSubscription::getUuid);
        ListF<DataRecord> records = dataApiManager.getRecords(
                new UserDatabaseSpec(uid, DEVICE_PUSH_FILTER_DB_REF),
                RecordsFilter.DEFAULT.withCollectionIds(uuids.map(uuid -> COLLECTION_ID_PREFIX + uuid))
        );
        return records.groupBy(dataRecord -> extractUuidOrEmpty(dataRecord.getCollectionId()))
                .mapValues(DevicePushFilterConfig::fromDataApiRecords);
    }

    private static String extractUuidOrEmpty(String collectionId) {
        return COLLECTION_ID_PATTERN.findFirstGroup(collectionId)
                .getOrElse("");
    }

    static NotificationPushFilter getFilter(
            ListF<XivaSubscription> subs,
            MapF<String, DevicePushFilterConfig> uuidToPushFilter,
            GlobalSubscriptionChannelGroup channelGroup,
            String group, String action,
            XivaEvent event)
    {
        if (!channelGroup.equals(GlobalSubscriptionChannelGroup.MOBILE_V2_BRIGHT)) {
            return NotificationPushFilter.ALL_ENABLED;
        }

        logger.info("Get filter params: subs={}, uuidToPushFilter={}, channelGroup={}, group={}, action={}, event={}",
                subs, uuidToPushFilter, channelGroup, group, action, event);

        Option<AppVersionRange> iosCompVersionRange = event.getBody()
                .filter(body -> body instanceof NotificationMobilePushBody)
                .<NotificationMobilePushBody>cast()
                .map(NotificationMobilePushBody::getServiceSpecificParams)
                .filterMap(map -> map.getO(XivaPushService.APNS));

        return NotificationPushFilter.includeSubscriptionIds(
                subs.filter(NotificationPushFilterManager::hasBrightTag)
                        .filter(xivaSubscription -> hasValidBrightPushImpl(xivaSubscription.getFilter(), iosCompVersionRange))
                        .filter(consSendToDeviceF(uuidToPushFilter, group, action))
                        .map(XivaSubscription::getId)
        );
    }

    static boolean hasValidBrightPushImpl(Option<String> xivaFilter, Option<AppVersionRange> compVersionRange) {
        if (!compVersionRange.isPresent()) {
            return true;
        }

        return xivaFilter.map(filter -> {
            JsonNode node = JsonNodeUtils.getNode(filter);
            node = node.get("vars");
            if (node == null) {
                // no "vars" in filter - may be not ios
                return false;
            }

            node = node.get("Version");
            if (node == null) {
                // has no Version in subscription but compVersionRange is present
                return false;
            }

            node = node.get("$has_tags");
            // Эти проверки тут на случай изменения формата. Чтобы изменения не прошли незаметно, а все бы сломалось.
            Check.notNull(node, "no field '$has_tags' in Version");
            Check.isTrue(node.isArray(), "field '$has_tags' is not array");

            String appVersion = node.get(0).textValue();

            return compVersionRange.get().isInRangeBothInclusive(new AppVersion(appVersion));
        }).getOrElse(false);
    }

    @NotNull
    private static Function1B<XivaSubscription> consSendToDeviceF(MapF<String, DevicePushFilterConfig> uuidToPushFilter,
            String group, String action)
    {
        return s -> s.getUuid()
                    .filterMap(uuidToPushFilter::getO)
                    .getOrElse(ALWAYS_ENABLED_CONFIG)
                    .isEnabled(group, action);
    }

    private static boolean hasBrightTag(XivaSubscription subscription) {
        return subscription.getFilter()
                .isMatch(f -> f.contains("ios_bright"));
    }
}
